# BAMROC,  Version 1.0,  March 27, 2001 
# Binormal Association Marginal Models for ordinal 
# Receiver Operating Characteristic Data  
#
# Author:  	Thor Aspelund
#		Department of Statistics and Actuarial Science
#		241 Schaeffer Hall
#		The University of Iowa, Iowa City, Iowa 52242
#
# Reference:	Lang, J. B. and Aspelund, T. (2001).  Binormal 
#		Marginal-Association Models for Empirically 
#		Evaluating and Comparing Diagnostics.  
#		Statistical Modelling. Forthcoming.
#
# This S-PLUS program fits, via maximum likelihood, 
# Binormal Association Marginal (BAM) models, as described in 
# Lang and Aspelund (2001).    
#
# Use the source command in S-PLUS to read in the code in this file:
# > source("name.of.this.file")
# 
# To avoid conflicts, run the source command in a new directory 
# or workspace of S-PLUS.  The following are the names of the 
# functions created.
#
#       "Amatrix","bamroc","bamroc1","bamroc.area","bn.area",
#	"bn.curves","DA","DirectSum","dL","dML","dnum","domega.dtheta",
#	"dpi.domega","eta.ls","gof","gof.bam","info.alpha","info.omega",
#	"info.theta","initial.loglinear","L","L2norm","label.bam",
#	"local.odds","LowerOne","MarginalsB","ML","omega.theta",
#	"pi.omega","roc.curve","roc.points","score.alpha","score.omega",
#	"score.theta","shiftX","solve.eta","solve.theta.efficient",
#	"standardM","standardW","summary.bam","wald","wald.bam",
#	"wls.approx" 
#	
# NOTES:
# The basic call to the fitting procedure is
#
#		> bamroc(y,c(nr,nc),Amodel=i,EqualA=F)
#
# where y are vectorized contingency tables, one column per 
# population and the first index moves fastest (see the 
# examples file BAMROC.examples); nr is the number of rows; 
# nc is the number of columns; and Amodel=i is the choice of 
# association model, i can be a number from 1 to 8:
#		1) Independence
#		2) Quasi independence - one diagonal parameter
#		3) Quasi independence - saturated diagonal
#		4) Linear by linear 
#		5) Linear by linear plus agreement - one diagonal parameter	
#		6) Linear by linear plus agreement - saturated diagonal
#		7) Quasi symmetry
#		8) Saturated model.
#
# EqualA is a flag (T/F) to indicate if association parameters
# should be equal across populations.
#
# SOME DETAILS:
# A small constant can be added to the data for a few iterations
# to enhance convergence.  The call is, for example, 
#
#		> bamroc(y,c(nr,nc),Amodel=i,EqualA=F,small.constant=0.05)
#
# which has the effect that 0.05 is added to empty cells for 4 iterations 
# (the default).   More details are in the documentation.
#
# 

"Amatrix"<-
function(d.list, Amodel)
{
# Log linear design matrices for two factors.
# d is a list vector with two elements containing the numbered levels of the margins.
# 
# 1) Independence
# 2) Quasi independence - one diagonal parameter
# 3) Quasi independence - saturated diagonal
# 4) Linear by linear 
# 5) Linear by linear plus agreement - one diagonal parameter 
# 6) Linear by linear plus agreement - saturated diagonal
# 7) Quasi symmetry
# 8) Saturated model
#
	ld1 <- length(d.list[[1]])
	ld2 <- length(d.list[[2]])
	square <- (ld1 == ld2)
	nr <- ld1 * ld2	# Independence
	if(Amodel == 1) {
		W2 <- NULL
	}
	else if(square & Amodel == 2) {
		cell.index <- expand.grid(1:ld1, 1:ld2)
		W2 <- matrix(0, nr, 1)
		W2[, 1] <- 1 * (cell.index[, 1] == cell.index[, 2])
	}
	else if(square & Amodel == 3) {
		cell.index <- expand.grid(1:ld1, 1:ld2)
		W2 <- matrix(0, nr, ld1)
		for(i in 1:ld1)
			W2[, i] <- 1 * ((cell.index[, 1] == cell.index[, 2]) & (
				cell.index[, 1] == rep(i, nr)))
	}
	else if(Amodel == 4) {
		W2 <- matrix(0, nr, 1)
		W2[, 1] <- outer(d.list[[1]], d.list[[2]])
	}
	else if(square & Amodel == 5) {
		cell.index <- expand.grid(1:ld1, 1:ld2)
		W2 <- matrix(0, nr, 2)
		W2[, 1] <- outer(d.list[[1]], d.list[[2]])
		W2[, 2] <- 1 * (cell.index[, 1] == cell.index[, 2])
	}
	else if(square & Amodel == 6) {
		cell.index <- expand.grid(1:ld1, 1:ld2)
		W2 <- matrix(0, nr, ld1 + 1)
		W2[, 1] <- outer(d.list[[1]], d.list[[2]])
		for(i in 1:ld1)
			W2[, i + 1] <- 1 * (cell.index[, 1] == cell.index[, 2] & 
				cell.index[, 1] == i)
	}
	else if(square & Amodel == 7) {
		C2 <- local.odds(ld1, ld2)
		odds.index <- expand.grid(ld1 - 1, ld2 - 1)
		nodds <- ((ld1 - 1) * ld1)/2
		symmetry.matrix <- matrix(0, ld1 - 1, ld2 - 1)
		k <- 1
		symmetry.matrix[ld1 - 1, ld2 - 1] <- nodds
		for(j in 1:(ld2 - 2))
			for(i in j:(ld1 - 1)) {
				symmetry.matrix[i, j] <- (1:nodds)[k]
				symmetry.matrix[j, i] <- (1:nodds)[k]
				k <- k + 1
			}
		X <- matrix(0, (ld1 - 1) * (ld2 - 1), nodds)
		for(i in 1:((ld1 - 1) * (ld2 - 1)))
			X[i, symmetry.matrix[i]] <- 1
		W2 <- t(C2) %*% solve(C2 %*% t(C2)) %*% X
	}
	else if(Amodel == 8) {
		C2 <- local.odds(ld1, ld2)
		W2 <- t(C2) %*% solve(C2 %*% t(C2))
	}
	else {
		print("Allowed amodels are 1 through 8")
		return(1)
	}
	W2
}
"bamroc"<-
function(y, score.list, Amodel = 1, EqualA = F, initial.beta = 0, 
	small.constant = 0, maxiter = 25, sub.maxiter = 4)
{
# Input verification
	if(length(score.list) == 1) {
		return(DA(y, score.list, maxiter))
	}
	if(is.list(score.list))
		d.list <- score.list
	else {
		d.list <- list(1:score.list[1], 1:score.list[2])
	}
	d <- c(length(d.list[[1]]), length(d.list[[2]]))
	if(!(is.element(Amodel, 1:8))) {
		print("Allowed associaton model choices are 1 through 8.")
		return(1)
	}
	W1 <- standardW(d)
	W2 <- Amatrix(d.list, Amodel)
	W <- cbind(W1, W2)
	assign("EqualA", EqualA, frame = 0)
	assign("r1", d[1], frame = 0)
	assign("r2", d[2], frame = 0)
	assign("a", ncol(W2), frame = 0)
	if(Amodel == 1) {
		h1 <- function(theta)
		{
			r1 <- get("r1", where = 0)
			r2 <- get("r2", where = 0)
			c(pnorm(theta[1:(r1 - 1)]), pnorm(theta[(r1 + 2):(r1 + 
				r2)]))
		}
		h2 <- function(theta)
		{
			r1 <- get("r1", where = 0)
			r2 <- get("r2", where = 0)
			c(pnorm(theta[1:(r1 - 1)], theta[r1], exp(theta[r1 + 1]
				)), pnorm(theta[(r1 + 2):(r1 + r2)], theta[r1 + 
				r2 + 1], exp(theta[r1 + r2 + 2])))
		}
	}
	else {
		h1 <- function(theta)
		{
			r1 <- get("r1", where = 0)
			r2 <- get("r2", where = 0)
			m <- r1 + r2 + 2
			a <- get("a", where = 0)
			c(pnorm(theta[1:(r1 - 1)]), pnorm(theta[(r1 + 2):(r1 + 
				r2)]), theta[(m + 1):(m + a)])
		}
		h2 <- function(theta)
		{
			r1 <- get("r1", where = 0)
			r2 <- get("r2", where = 0)
			m <- r1 + r2 + 2
			a <- get("a", where = 0)
			EqualA <- get("EqualA", where = 0)
			c(pnorm(theta[1:(r1 - 1)], theta[r1], exp(theta[r1 + 1]
				)), pnorm(theta[(r1 + 2):(r1 + r2)], theta[r1 + 
				r2 + 1], exp(theta[r1 + r2 + 2])), theta[EqualA *
				((m + 1):(m + a)) + (1 - EqualA) * ((m + a + 1):
				(m + 2 * a))])
		}
	}
	h <- list(h1, h2)
	value <- bamroc1(y, d, W, h, EqualA, initial.beta, small.constant, 
		maxiter, sub.maxiter)
	value$cov.theta <- solve(value$information)
	value$AmodelName <- switch(Amodel,
		"1) Independence",
		"2) Quasi independence - one diagonal parameter",
		"3) Quasi independence - saturated diagonal",
		"4) Linear by linear",
		"5) Linear by linear plus agreement - one diagonal parameter",
		"6) Linear by linear plus agreement - saturated diagonal",
		"7) Quasi symmetry",
		"8) Saturated model")
	value$EqualA <- EqualA
	value$d <- d
	value$score.list <- d.list
	value$area <- bamroc.area(value, bn.area)
	value$pi <- pi.omega(value$omega, value$W)
	value$cov.pi <- matrix(0, 2 * nrow(y), 2 * nrow(y))
	dpido1 <- dpi.domega(value$omega[, 1], value$W) %*% domega.dtheta(value$
		theta, h[[1]], value$omega[, 1], value$M, value$W)
	dpido2 <- dpi.domega(value$omega[, 2], value$W) %*% domega.dtheta(value$
		theta, h[[2]], value$omega[, 2], value$M, value$W)
	value$cov.pi[1:nrow(y), 1:nrow(y)] <- dpido1 %*% value$cov.theta %*% t(
		dpido1)
	value$cov.pi[(nrow(y) + 1):(2 * nrow(y)), (nrow(y) + 1):(2 * nrow(y))] <- 
		dpido2 %*% value$cov.theta %*% t(dpido2)
	value$cov.pi[1:nrow(y), (nrow(y) + 1):(2 * nrow(y))] <- dpido1 %*% 
		value$cov.theta %*% t(dpido2)
	value$cov.pi[(nrow(y) + 1):(2 * nrow(y)), 1:nrow(y)] <- t(value$cov.pi[
		1:nrow(y), (nrow(y) + 1):(2 * nrow(y))])
	remove("r1", where = 0)
	remove("r2", where = 0)
	remove("a", where = 0)
	remove("EqualA", where = 0)
	summary.bam(value)
	value
}
"bamroc1"<-
function(y, d, W, h, SharedA = F, initial.beta = 0, small.constant = 0, maxiter
	 = 25, sub.maxiter = 4)
{
	M <- standardM(d)
	omega.star <- initial.loglinear(y + small.constant * (y == 0), W)
	omega <- omega.star[-1,  ]
	alpha <- NULL
	if(ncol(W) > nrow(M)) {
		if(SharedA)
			alpha <- apply(matrix(omega[ - c(1:nrow(M)),  ], ncol
				 = 2), 1, mean)
		else alpha <- omega[ - c(1:nrow(M)),  ]
	}
	if(all(initial.beta))
		beta <- initial.beta
	else beta <- wls.approx(pi.omega(omega, W), d)
	theta <- c(beta, alpha)
	if(small.constant) {
		value <- solve.theta.efficient(theta, omega, h, y + 
			small.constant * (y == 0), M, W, maxiter = sub.maxiter)[
			1:2]
		theta <- value[[1]]
		omega <- value[[2]]
		cat("\n", "Iterations with small constant done.", "\n")
	}
	fit <- solve.theta.efficient(theta, omega, h, y, M, W, maxiter)
	fit
}
"bamroc.area"<-
function(bam.fit, area.function = bn.area)
{
	d <- bam.fit$d
	theta <- bam.fit$theta
	inverse.info <- solve(bam.fit$information)
	muI <- d[1]
	ksiI <- d[1] + 1
	AI <- area.function(theta, c(muI, ksiI))
	dAI <- dnum(area.function, theta, c(muI, ksiI))
	seAI <- sqrt(dAI %*% inverse.info %*% t(dAI))
	value <- list(A = AI, seA = seAI)
	if(length(d) > 1) {
		muII <- d[1] + 1 + d[2]
		ksiII <- d[1] + 1 + d[2] + 1
		AII <- area.function(theta, c(muII, ksiII))
		dAII <- dnum(area.function, theta, c(muII, ksiII))
		seAII <- sqrt(dAII %*% inverse.info %*% t(dAII))
		seAI.AII <- sqrt((dAI - dAII) %*% inverse.info %*% t(dAI - dAII
			))
		value <- list(AI = AI, seAI = seAI, AII = AII, seAII = seAII, 
			"AI-AII" = AI - AII, "se(AI-AII)" = seAI.AII)
	}
	return(value)
}
"bn.area"<-
function(theta, index)
{
	pnorm(theta[index[1]]/sqrt(1 + exp(2 * theta[index[2]])))
}
"bn.curves"<-
function(bamfit, create.plot = T, return.list = F, from = -6, to = 6, step.size
	 = 0.05)
{
	y <- bamfit$y
	d <- bamfit$d
	theta <- bamfit$theta
	n <- apply(y, 2, sum)
	yI <- MarginalsB(d, c(1, 0)) %*% y
	epIx <- t(t(standardM(d[1]) %*% yI[d[1]:1, 1])/n[1])
	epIy <- t(t(standardM(d[1]) %*% yI[d[1]:1, 2])/n[2])
	muI <- theta[d[1]]
	ksiI <- theta[d[1] + 1]
	co <- seq(from, to, step.size)
	xI <- 1 - pnorm(co)
	yI <- 1 - pnorm(co, muI, exp(ksiI))
	curveI <- list(x = xI, y = yI)
	if(length(d) > 1) {
		yII <- MarginalsB(d, c(0, 1)) %*% y
		epIIx <- t(t(standardM(d[2]) %*% yII[d[2]:1, 1])/n[1])
		epIIy <- t(t(standardM(d[2]) %*% yII[d[2]:1, 2])/n[2])
		muII <- theta[d[1] + 1 + d[2]]
		ksiII <- theta[d[1] + 1 + d[2] + 1]
		xII <- xI
		yII <- 1 - pnorm(co, muII, exp(ksiII))
		curveII <- list(x = xII, y = yII)
	}
	if(create.plot) {
		roc.curve(xI, yI)
		roc.points(epIx, epIy, add = T)
		if(length(d) > 1) {
			roc.curve(xII, yII, lty = 4, add = T)
			roc.points(epIIx, epIIy, pch = 1, add = T)
			legend(0.4, 0.2, c("System 1", "System 2"), marks = c(0,
				1), lty = c(3, 4))
		}
		mtext("False positive", side = 1, line = 3, cex = 1.2)
		mtext("True positive", side = 2, line = -3, cex = 1.2)
	}
	if(length(d) > 1)
		value <- list(curveI = curveI, curveII = curveII, epIx = epIx, 
			epIy = epIy, epIIx = epIIx, epIIy = epIIy)
	else value <- list(curve = curveI, epx = epIx, epy = epIy)
	if(return.list)
		return(value)
}
"DA"<-
function(y, d, maxiter = 25)
{
	if(is.list(d)) {
		d.list <- d
		d <- length(d[[1]])
	}
	else {
		d <- d[1]
		d.list <- 1:d
	}
	W <- standardW(d)
	h1 <- function(theta)
	{
		lt <- length(theta)
		pnorm(theta[1:(lt - 2)])
	}
	h2 <- function(theta)
	{
		lt <- length(theta)
		pnorm(theta[1:(lt - 2)], theta[lt - 1], exp(theta[lt]))
	}
	h <- list(h1, h2)
	M <- standardM(d)
	omega.star <- initial.loglinear(y, W)
	omega <- omega.star[-1,  ]
	theta <- wls.approx(pi.omega(omega, W), d)
	value <- solve.theta.efficient(theta, omega, h, y, M, W, maxiter)
	value$AmodelName <- ""
	value$d <- d
	value$score.list <- d.list
	cat("\n")
	value$area <- bamroc.area(value, bn.area)
	summary.bam(value)
	value
}
"DirectSum"<-
function(A, B)
{
	dim.A <- dim(A)
	dim.B <- dim(B)
	zero.A <- matrix(0, dim.A[1], dim.B[2])
	zero.B <- matrix(0, dim.B[1], dim.A[2])
	rbind(cbind(A, zero.A), cbind(zero.B, B))
}
"dL"<-
function(omega, M, W)
{
	p1 <- dML(omega, M, W)
	if(ncol(W) == nrow(M))
		return(p1)
	else {
		gamma0 <- omega[(nrow(M) + 1):ncol(W)]
		p2 <- cbind(matrix(0, length(gamma0), nrow(M)), diag(length(
			gamma0)))
		rbind(p1, p2)
	}
}
"dML"<-
function(omega, M, W)
{
	M %*% dpi.domega(omega, W)
}
"dnum"<-
function(g, theta, ...)
{
	nr <- nrow(matrix(g(theta, ...)))
	nc <- nrow(matrix(theta))
	value <- matrix(0, nr, nc)
	eps <- (.Machine$double.eps)^(1/3)
	h <- eps * theta + eps
	hj <- h * 0
	for(j in 1:nc) {
		hj[j] <- h[j]
		value[, j] <- (g(theta + hj, ...) - g(theta - hj, ...))/(2 * hj[
			j])
		hj[j] <- 0
	}
	value
}
"domega.dtheta"<-
function(theta, h, omega, M, W)
{
	solve(dL(omega, M, W), dnum(h, theta))
}
"dpi.domega"<-
function(omega, W)
{
	piv <- pi.omega(omega, W)
	(diag(piv, length(piv)) - piv %*% t(piv)) %*% W
}
"eta.ls"<-
function(eta0, av)
{
	omega0 <- av[[1]]
	omega0[1:nrow(M)] <- eta0
	fbeta <- av[[2]]
	M <- av[[3]]
	W <- av[[4]]
	L2norm(M %*% pi.omega(omega0, W) - fbeta)
}
"gof"<-
function(y, omega, W, fv = F)
{
	yhat <- t(t(pi.omega(omega, W)) * apply(y, 2, sum))
	X2 <- apply((y - yhat)^2/yhat, 2, sum)
	G2 <- apply(2 * y * log(y/yhat), 2, sum, na.rm = T)
	value <- list("X2 by population" = X2, X2 = sum(X2), "G2 by population"
		 = G2, G2 = sum(G2))
	if(fv == T)
		value$yhat <- yhat
	value
}
"gof.bam"<-
function(bamfit, fv = F)
{
	y <- bamfit$y
	omega <- bamfit$omega
	W <- bamfit$W
	value <- gof(y, omega, W, fv)
	value$df <- length(y) - ncol(y) - length(bamfit$theta)
	value
}
"info.alpha"<-
function(alpha, y, Wstar)
{
	mu <- exp(Wstar %*% alpha)
	X <- Wstar * c(sqrt(mu))
	t(X) %*% X
}
"info.omega"<-
function(omega, y, n, W)
{
	n * t(W) %*% dpi.domega(omega, W)
}
"info.theta"<-
function(theta, h, omega, y, n, M, W)
{
	dodt <- domega.dtheta(theta, h, omega, M, W)
	t(dodt) %*% info.omega(omega, y, n, W) %*% dodt
}
"initial.loglinear"<-
function(y, W)
{
	k <- ncol(y)
	fit.list <- vector("list", k)
	alpha <- matrix(0, ncol(W) + 1, k)
	fit.list[[1]] <- glm(y[, 1] ~ W, family = poisson, control = 
		glm.control(maxit = 15))
	alpha[, 1] <- coef(fit.list[[1]])
	if(k > 1) {
		for(i in 2:k) {
			fit.list[[i]] <- glm(y[, i] ~ W, family = poisson, 
				control = glm.control(maxit = 15))
			alpha[, i] <- coef(fit.list[[i]])
		}
	}
	return(alpha)
}
"L"<-
function(omega, M, W)
{
	if(ncol(W) == nrow(M))
		gamma0 <- NULL
	else if(is.matrix(omega)) {
		gamma0 <- omega[(nrow(M) + 1):ncol(W),  ]
		value <- rbind(ML(omega, M, W), gamma0)
	}
	else {
		gamma0 <- omega[(nrow(M) + 1):ncol(W)]
		value <- matrix(c(ML(omega, M, W), gamma0))
	}
	return(value)
}
"L2norm"<-
function(x)
{
	sqrt(sum(x^2))
}
"label.bam"<-
function(bamfit)
{
	value <- vector("character", length(bamfit$theta))
	d <- bamfit$d
	lt <- length(bamfit$theta)
	if(length(d) == 1) {
		value[1:(d - 1)] <- paste("c", 1:(d - 1), sep = "")
		value[d] <- "mu"
		value[d + 1] <- "ksi"
	}
	else if(length(d) == 2) {
		lb <- d[1] + d[2] + 2
		value[1:(d[1] - 1)] <- paste("c1", 1:(d[1] - 1), sep = "")
		value[d[1]] <- "mu1"
		value[d[1] + 1] <- "ksi1"
		value[(d[1] + 2):(d[1] + d[2])] <- paste("c2", 1:(d[2] - 1), 
			sep = "")
		value[lb - 1] <- "mu2"
		value[lb] <- "ksi2"
		if(lt > lb)
			if(bamfit$EqualA)
				value[(lb + 1):lt] <- paste("alpha", 1:(lt - lb
				  ), sep = "")
			else {
				value[(lb + 1):(lb + (lt - lb)/2)] <- paste(
				  "alpha0", 1:((lt - lb)/2), sep = "")
				value[(lb + (lt - lb)/2 + 1):lt] <- paste(
				  "alpha1", 1:((lt - lb)/2), sep = "")
			}
	}
	else value <- NULL
	return(value)
}
"local.odds"<-
function(p, q)
{
# Local odds for a two - way table.
	return(kronecker(cbind(diag(q - 1), 0) - cbind(0, diag(q - 1)), cbind(
		diag(p - 1), 0) - cbind(0, diag(p - 1))))
}
"LowerOne"<-
function(n)
{
	value <- matrix(0, n, n)
	for(i in 1:n)
		for(j in 1:i)
			value[i, j] <- 1
	return(value)
}
"MarginalsB"<-
function(dim.vector, binary.vector = NULL)
{
	ldv <- length(dim.vector)
	marginal.vector <- (1:ldv)[binary.vector == 1]
	if(is.null(marginal.vector))
		return(matrix(1, 1, cumprod(dim.vector)[ldv]))
	value <- matrix(1, 1, 1)
	for(i in 1:ldv)
		if(is.element(i, marginal.vector)) value <- kronecker(diag(
				dim.vector[i]), value) else value <- kronecker(
				t(rep(1, dim.vector[i])), value)
	return(value)
}
"ML"<-
function(omega, M, W)
{
	M %*% pi.omega(omega, W)
}
"omega.theta"<-
function(theta, omega0, h, M, W)
{
	htheta <- h(theta)
	dim.mr <- nrow(M)
	fbeta <- htheta[1:dim.mr]
	eta0 <- omega0[1:dim.mr]
	if(ncol(W) > nrow(M)) {
		gamma0 <- htheta[(dim.mr + 1):length(htheta)]
		eta <- solve.eta(fbeta, c(eta0, gamma0), M, W)
		value <- c(eta, gamma0)
	}
	else {
		eta <- solve.eta(fbeta, omega0, M, W)
		value <- eta
	}
	value
}
"pi.omega"<-
function(omega, W)
{
	num <- exp(W %*% omega)
	t(t(num)/apply(num, 2, sum))
}
"roc.curve"<-
function(x, y, lty = 3, add = F)
{
	par(pty = "s")
	if(add)
		lines(x, y, lwd = 1.5, lty = lty)
	else {
		plot(x, y, axes = F, type = "n", xlim = c(0, 1), ylim = c(0, 1),
			xlab = "", ylab = "")
		axis(1)
		axis(2)
		box()
		lines(x, y, lwd = 1.5, lty = lty)
	}
	par(pty = "m")
}
"roc.points"<-
function(x, y, pch = 0, add = F)
{
	par(pty = "s")
	if(add)
		points(x, y, pch = pch)
	else {
		plot(x, y, axes = F, type = "n", xlim = c(0, 1), ylim = c(0, 1),
			xlab = "", ylab = "")
		axis(1)
		axis(2)
		box()
		points(x, y, pch = pch)
	}
	par(pty = "m")
}
"score.alpha"<-
function(alpha, y, M, W)
{
	t(W) %*% (y - exp(W %*% alpha))
}
"score.omega"<-
function(omega, y, n, W)
{
	t(W) %*% (y - n * pi.omega(omega, W))
}
"score.theta"<-
function(theta, h, omega, y, n, M, W)
{
	t(domega.dtheta(theta, h, omega, M, W)) %*% score.omega(omega, y, n, W)
}
"shiftX"<-
function(d)
{
	m <- length(d)
	X1 <- cbind(diag(d[1] - 1), 0)
	X2 <- cbind(diag(d[1] - 1), -1)
	if(m == 2) {
		X12 <- cbind(diag(d[2] - 1), 0)
		X22 <- cbind(diag(d[2] - 1), -1)
		X1 <- DirectSum(X1, X12)
		X2 <- DirectSum(X2, X22)
	}
	rbind(X1, X2)
}
"solve.eta"<-
function(fbeta, omega0, M, W)
{
	diff0 <- 1
	dim.mr <- nrow(M)
	eta0 <- omega0[1:dim.mr]
	eta0original <- eta0
	omega0original <- omega0
	eta1 <- eta0
	omega1 <- omega0
	while((L2norm(diff0) > 5e-006)) {
		diff0 <- ML(omega0, M, W) - fbeta
		svd.dML <- svd(dML(omega0, M, W)[, 1:dim.mr])
		cond <- min(svd.dML$d)
		if(cond < 10^(-5))
			break
		deta <-  - t(svd.dML$u %*% (t(svd.dML$v) * (1/svd.dML$d))) %*% 
			diff0
		i <- 0
		eta1 <- eta0 + deta
		omega1[1:dim.mr] <- eta1
		diff1 <- ML(omega1, M, W) - fbeta
		if(any(is.na(diff1)))
			break
		while((L2norm(diff1) > L2norm(diff0) - 2 * 0.0001 * L2norm(
			diff0)) && (i < 10)) {
			i <- i + 1	#print(i)
			eta1 <- eta0 + deta/2^i
			omega1[1:dim.mr] <- eta1
			diff1 <- ML(omega1, M, W) - fbeta
		}
#print(eta1)
		eta0 <- eta1
		omega0 <- omega1
	}
	if(cond < 10^(-5) || any(is.na(diff1))) {
		print("Using nlminb")
		eta0 <- nlminb(eta0original, eta.ls, control = nlminb.control(
			iter.max = 150, abs.tol = 10^(-12), rel.tol = 10^(-6)), 
			av = list(omega0original, fbeta, M, W))[[1]]
		omega0[1:dim.mr] <- eta0
		diff0 <- ML(omega0, M, W) - fbeta
		if(L2norm(diff0) > 5e-006)
			print("Spurios convergence")
	}
	eta0
}
"solve.theta.efficient"<-
function(theta, omega, h, y, M, W, maxiter = 25)
{
	n <- apply(y, 2, sum)
	p <- ncol(y)
	lt <- length(theta)
	score <- matrix(0, lt, p)
	score.sum <- matrix(0, lt, 1)
	information <- array(0, c(lt, lt, p))	# Using the transpose:
	tdomega.dtheta.value <- array(0, c(lt, nrow(omega), p))
	information.sum <- matrix(0, lt, lt)
	i <- 0
	for(k in 1:p) {
# Update omega
		omega[, k] <- omega.theta(theta, omega[, k], h[[k]], M, W)
		tdomega.dtheta.value[,  , k] <- t(domega.dtheta(theta, h[[k]], 
			omega[, k], M, W))
		score[, k] <- tdomega.dtheta.value[,  , k] %*% score.omega(
			omega[, k], y[, k], n[k], W)
		information[,  , k] <- tdomega.dtheta.value[,  , k] %*% 
			info.omega(omega[, k], y[, k], n[k], W) %*% t(
			tdomega.dtheta.value[,  , k])
	}
	score.sum <- score[, 1]
	information.sum <- information[,  , 1]
	if(p > 1) {
		for(k in 2:p) {
			score.sum <- score.sum + score[, k]
			information.sum <- information.sum + information[,  , k
				]
		}
	}
# Update theta
	theta <- theta + solve(information.sum, score.sum)
	i <- i + 1
	cat("  Iteration", "Score norm", "\n")
	cat("\t", i, L2norm(score.sum), "\n")
	while(L2norm(score.sum) > 5e-007 && i < maxiter) {
		for(k in 1:p) {
# Update omega
			omega[, k] <- omega.theta(theta, omega[, k], h[[k]], M, 
				W)
			tdomega.dtheta.value[,  , k] <- t(domega.dtheta(theta, 
				h[[k]], omega[, k], M, W))
			score[, k] <- tdomega.dtheta.value[,  , k] %*% 
				score.omega(omega[, k], y[, k], n[k], W)
			information[,  , k] <- tdomega.dtheta.value[,  , k] %*% 
				info.omega(omega[, k], y[, k], n[k], W) %*% t(
				tdomega.dtheta.value[,  , k])
		}
		score.sum <- score[, 1]
		information.sum <- information[,  , 1]
		if(p > 1) {
			for(k in 2:p) {
				score.sum <- score.sum + score[, k]
				information.sum <- information.sum + 
				  information[,  , k]
			}
		}
# Update theta
		theta <- theta + solve(information.sum, score.sum)
		i <- i + 1
		cat("\t", i, L2norm(score.sum), "\n")
	}
	list(theta = theta, omega = omega, score = score.sum, information = 
		information.sum, W = W, M = M, y = y, n.iter = i)
}
"standardM"<-
function(dim.vector, cumulative = T)
{
	if(length(dim.vector) == 1)
		dim.vector <- c(dim.vector, 1)
	id <- diag(length(dim.vector))
	if(cumulative) {
		M <- (LowerOne(dim.vector[1]) %*% MarginalsB(dim.vector, id[1,  
			]))[ - dim.vector[1],  ]
		for(i in 2:length(dim.vector)) {
			M <- rbind(M, (LowerOne(dim.vector[i]) %*% MarginalsB(
				dim.vector, id[i,  ]))[ - dim.vector[i],  ])
		}
	}
	else {
		M <- (MarginalsB(dim.vector, id[1,  ]))[ - dim.vector[1],  ]
		for(i in 2:length(dim.vector)) {
			M <- rbind(M, (MarginalsB(dim.vector, id[i,  ]))[ - 
				dim.vector[i],  ])
		}
	}
	M
}
"standardW"<-
function(dim.vector)
{
	if(length(dim.vector) == 1)
		dim.vector <- c(dim.vector, 1)
	id <- diag(length(dim.vector))
	W1 <- (MarginalsB(dim.vector, id[1,  ]))[ - dim.vector[1],  ]
	for(i in 2:length(dim.vector)) {
		W1 <- rbind(W1, (MarginalsB(dim.vector, id[i,  ]))[ - 
			dim.vector[i],  ])
	}
	W <- t(W1)
	W
}
"summary.bam"<-
function(bamfit)
{
	cat("\n")
	cat("Area under the curve (AUC) estimate with standard error")
	area.output <- round(unlist(bamfit$area), 4)
	if(length(area.output) > 2)
		cat(" for system I and II:\n")
	else cat(":\n")
	print(round(unlist(bamfit$area), 4))
	cat("\n")
	gof.output <- round(unlist(gof.bam(bamfit))[c(3, 6, 7)], 3)
	names(gof.output) <- c("X2", "G2", "df")
	print(gof.output)
	cat("\n")
	param.output <- data.frame(Parameter = label.bam(bamfit), round(
		wald.bam(bamfit), 3))
	print(param.output)
	cat("\n")
	cat("log-likelihood: ", round(sum(diag(t(bamfit$y) %*% log(pi.omega(
		bamfit$omega, bamfit$W)))), 3))
	cat("\n")
	if(nchar(bamfit$AmodelName)) {
		cat("Association model:", bamfit$AmodelName, "\n\n")
		if(bamfit$EqualA)
			cat("Model fitted with equal association parameters across populations.",
				"\n\n")
		else cat("")
	}
	else cat("")
}
"wald"<-
function(theta, information)
{
	se <- sqrt(diag(solve(information)))
	walds <- theta/se
	data.frame(Estimate = theta, SE = se, Wald = walds, pvalue = 2 * pnorm( -
		abs(walds)))
}
"wald.bam"<-
function(bamfit)
{
	theta <- bamfit$theta
	information <- bamfit$information
	wald(theta, information)
}
"wls.approx"<-
function(y, d)
{
# This function returns initial values for univariate or bivariate ROC data.  The initial values
# are returned as (cutp[1],...,cutp[d-1],mu,0) for each population.  Note that the initial value for the
# scale parameter is zero.
	M <- standardM(d)
	X <- shiftX(d)
	n <- apply(y, 2, sum)
	piv <- y
	if(any(n > 1.5))
		piv <- t(t(y)/n)
	Ms <- DirectSum(M, M)
	Ms.piv <- Ms %*% c(piv)
	Q <- dnum(qnorm, Ms.piv) %*% Ms
	V1 <- (diag(piv[, 1]) - piv[, 1] %*% t(piv[, 1]))
	V2 <- (diag(piv[, 2]) - piv[, 2] %*% t(piv[, 2]))
	V <- DirectSum(V1, V2)
	VF <- Q %*% V %*% t(Q)
	invVF <- solve(VF)
	beta <- solve((t(X) %*% invVF %*% X)) %*% t(X) %*% invVF %*% qnorm(
		Ms.piv)
	if(length(d) == 1)
		c(beta, 0)
	else c(beta[1:d[1]], 0, beta[(d[1] + 1):length(beta)], 0)
}
