/*
The source code is modified by ivind Skare
for use in vorSim2D, Version 1.0, Feb 15, 2001.
*/
/* hull.c : "combinatorial" functions for hull computation */

/*
 * Ken Clarkson wrote this.  Copyright (c) 1995 by AT&T..
 * Permission to use, copy, modify, and distribute this software for any
 * purpose without fee is hereby granted, provided that this entire notice
 * is included in all copies of any software which is or includes a copy
 * or modification of this software and in all copies of the supporting
 * documentation for such software.
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE ANY
 * REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY
 * OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <stdarg.h>
#include <assert.h>
#include <string.h>

#include "hull.h"
#include "hullOwn.h"
#include "../vertex.h"

/*#define PRINT 1*/

site p;
long pnum=0;

int	rdim,	/* region dimension: (max) number of sites specifying region */
  cdim,	/* number of sites currently specifying region */
  site_size, /* size of malloc needed for a site */
  point_size;  /* size of malloc needed for a point */

STORAGE(simplex)

  // #define push(x) *(st+tms++) = x;
  // #define pop(x)  x = *(st + --tms);
  // #define initStackMemory() if (!st) st = (simplex **)malloc((ss+MAXDIM+1)*sizeof(simplex*));

// #define stackMemory() if (tms>ss) assert(st=(simplex**)realloc(st, ((ss+=ss)+MAXDIM+1)*sizeof(simplex*)));

  int checkIfRemoved(simplex * s);


void shareEdge(simplex * s1, simplex * s2, point * a, int * indexI, int * indexJ);

void *visit_triang_gen(simplex *s, visit_func *visit, test_func *test) {
  /* 
   * starting at s, visit simplices t such that test(s,i,0) is true,
   * and t is the i'th neighbor of s;
   * apply visit function to all visited simplices;
   * when visit returns nonNULL, exit and return its value
   */
  neighbor *sn;
  void *v;
  simplex *t;
  int i;
  long tms = 0;
  static long vnum = -1;
  static long ss = 2000;
  static simplex **st;

  vnum--;
  if (!st) assert(st=(simplex**)malloc((ss+MAXDIM+1)*sizeof(simplex*)));
  if (s) push(s);
  while (tms) {

    if (tms>ss) {DEBEXP(-1,tms);
    assert(st=(simplex**)realloc(st,
				 ((ss+=ss)+MAXDIM+1)*sizeof(simplex*)));
    }
    pop(t);
    if (!t || t->visit == vnum) continue;
    t->visit = vnum;
    v=(*visit)(t,0);
    if (v) {return v;}
    for (i=-1,sn = t->neigh-1;i<cdim;i++,sn++)
      if ((sn->simp->visit != vnum) && sn->simp && test(t,i,0))
	push(sn->simp);
  }
  return NULL;
}

int truet(simplex *s, int i, void *dum) {return 1;}

void *visit_triang(simplex *root, visit_func *visit)
  /* visit the whole triangulation */
{return visit_triang_gen(root, visit, truet);}


int hullt(simplex *s, int i, void *dummy) {return i>-1;}

void *facet_test(simplex *s, void *dummy) {return (!s->peak.vert) ? s : NULL;}

void *visit_hull(simplex *root, visit_func *visit)
  /* visit all simplices with facets of the current hull */
{return visit_triang_gen((simplex *) visit_triang(root, &facet_test),
			 visit, hullt);}


neighbor * lookupsimp(simplex * a, simplex * b) {
  int i;
  neighbor *x;

  for (i=0, x = a->neigh; (x->simp != b) && (i<cdim) ; i++, x++);

  if (i<cdim)
    return x;
  else {
    printf("Error(lookupsimp): adjacency failure - iteration %d\n", gIt);
    printTriangle(a);
    printTriangle(b);
    exit(-1);
  }
  
  return x;
}
	
neighbor * lookupvert(simplex * a, site b) {
  int i;
  neighbor *x;
  for (i=0, x = a->neigh; (x->vert != b) && (i<cdim) ; i++, x++);
  if (i<cdim)
    return x;
  else {
    printf("Error(lookupvert): adjacency failure - iteration %d\n", gIt);
    printTriangle(a);
    printf("b: %6.2f %6.2f\n", b[0], b[1]);
    exit(-1);
  }
  
  return x;
}

#define lookup(a,b,what,whatt)						\
{									\
	int i;								\
	neighbor *x;							\
	for (i=0, x = a->neigh; (x->what != b) && (i<cdim) ; i++, x++);	\
	if (i<cdim)							\
		return x;						\
	else {								\
		fprintf(DFILE,"adjacency failure,op_" #what ":\n");	\
		DEBTR(-10)						\
		print_simplex_f(a, DFILE, &print_neighbor_full);	\
		print_##whatt(b, DFILE);				\
		fprintf(DFILE,"---------------------\n");		\
		print_triang(a,DFILE, &print_neighbor_full);		\
		exit(1);						\
		return 0;						\
	}								\
}									\


neighbor *op_simp(simplex *a, simplex *b) {
  //  lookup(a,b,simp,simplex)
  return lookupsimp(a, b);
}
/* the neighbor entry of a containing b */

neighbor *op_vert(simplex *a, site b) {
  //  lookup(a,b,vert,site)
  return lookupvert(a, b);
}
/* the neighbor entry of a containing b */


void connect(simplex *s) {
  /* make neighbor connections between newly created simplices incident to p */

  site xf,xb,xfi;
  simplex *sb, *sf, *seen;
  int i;
  neighbor *sn;

  if (!s) return;
  assert(!s->peak.vert
	 && s->peak.simp->peak.vert==p
	 && !op_vert(s,p)->simp->peak.vert);
  if (s->visit==pnum) return;
  s->visit = pnum;
  seen = s->peak.simp;
  xfi = op_simp(seen,s)->vert;
  for (i=0, sn = s->neigh; i<cdim; i++,sn++) {
    xb = sn->vert;
    if (p == xb) continue;
    sb = seen;
    sf = sn->simp;
    xf = xfi;
    if (!sf->peak.vert) {	/* are we done already? */
      sf = op_vert(seen,xb)->simp;
      if (sf->peak.vert) continue;				
    } else do {
      xb = xf;
      xf = op_simp(sf,sb)->vert;
      sb = sf;
      sf = op_vert(sb,xb)->simp;
    } while (sf->peak.vert);

    sn->simp = sf;
    op_vert(sf,xf)->simp = s;

    connect(sf);
  }

}

				
simplex *make_facets(simplex *seen) {
  /*
   * visit simplices s with seesOwn(p,s), and make a facet for every neighbor
   * of s not seen by p
   */
  int isSeen;
  simplex * s;
  simplex *n;
  static simplex *ns;
  neighbor *bn;
  int i, j;


  if (!seen) return NULL;
  seen->peak.vert = p;

  removeTriangle(seen);

  for (i=0,bn = seen->neigh; i<cdim; i++,bn++) {
    n = bn->simp;
    if (pnum != n->visit) {
      n->visit = pnum;
      isSeen = seesOwn(p,n);
      if (isSeen == 1) {
	s = make_facets(n);
	if (!s) // Areasign has returned 0
	  return NULL;
      }
      else if (isSeen == -1)
	return NULL; // Areasign has returned 0
    } 
    if (n->peak.vert)
      continue;
    copy_simp(ns,seen);
    ns->visit = 0;
    ns->peak.vert = 0;
    ns->normal = 0;
    ns->infinity = 0; /* SKARE */
    ns->removed = 0; /* SKARE */
    for (j=0;j<pdim;j++) {
      ns->centerpoint[j] = 0.0; /* SKARE */
      ns->direction[j] = -1000.00; /* SKARE */
    }
    ns->peak.simp = seen;
    /*		ns->Sb -= ns->neigh[i].basis->sqb; */
    NULLIFY(basis_s,ns->neigh[i].basis);
    ns->neigh[i].vert = p;
    bn->simp = op_simp(n,seen)->simp = ns;

    addTriangle(ns);
  }
  return ns;
}



simplex *extend_simplices(simplex *s) {
  /*
   * p lies outside flat containing previous sites;
   * make p a vertex of every current simplex, and create some new simplices
   */

  int	i,
    ocdim=cdim-1;
  simplex *ns;
  neighbor *nsn;

  if (s->visit == pnum) return s->peak.vert ? s->neigh[ocdim].simp : s;
  s->visit = pnum;
  s->neigh[ocdim].vert = p;
  NULLIFY(basis_s,s->normal);
  NULLIFY(basis_s,s->neigh[0].basis);
  if (!s->peak.vert) {
    s->neigh[ocdim].simp = extend_simplices(s->peak.simp);
    return s;
  } else {
    copy_simp(ns,s);
    s->neigh[ocdim].simp = ns;
    ns->peak.vert = NULL;
    ns->peak.simp = s;
    ns->neigh[ocdim] = s->peak;
    inc_ref(basis_s,s->peak.basis);
    for (i=0,nsn=ns->neigh;i<cdim;i++,nsn++)
      nsn->simp = extend_simplices(nsn->simp);
  }
  return ns;
}


simplex *search(simplex *root) {
  /* return a simplex s that corresponds to a facet of the 
   * current hull, and seesOwn(p, s) */

  int isSeen;
  simplex *s;
  static simplex **st;
  static long ss = MAXDIM;
  neighbor *sn;
  int i;
  long tms = 0;

  if (!st) st = (simplex **)malloc((ss+MAXDIM+1)*sizeof(simplex*));
  push(root->peak.simp);
  root->visit = pnum;
  isSeen = seesOwn(p,root);
  if (isSeen == -1)
    return NULL; // Area sign = 0, must abort building a hull
  else if (isSeen == 0)
    for (i=0,sn=root->neigh;i<cdim;i++,sn++) push(sn->simp);
  while (tms) {
    if (tms>ss) 
      assert(st=(simplex**)realloc(st,
				   ((ss+=ss)+MAXDIM+1)*sizeof(simplex*)));
    pop(s);
    if (s->visit == pnum) continue;
    s->visit = pnum;
    isSeen = seesOwn(p,s);
    if (isSeen == -1)
      return NULL; // Area sign = 0, must abort building a hull
    else if (isSeen == 1) {
      if (!s->peak.vert) return s;
    }
    for (i=0, sn=s->neigh; i<cdim; i++,sn++) push(sn->simp);
  }
  return NULL;
}

void removeSimplexInMatrix(simplex * s) {
  point pt = s->centerpoint;
  int index;
  // Not correct - must remove simplex from gSimplex..
  index = (int) (floor(pt[0]/2.5) + 40 * floor(pt[1]/2.5));
  if (n40[index] > 0)
    n40[index]--;

  index = (int) (floor(pt[0]/5) + 20 * floor(pt[1]/5));
  if (n20[index] > 0)
    n20[index]--;

  index = (int) (floor(pt[0]/10) + 10 * floor(pt[1]/10));
  if (n10[index] > 0)
    n10[index]--;

  return;
}

void insertSimplexInMatrix(simplex * s) {
  point pt = s->centerpoint;
  int index;
  
  index = (int) (floor(pt[0]/2.5) + 40 * floor(pt[1]/2.5));
  if (n40[index] < 10)
    gSimplex40[index][n40[index]++] = s;

  index = (int) (floor(pt[0]/5) + 20 * floor(pt[1]/5));
  if (n20[index] < 40)
    gSimplex20[index][n20[index]++] = s;

  index = (int) (floor(pt[0]/10) + 10 * floor(pt[1]/10));
  if (n10[index] < 160)
    gSimplex10[index][n10[index]++] = s;

  return;
}


simplex ** startSearch() {
  int index;
  
  index = (int) (floor(p[0]/2.5) + 40 * floor(p[1]/2.5));
  if (gSimplex40[index][0])
    return gSimplex40[index];    
  
  index = (int) (floor(p[0]/5) + 20 * floor(p[1]/5));
  if (gSimplex20[index][0])
    return gSimplex20[index];
  
  index = (int) (floor(p[0]/10) + 10 * floor(p[1]/10));
  if (gSimplex10[index][0])
    return gSimplex10[index];
  
//   index = floor(p[0]/20) + 5 * floor(p[1]/20);
//   if (gSimplex5[index][0])
//     return gSimplex5[index];
    
  return NULL;
}

point get_another_site(void) {
  point pnext;

  pnext = (*get_site)();
  if (!pnext) return NULL;
  pnum++; // = site_num(pnext)+2;
  return pnext;
}

int counter = -1;

void addEdge(Edge ** first, int * nEdges, point p1, point p2) {
  int ind=0,  newInd;
  int tmp;
  Edge * e;
  Edge * newE = (Edge *) malloc(sizeof(Edge));

  newE->pt1 = newE->pt2 = NULL;
  newE->p1 = site_num(p1);
  newE->p2 = site_num(p2);
  assert(newE->p1 >= 0 && newE->p2 >= 0);
  if (newE->p1 < newE->p2) {
    tmp = newE->p1;
    newE->p1 = newE->p2;
    newE->p2 = tmp;
  }

  newInd = newE->p1 * gMaxPos + newE->p2;

  (*nEdges)++;

  e = *first;
  if (e == NULL) {
    (*first) = newE;
    newE->next = NULL;
    newE->prev = NULL;
  }
  else {
    while (e->next) {
      ind = gMaxPos * e->p1 + e->p2;
      if (ind < newInd)
	e = e->next;
      else
	break;
    }
    if (ind == newInd) {
      (*nEdges)--;
      free(newE);
    }
    else if (ind > newInd) {
      newE->prev = e->prev;
      newE->next = e;
      if (e->prev != NULL)
	e->prev->next = newE;
      else
	(*first) = newE;
      e->prev = newE;
    }
    else if (ind < newInd) { /* e->next == NULL */
      ind = gMaxPos * e->p1 + e->p2;
      if (ind == newInd) {
	(*nEdges)--;
	free(newE);
      }
      else if (ind < newInd) {
	e->next = newE;
	newE->prev = e;
	newE->next = NULL;
      }
      else {
	newE->prev = e->prev;
	newE->next = e;
	if (e->prev != NULL)
	  e->prev->next = newE;
	else
	  (*first) = newE;
	e->prev = newE;
      }
    }
  }

  if (gDebugPrint >= 2) {
    if (ind != newInd) {
      printf("Add Edge (%d %d) - # = %d\n", newE->p1, newE->p2, gnEdges);
      if (newE->prev)
	printf("prev edge (%d %d)\n", newE->prev->p1, newE->prev->p2);
      else
	printf("prev edge = NULL\n");
    }
  }
  return;
}

/* t contains the vertices of a triangle */
void findCircle(point * p, simplex *s) {
  int i;
  int index = -1;
  double A, B, C, D, E, F, G;
  double p0, p1;
  point pt1, pt2;

  for (i=0;i<cdim;i++)
    if (s->neigh[i].vert == infinity) {
      index = i;
      break;
    }

  s->infinity = (index >= 0);
  /*  assert((s->infinity && index >= 0) || (!s->infinity && index < 0)); */

  if (index >= 0) { /* s->infinity == 1 */
    if (index == 0) {
      pt1 = p[1];
      pt2 = p[2];
    }
    else if (index == 1) {
      pt1 = p[0];
      pt2 = p[2];
    }
    else {
      pt1 = p[0];
      pt2 = p[1];
    }

    s->centerpoint[0] = 0.5*(pt1[0] + pt2[0]);
    s->centerpoint[1] = 0.5*(pt1[1] + pt2[1]);

    /*    s->direction = (double *) malloc(2*sizeof(double)); */
    s->direction[0] = pt2[1]-pt1[1];
    s->direction[1] = -(pt2[0]-pt1[0]);
  }
  else {
    A = p[1][0] - p[0][0];
    B = p[1][1] - p[0][1];
    C = p[2][0] - p[0][0];
    D = p[2][1] - p[0][1];
    E = A*(p[0][0]+p[1][0]) + B*(p[0][1]+p[1][1]);
    F = C*(p[0][0]+p[2][0]) + D*(p[0][1]+p[2][1]);
    G = 2*(A*(p[2][1]-p[1][1]) - B*(p[2][0]-p[1][0]));
  
    p0 = (D*E - B*F)/G;
    p1 = (A*F - C*E)/G;
  

    s->centerpoint[0] = p0;
    s->centerpoint[1] = p1;

    s->radius = (p[0][0]-p0)*(p[0][0]-p0) + (p[0][1]-p1)*(p[0][1]-p1);
  }
  return;
}



void printAllNeigh() {
  Vertex * v = gFirstVertex;

  while (v) {
    printf("Vertex: ");
    v->print();
    v->printNeigh();

    v = v->next;
  }
  return;
}    

void printVertices() {
  Vertex * v = gFirstVertex;

  while (v) {
    v->print();
    v = v->next;
  }

  return;
}

void printEdge(Edge * e) {
  printf("(%6.2f %6.2f) (%6.2f %6.2f)\n", e->pt1[0], e->pt1[1],
	 e->pt2[0], e->pt2[1]);
  return;
}
void printEdges(Vertex * v) {
  Edge * e = v->dirEdges;
  while (e) {
    printf("(%6.2f %6.2f) (%6.2f %6.2f)\n", e->pt1[0], e->pt1[1],
	   e->pt2[0], e->pt2[1]);
    e = e->next;
  }

  return;
}

simplex * findFirst(simplex * s) {
  int i;
  simplex * t;
  simplex * u;

  s->visit = counter;
  if (!s->peak.vert) {
    if (s->infinity) {
      for (i=0;i<cdim;i++)
	if (s->neigh[i].vert == infinity)
	  break;
      s = s->neigh[i].simp;
    }
    return s;
  }
  for (i=0;i<cdim;i++) {
    t = s->neigh[i].simp;
    if (t->visit != counter) {     
      u = findFirst(t);
      if (u) {
	if (u->infinity) {
	  for (i=0;i<cdim;i++)
	    if (u->neigh[i].vert == infinity)
	      break;
	  u = u->neigh[i].simp;
	}
	return u;
      }
    }
  }
  return NULL;
}

void printAllRec(simplex * s) {
  int i;
  simplex * t;

  s->visit = counter;
  assert(!s->peak.vert);
  printTriangle(s);
  if (gDebugCheck)
    checkIfRemoved(s);

  for (i=0;i<cdim;i++) {
    t = s->neigh[i].simp;
    if (t->visit != counter) {     
      printAllRec(t);
    }
  }
  return;
}

void printAll(simplex * s) {
  simplex * start;
  
  counter--;
  start = findFirst(s);
  assert(!start->peak.vert);
  counter--;
  printAllRec(start);

  return;
}

void printTriangle(simplex * s) {
  printf("(%6.2f %6.2f) (%6.2f %6.2f) (%6.2f %6.2f)\n",
	 s->neigh[0].vert[0], s->neigh[0].vert[1],
	 s->neigh[1].vert[0], s->neigh[1].vert[1],
	 s->neigh[2].vert[0], s->neigh[2].vert[1]);
  return;
}

void printTriangles(Vertex * v) {
  int i;

  for (i=0;i<v->nSimp;i++)
    printTriangle(v->simp[i]);

  return;
}

int checkIfRemoved(simplex * s) {
  int i;

  for (i=0;i<cdim;i++) {
    if (s->neigh[i].vert[0] == REMOVED_MARK)
      break;
  }

  return (i == cdim);
}

int checkAllRec(simplex * s) {
  int i;
  int ok = 1;
  double tolerance = 0.0000001;
  simplex * t;
  point c;

  s->visit = counter;
  assert(!s->peak.vert);
  ok = checkIfRemoved(s);

  if (!ok)
    return ok;

  if (s->it < 0) {
    printf("Simplex with it < 0: simplex from rootNew-structure\n");
    printTriangle(s);
    ok = 0;
  }

  if (!ok)
    return ok;

  if (s->normal) {
    c = s->normal->vecs;
    assert(!(s->infinity &&
	     (c[2] < -tolerance || c[2]> tolerance)));
  }


  for (i=0;i<cdim;i++) {
    t = s->neigh[i].simp;
    if (t->visit != counter) {     
      ok = checkAllRec(t);
      if (!ok)
	break;
    }
  }
  return ok;
}

int simplexInVertex(simplex * s) {
  int i, j;
  int found = 1;
  Vertex * v;

  for (i=0;i<cdim;i++) {
    if (s->neigh[i].vert != infinity) {
      v = findVertex(s->neigh[i].vert);
      for (j=0;j<v->nSimp;j++)
	if (v->simp[j] == s)
	  break;
      if (j == v->nSimp) {
	printf("Error(simplexInVertex): Simplex not found in v\n");
	printf("Simplex: ");
	printTriangle(s);
	printf("Vertex: ");
	v->print();
	printf("Neighbouring simplices of v\n");
	printTriangles(v);
      	
	found = 0;
	break;
      }
    }
  }

  return found;
}

int checkSimplex(simplex * s);

int checkAll(simplex * s) {
  int i;
  int ok;
  simplex * t;
  simplex * w;
  simplex * start;
  Vertex * v = gFirstVertex;

  static simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

 
  // Check simplex structure if any REMOVED_MARK appears
  counter--;
  start = findFirst(s);
  assert(!start->peak.vert);
  counter--;
  ok = checkAllRec(start);

  if (!ok)
    return ok;

  // Check if every simplex of v is a member of root
  while (v) {
    if (!v->nSimp) {
      printf("Error(checkAll): v->nSimp == 0\n");
      v->print();
      ok = 0;
      break;
    }
    else if (!v->nNeigh) {
      printf("Error(checkAll): v->nNeigh == 0\n");
      v->print();
      ok = 0;
      break;
    }
    else {
      for (i=0;i<v->nSimp;i++) {
	if (v->simp[i]->visit != counter) {
	  printf("Error(checkAll): v->simp[i]->visit != counter\n");
	  printf("Vertex: ");
	  v->print();
	  printf("Triangle: ");
	  printTriangle(v->simp[i]);

	  ok = 0;
	  break;
	}
      }
    }
    if (!ok)
      break;

    v = v->next;
  }
  
  if (!ok)
    return ok;

  // Check: s member of root => there exists a vertex v such
  // that s is a neighbouring simplex of v

  counter--;
  initStackMemory();
  push(start);
  start->visit = counter;
  while (tms) {
    stackMemory();
    pop(t);

    ok = simplexInVertex(t);

    if (!ok)
      break;
    else {
      ok = checkSimplex(t);
      if (!ok)
	break;
    }

    for (i=0;i<cdim;i++) {
      w = t->neigh[i].simp;
      if (w->visit != counter) {
	w->visit = counter;
	push(w);
      }
    }    
  }

  return ok;
}

void addTriangle(simplex * s) {
  int i, j;
  neighbor *neigh = s->neigh;
  point * pt;
  Vertex * v[3];

  if (cdim < rdim)
    return;

  pt = (point *) malloc(cdim*sizeof(point));

  pt[0] = neigh[0].vert;
  pt[1] = neigh[1].vert;
  pt[2] = neigh[2].vert;

  /*assert(s->centerpoint[0] == 0.0);*/
  /*s->centerpoint = (point) malloc(cdim*point_size);*/

  s->it = -gIt;

  if (gDebugPrint >= 2) {
    printf("Adds triangle:\t  ");
    printTriangle(s);  
  }

  findCircle(pt, s);


  if (!gLocalSimplexStructure) {
    s->it = gIt;
    /* for all neighbours of s */
    for (i=0;i<cdim; i++) {
      if (pt[i] != infinity) {
	/* Find vertex pt[i] */
	v[i] = findVertex(pt[i]);
	if (v[i]->nSimp >= 10)
	  v[i]->simp = (simplex **) realloc(v[i]->simp, (v[i]->nSimp+1)*sizeof(simplex *));
	/* Insert simplex into the data structure of v */
	v[i]->simp[v[i]->nSimp++] = s;
      }
    }
    if (!s->infinity) {
      for (i=0;i<cdim-1; i++) {
	for (j=i+1;j<cdim; j++) {
	  addNeigh(v[i], v[j]);
	  addNeigh(v[j], v[i]);
	}
      }
    }
    //    insertSimplexInMatrix(s);
  }

  free(pt);

  return;
}
  
void removeTriangle(simplex * s) {
  int i, j, k;
  neighbor * neigh = s->neigh;
  point * pt;
  Vertex * v[3];

  if (cdim < rdim)    
    return;

  pt = (point *) malloc(cdim*sizeof(point));

  pt[0] = neigh[0].vert;
  pt[1] = neigh[1].vert;
  pt[2] = neigh[2].vert;

  if (gDebugPrint >= 2) {
    printf("Removes triangle: ");
    printTriangle(s);
  }

  if (!gLocalSimplexStructure) {
    /* for all neighbours of s */
    for (i=0;i<cdim; i++) {
      /* Find vertex pt[i] */
      if (pt[i] != infinity) {
	v[i] = findVertex(pt[i]);

	/* Remove simplex from the data structure of v */
	for (j=0;j<v[i]->nSimp;j++) {
	  if (v[i]->simp[j] == s)
	    break;
	}
	if (j == v[i]->nSimp) {
	  printf("Error(removeTriangle): Neighbouring simplex not found\n");
	  printf("Simplex: ");
	  printTriangle(s);
	  printf("Vertex: ");
	  v[i]->print();
	  printf("with neighbouring simplices:\n");
	  printTriangles(v[i]);

	  assert(j<v[i]->nSimp);
	}
	k = j;

	for (j=k;j<v[i]->nSimp-1;j++) {
	  v[i]->simp[j] = v[i]->simp[j+1];
	}
	v[i]->simp[v[i]->nSimp-1] = NULL;
	v[i]->nSimp--;
      }
    }
    if (!s->infinity) {
      for (i=0;i<cdim-1; i++) {
	for (j=i+1;j<cdim; j++) {
	  removeNeigh(v[i], v[j]);
	  removeNeigh(v[j], v[i]);
	}
      }
    }
    //    removeSimplexInMatrix(s);
  }

  free(pt);

  return;
}

void initializeVertexList(simplex * s) {
  int i;

  /*assert(s->peak.vert == NULL);*/

  if (s->visit != counter) {
    s->visit = counter;

    for (i=0;i<cdim;i++)
      initializeVertexList(s->neigh[i].simp);
    
    if (s->peak.vert != NULL)
      return;
    
    for (i=0;i<cdim;i++) {
      if (s->neigh[i].vert == infinity) {
	s->infinity = 1;
	break;
	/*	return;*/
      }
    }
    
    addTriangle(s);
  }

  return;
}
  
int out_of_flatOwn(simplex *root, point p) {
  int out = out_of_flat(root,p);
  int outOwn;
  int area;

  if (cdim < 2)
    outOwn = 1;
  else if (root->neigh[0].vert == infinity || root->neigh[1].vert == infinity) {
    outOwn = -1; // Not defined
  }
  else {
    area = Areasign(root->neigh[0].vert, root->neigh[1].vert, p);
    outOwn = (area != 0);
  }

  if (outOwn != out && outOwn != -1) {
    if (outOwn == 0 || gDebugPrint > 0)
      printf("Warning(out_of_flatOwn): out %d != outOwn %d\n", out, outOwn);
    out = outOwn;
    cdim++;
  }

  return out;
}

int buildhull(simplex *root) {
  //  int i;
  //  double eps = 0.00000001;
  simplex * s = NULL;
  while (cdim < rdim) {
    p = get_another_site();
    if (!p) return 1;
    if (out_of_flatOwn(root,p))
      extend_simplices(root);
    else {
      if (gDebugPrint > 0)
	printf("Warning(buildhull): Not out of flat\n");
      return 0;
    }
  }
  /* Initialize gFirstEdge - structure */

  initializeVertexList(root);
  p = get_another_site();
  while (p) {
    s = search(root);
    if (!s) {
      printf("Warning(buildhull): search(root) returns NULL\n");
      return 0;
    }

    //    do {
    s = make_facets(s);
    if (!s) {
      printf("Warning(buildhull): make_facets(s) returns NULL\n");
      return 0;
      //      printf("Pertubate input point\n");
      //      for (i=0;i<pdim;i++)
      //	p[i] += eps;
    }
    //    } while (!s);

    connect(s);
    p = get_another_site();
  }
  return 1;
}

int addPointToConvexHull(simplex *root) {
  simplex * s = NULL;
  p = get_another_site();

  s = search(root);
  if (!s) {
    printf("Warning(addPointToConvexHull): search(root) returns NULL\n");
    //    assert(s);
    return 0;
  }
  s = make_facets(s);
  if (!s) {
    printf("Warning(addPointToConvexHull): make_facets(s) returns NULL\n");
    //    assert(s);
    return -1;
  }
  connect(s);

  return 1;
}

void merge(simplex ** root, simplex * rootNew, Vertex * v);

double dist(point pt1, point pt2) {
  int i;
  double d;
  double sum;

  sum = 0.0;
  for (i=0;i<pdim;i++) {
    d = pt1[i]-pt2[i];
    sum += d*d;
  }

  return sum;
}

int checkSimplex(simplex * s) {
  int j, k, l;
  int n;
  int sign1, sign2;
  double r;
  double tolerance = 0.000001;
  point pt;
  point a;
  point pts[3];
  int check = 1;
  simplex * t;

  if (!s->infinity) {

    if (gDebugPrint >= 2)
      printTriangle(s);

    pt = s->centerpoint;
    r = s->radius;


    for (j=0;j<cdim;j++) {
      t = s->neigh[j].simp;
      if (!t->infinity) {
	for (k=0;k<cdim;k++) {	
	  for (l=0;l<cdim;l++) {	
	    if (t->neigh[k].vert == s->neigh[l].vert)
	      break;
	  }
	  if (l==cdim)
	    break;
	}

	assert(k<cdim);
	  
	if (dist(pt, t->neigh[k].vert) < r - tolerance) {
	  printf("Error(checkSimplex): vertex within circle of s\n");
	  printf("Vertex: ");
	  findVertex(t->neigh[k].vert)->print();
	  printf("Triangle s: ");
	  printTriangle(s);
	  printf("Triangle t: ");
	  printTriangle(t);
	  check = 0;
	  break;
	}
	else {
	  // Check if the two neighbouring simplices are on opposite sides
	  n = 0;
	  for (l=0;l<cdim;l++) {
	    if (l != j)
	      pts[n++] = s->neigh[l].vert;
	  }
	  pts[cdim-1] = s->neigh[j].vert;
	  a = t->neigh[k].vert;

	  sign1 = Areasign(pts[0], pts[1], pts[2]);
	  sign2 = Areasign(pts[0], pts[1], a);
	  if (!(sign1 == 0 || sign2 == 0))	    
	    check = (sign1 != sign2);

	  if (!check) {
	    printf("Error(checkSimplex): simplex s and t on same side\n");
	    printf("s: ");
	    printTriangle(s);
	    printf("t: ");
	    printTriangle(t);
	  }
	}
      }
      if (!check)
	break;
    }
  }
  return check;
}

int checkNewRoot(simplex * root) {
  int i;
  int check = 1;
  simplex * s;
  simplex * t;
  simplex * start;

  static simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  counter--;
  start = findFirst(root);
  counter--;
  initStackMemory();
  push(start);
  start->visit = counter;
  while(tms) {
    stackMemory();
    pop(s);

    check = checkSimplex(s);
    if (!check)
      break;
    else {
      for (i=0;i<cdim;i++) {
	t = s->neigh[i].simp;
	if (t->visit != counter) {
	  push(t);
	  t->visit = counter;
	}
      }
    }
  }

  return check;
}


void removePointFromConvexHull(simplex ** root, site pDel,
			       gsitef * get_next_neigh,
			       site_n * site_numm_neigh) {
  int i, j, k, l, ind;
  int iPrev, jPrev;
  int j1, j2;
  int index1, index2;
  int newNeigh;
  int n;
  int outOfFlat;
  simplex * s = NULL;
  simplex * newS;
  simplex * t;
  simplex * setRoot;
  simplex * rootNew;
  simplex ** sVec;
  int nNeigh=0;
  site neigh[20]; /* Neighbours to pDel */
  site pt[3];
  site pt1;
  site pt2;
  site tmp;
  Vertex * v;
  gsitef * get_site_tmp;
  site_n * site_tmp;

  // Find: neighbours to pDel
//   while (v) { /* NBNB SKARE: Lite effektivt */
//     if (v->pt == pDel)
//       break;
//     v = v->next;
//   }

  v = findVertex(pDel);

  if (!v->nSimp)
    assert(v->nSimp);

  if (gDebugPrint >= 2) {
    v->print();
    printTriangles(v);
    printf("--------------------\n");
    printAll(*root);
  }

  pt1 = NULL;
  pt2 = NULL;
  j1 = j2 = -1;
  iPrev = -1;
  for (i=0;i<v->nSimp;i++) {
    s = v->simp[i];
    for (j=0;j<cdim;j++) {
      ind = site_num(s->neigh[j].vert);
      if (ind != site_num(pDel) && s->neigh[j].vert != infinity) {
	/* Add s->neigh[j].vert */
	for (k=0;k<nNeigh;k++) {
	  if (ind<=site_num(neigh[k]))
	    break;
	}
	newNeigh = 0;
	if (k == nNeigh) {
	  newNeigh = 1;
	  neigh[k] = s->neigh[j].vert;
	  nNeigh++;
	}
	else if (ind < site_num(neigh[k])) {
	  newNeigh = 1;
	  for (l=nNeigh;l>k;l--)
	    neigh[l]= neigh[l-1];
	  neigh[k] = s->neigh[j].vert;
	  nNeigh++;
	}
	if (iPrev == i && j1 < 0) {
	  pt1 = s->neigh[jPrev].vert;
	  pt2 = s->neigh[j].vert;
	}
	iPrev = i;
	jPrev = j;
      }
    }
  }

  if (!(pt1 && pt2))
    assert(pt1 && pt2);

  if (nNeigh > gnMaxNeigh) {
    printf("WARNING: # neighbours = %d\n", nNeigh);
    gnMaxNeigh = nNeigh+5;
    gDelNeigh = (site *) realloc(gDelNeigh, gnMaxNeigh*sizeof(site));
  }

  gnDelNeigh = nNeigh;
  for (i=0;i<nNeigh;i++) {
    if (gDebugPrint >= 2) 
      printf("%6.4f %6.4f\n", neigh[i][0], neigh[i][1]);

    gDelNeigh[i] = neigh[i];
  }

  // Mark/Remove: all simplices with pDel
  for (i=0;i<v->nSimp;i++) {
    s = v->simp[i];
    s->removed = 1;
  }

  if (nNeigh >= 3) {
    // Make new hull of the neighbours of v

    site_tmp = site_num;
    get_site_tmp = get_site;
    site_num = site_numm_neigh;
    get_site = get_next_neigh;
    gLocalSimplexStructure = 1;

    if (gDebugPrint >= 2)
      printf("Build up rootNew structure\n");

    rootNew = NULL;
    n = 0;
    outOfFlat = 0;
    do {
      if (!outOfFlat) {
	if (rootNew) {
	  freeRootNew(rootNew);
	}
	rootNew = NULL; // Memeory leakage?
	NEWL(simplex, rootNew);
	
	set_ch_root(rootNew);
	
	p = infinity;
	copy_simp(s,rootNew);
	rootNew->peak.vert = p;
	rootNew->peak.simp = s;
	s->peak.simp = rootNew;
	cdim = 0;
	setNumDel(0);	//	s_numDel = 0;

	if (n>0) {
	  tmp = gDelNeigh[n-1];
	  gDelNeigh[n-1] = gDelNeigh[nNeigh-1];
	  gDelNeigh[nNeigh-1] = tmp;
	}
      }

      outOfFlat = buildhull(rootNew); // NBNB p hentes fra neigh...
      n++;
    } while (!outOfFlat && n<1); // changed from n<4
    
    if (!outOfFlat) {
      printf("Warning(removePointFromConvexHull): Not out of flat\n");
      v->print();
      v->printNeigh();
      gAccept = 0;
    }
    else if (gDebugCheck) {
      gAccept = checkNewRoot(rootNew);
    }
    

    if (gDebugPrint >= 2)
      printf("-----------\n");

    gLocalSimplexStructure = 0;
    site_num = site_tmp;
    get_site = get_site_tmp;
    cdim = 3;
    set_ch_root(*root);

    if (gDebugPrint >= 2) {
      writeToSFile(rootNew, "rootNew");
      writeToSFile(*root, "root");
    }
    if (gAccept)
      merge(root, rootNew, v);
    

    if (gDebugPrint >= 2)
      printAll(rootNew);
  }
  else {
    i = 0;
    do {
      s = v->simp[i++];
    } while (s->infinity);

    copy_simp(newS, s);
    for (i=0;i<cdim;i++) {
      if (s->neigh[i].vert == pDel)
	break;
    }

    NULLIFY(basis_s,newS->normal);
    NULLIFY(basis_s,newS->neigh[0].basis);

    ind = i;
    newS->neigh[ind].vert = infinity;

    for (i=0;i<cdim;i++) {
      t = s->neigh[i].simp;
      if (i != ind) {
	for (j=0;j<cdim;j++) {
	  if (t->neigh[j].vert == pDel)
	    break;
	}
	t = t->neigh[j].simp;
      }
      shareEdge(newS, t, pt, &index1, &index2);
      t->neigh[index2-1].simp = newS;
      newS->neigh[index1-1].simp = t;
    }

    newS->infinity = 1;
    setRoot = findFirst(newS); // NBNB Memory leakage

    copy_simp(s, setRoot);// Kopiere setRoot inn i s (den nye roten)
    s->peak.vert = infinity;
    s->peak.simp = setRoot;
    setRoot->peak.simp = s;

    (*root) = s;
    set_ch_root(*root);

    addTriangle(newS);
    n = v->nSimp;
    sVec = (simplex **) malloc(n * sizeof(simplex *));
    for (i=0;i<n;i++)
      sVec[i] = v->simp[i];
    for (i=0;i<n;i++) {
      removeTriangle(sVec[i]);
      freeSimp(sVec[i]);
    }
    free(sVec);

    // remove simplex consisting of pDel and its two neighbours, update
    // neighbour relations
    //    assert(0);
  }

  return;
}

void shareEdge(simplex * s1, simplex * s2, point * a, int * indexI, int * indexJ) {
  int i, j, k;
  int n = 0;
  int * I = (int *) calloc(cdim, sizeof(int));
  int * J = (int *) calloc(cdim, sizeof(int));

  (*indexI) = 0;
  (*indexJ) = 0;

  for (i=0;i<cdim;i++) {
    for (j=0;j<cdim;j++) {
      if (s1->neigh[i].vert == s2->neigh[j].vert) {
	I[i] = 1;
	J[j] = 1;
	a[n++] = s2->neigh[j].vert;
	break;
      }
    }
  }

  if (n>2)
    assert(n <= 2);

  if (n ==2 ) {
    for (k=0;k<cdim;k++) {
      if (!I[k])
	(*indexI) = k;
      if (!J[k])
	(*indexJ) = k;
    }
    a[n] = s2->neigh[(*indexJ)].vert;

    (*indexI)++;
    (*indexJ)++;
  }

  free(I);
  free(J);

  return;
}

// update simplex structure after the deletion of a point (pDel)
void merge(simplex ** root, simplex * rootNew, Vertex * v) {
  // go through rootnew and mark those simplices that shall not be marked
  // with removed = 1
  int i, k;
  int n;
  int infEdge;
  int maxPost = 10;
  int maxInf = 10;
  int maxNorm = 10;
  int * indexT = (int *) calloc(3, sizeof(int));
  int * index1 = (int *) calloc(3, sizeof(int));
  int * index2 = (int *) calloc(3, sizeof(int));
  int nNeighSimp;
  int insertS;
  int sign1, sign2;
  int iInf, iNorm;
  int nInf, nNorm, nAddPost, nRemPost;
  int ccdim = 3;
  point * pt[3];
  point a;
  simplex * start;
  simplex * s;
  simplex ** sVec;
  simplex ** sInf;
  simplex ** sNorm;
  simplex ** sPost;
  simplex * w;
  simplex * t[3];
  simplex * sNeigh[3];
  simplex * setRoot = NULL;
  static simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  for (i=0;i<3;i++) {
    pt[i] = (point *) malloc(ccdim*sizeof(point));
  }

  counter--;
  start = findFirst(rootNew);
  counter--;

  assert(!start->peak.vert); // NBNB: vert in rootNew should point to gPoints...

  if (!st) st = (simplex **)malloc((ss+MAXDIM+1)*sizeof(simplex*));
  push(start);
  start->visit = counter;

  sInf = (simplex **) malloc(maxInf*sizeof(simplex*));
  sNorm = (simplex **) malloc(maxNorm*sizeof(simplex*));
  sPost = (simplex **) malloc(maxNorm*sizeof(simplex*));

  nInf = 0;
  nNorm = 0;
  nAddPost = nRemPost = 0;
  while (tms) {
    if (tms>ss) 
      assert(st=(simplex**)realloc(st,
				   ((ss+=ss)+MAXDIM+1)*sizeof(simplex*)));
    if (nInf>maxInf-1) {
      maxInf = nInf + 5;
      assert(sInf=(simplex**)realloc(sInf, maxInf*sizeof(simplex*)));
    }
    if (nNorm>maxNorm-1) {
      maxNorm = nNorm + 5;
      assert(sNorm=(simplex**)realloc(sNorm, maxNorm*sizeof(simplex*)));
    }
    pop(s);

    if (s->infinity)
      sInf[nInf++] = s;
    else
      sNorm[nNorm++] = s;

    //    printTriangle(s);

    for (i=0;i<ccdim;i++) {
      w = s->neigh[i].simp;
      if (w->visit != counter) {
	w->visit = counter;
	push(w);
      }
    }
  }

  counter--;

  iNorm = nNorm;
  iInf = nInf;

  while (1) {
    if (iNorm > 0)
      s = sNorm[--iNorm];
    else if (nAddPost -nRemPost> 0) { // FIFO Queue
      s = sPost[nRemPost++];
    }
    else if (iInf > 0)
      s = sInf[--iInf];
    else
      break;

    s->visit = counter;
    if (gDebugPrint >= 2) {
      printTriangle(s);
    }
    //    if (!s->infinity) {
    nNeighSimp = 0;
    for (i=0;i<v->nSimp;i++) {
      shareEdge(v->simp[i], s, pt[nNeighSimp], &index1[nNeighSimp], &index2[nNeighSimp]);
      if (index1[nNeighSimp]) {
	assert(nNeighSimp <= 2);
	sNeigh[nNeighSimp++] = v->simp[i];
	if (nNeighSimp == 3)
	  break;
      }
    }
      
    if (index1[0]) { // s deler kant med minst en av v->simp[i]

      for (i=0;i<nNeighSimp;i++)
	t[i] = sNeigh[i]->neigh[index1[i]-1].simp;

      //	if (!t[0]->infinity) {

      for (i=0;i<nNeighSimp;i++) {
	for (k=0;k<ccdim;k++)
	  if (t[i]->neigh[k].vert != pt[i][0] && t[i]->neigh[k].vert != pt[i][1])
	    break;

	assert(i<ccdim);

	indexT[i] = k;
      }

      a = t[0]->neigh[indexT[0]].vert;

      infEdge = 1;
      for (i=0;i<nNeighSimp;i++) {
	infEdge = (pt[i][0] == infinity) || (pt[i][1] == infinity);
	if (!infEdge)
	  break;
      }

      insertS = 1;
      if (s->infinity) {
	if (!infEdge) {
	  // s ^ t[0] in the interior of the convex hull, s not inserted
	  for (k=0;k<ccdim;k++)
	    if (s->neigh[k].vert == infinity)
	      break;
	  w = s->neigh[k].simp;
	  // check if w is removed
	  if (!w->removed)
	    insertS = 0;
	}
	else if (infEdge) {
	  // check if s and t[] are equal
	  for (i=0;i<nNeighSimp;i++) {
	    if (s->neigh[index2[i]-1].vert == t[i]->neigh[indexT[i]].vert) {
	      insertS = 0;
	      break;
	    }
	  }
	  if (insertS) {
	    for (k=0;k<ccdim;k++)
	      if (s->neigh[k].vert == infinity)
		break;
	    w = s->neigh[k].simp;
	    // check if w is removed
	    if (w->removed)
	      insertS = 0;
	  }
	}
      }
      else {
	if (a != infinity) {
	  // check if s og t ligger paa samme side
	  sign1 = Areasign(pt[0][0], pt[0][1], pt[0][2]);
	  sign2 = Areasign(pt[0][0], pt[0][1], a);
	  insertS = (sign1 != sign2);
	}
      }

    }
    else {
      insertS = 0;
      if (s->infinity) {
	// Find neighbour in rootnew
	for (i=0;i<cdim;i++)
	  if (s->neigh[i].vert == infinity) {
	    break;
	  }
	if (!s->neigh[i].simp->removed)
	  insertS = 1;
      }
      else {
	for(i=0;i<cdim;i++) {
	  w = s->neigh[i].simp;
	  if (w->visit == counter) {
	    insertS = !w->removed;
	    break;
	  }
	}
	if (i == cdim) {
	  s->visit++;
	  sPost[nAddPost++] = s; // Utsette avgjoerelsen om aa insette den
	  if (nAddPost>maxPost-1) {
	    maxPost = nAddPost + 5;
	    assert(sPost=(simplex**)realloc(sPost, maxPost*sizeof(simplex*)));
	  }
	}
      }
    }
    if (insertS) {
      s->removed = 0;
      // Insert s for t[0] og t[1]
      for (i=0;i<nNeighSimp;i++) {
	t[i]->neigh[indexT[i]].simp = s;
	s->neigh[index2[i]-1].simp = t[i];
	// Should note visit simplices in root-structure
	t[i]->visit = counter;
      }
      if (gDebugPrint >= 2) {
	printf(" ***** ");
	printTriangle(s);
      }
      addTriangle(s);
      if (!setRoot) {
	setRoot = s;
      }
    
      //    }
    
    }
    else
      s->removed = 1;
  }

  for (i=0;i<nNorm;i++) {
    if (sNorm[i]->removed)
      freeSimp(sNorm[i]);
  }
  for (iInf=0;iInf<nInf;iInf++) {
    if (sInf[iInf]->removed)
      freeSimp(sInf[iInf]);
  }


  setRoot = findFirst(setRoot);

  copy_simp(s, setRoot);// Kopiere setRoot inn i s (den nye roten)
  s->peak.vert = infinity;
  s->peak.simp = setRoot;
  setRoot->peak.simp = s;

  (*root) = s; // NBNB Memory leakage
  set_ch_root(s);

  n = v->nSimp;
  sVec = (simplex **) malloc(n * sizeof(simplex *));
  for (i=0;i<n;i++)
    sVec[i] = v->simp[i];
  for (i=0;i<n;i++) {
    removeTriangle(sVec[i]);
    freeSimp(sVec[i]);
  }

  free(sPost);
  free(sNorm);
  free(sInf);
  free(sVec);
  free(index1);
  free(index2);
  free(indexT);
  for (i=0;i<3;i++) {
    free(pt[i]);
  }


  return;
}

void freeSimp(simplex * s) {

  return;
}
// void freeSimp(simplex * s) {
//  mod_refs(dec,s);
//  if (s->normal) {
//    FREEL(basis_s,s->normal);
//  }
//  FREEL(simplex, s);
// }

void freeRootNew(simplex * start) {
  int i;
  int n = 0;
  simplex * w;
  simplex * t = NULL;
  simplex * sVec[100];

  static simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  counter--;
  initStackMemory();

  start->visit = counter;
  push(start);

  start->peak.simp->visit = counter;
  push(start->peak.simp);

  for (i=0;i<cdim;i++) {
    w = start->neigh[i].simp;
    if (w->visit != counter) {
      w->visit = counter;
      push(w);
    }
  }

  while (tms) {
    stackMemory();
    pop(t);

    sVec[n++] = t;

    for (i=0;i<cdim;i++) {
      w = t->neigh[i].simp;
      if (w->visit != counter) {
	w->visit = counter;
	push(w);
      }
    }    
  }

  for (i=0;i<n;i++)
    freeSimp(sVec[i]);
}

int seesOwn(site p, simplex *s) {
  int i;
  int index;
  int n;
  int sign1, sign2;
  //  int isSeen;
  int isSeenOwn;
  double d;
  double sum = 0;
  simplex * t;
  point pts[3];

  //  isSeen = sees(p,s);

  if (!s->peak.vert) {
    if (s->infinity) {
      // Check if s is seen by p by see if new triangle are on opposite
      // side of the triangle s->neigh[x].simp (!= infinity)
      n = 0;
      for (i=0;i<cdim;i++) {
	if (s->neigh[i].vert != infinity)
	  pts[n++] = s->neigh[i].vert;
	else
	  index = i;
      }
      pts[cdim-1] = p;
      t = s->neigh[index].simp;
      for (i=0;i<cdim;i++)
	if (t->neigh[i].vert != pts[0] && t->neigh[i].vert != pts[1])
	  break;

      sign1 = Areasign(pts[0], pts[1], pts[2]);
      sign2 = Areasign(pts[0], pts[1], t->neigh[i].vert);

      if (sign1 == 0 || sign2 == 0) {
	isSeenOwn = -1; // Error: Should abort building a hull
	return isSeenOwn;
      }	
      else
	isSeenOwn = (sign1 != sign2);
    }
    else {
      for (i=0;i<pdim;i++) {
	d = s->centerpoint[i] - p[i];
	sum += d*d;
      }
      isSeenOwn = (sum < s->radius);
    }
  }

  return isSeenOwn;
}
