/*
vorSim2D: C++-routines for simulation from a coloured Voronoi
tessellation model.

Version 1.0

Copyright (C)  2001 ivind Skare

  This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU
  General Public License as published by the Free Software Foundation; either ve
rsion 2 of the
  License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT AN
Y WARRANTY; without
  even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPO
SE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along with t
his program; if not,
  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Bost
on, MA 02111-1307,
  USA.


  The author's contact information:

 ivind Skare                Office tel.: (+47) 22 85 26 56
 Norwegian Computing Center         Fax:         (+47) 22 69 76 60 
 P.O.Box 114 Blindern               E-mail:      Oivind.Skare@nr.no 
 NO-0314 Oslo, Norway
*/
/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <errno.h>
#include <assert.h>

#include "model_scan.h"


#include "hull.h"
#include "hullOwn.h"

#include "input.h"

#include "random.h"
#include "model.h"
#include "simulation.h"

#define NCOMMANDS 19
#define FILES 0
#define MODELPARAMETERS 1
#define DIMENSION 2
#define SEED 3
#define NCOLOUR 4
#define NITER 5
#define ACTIONPROBS 6
#define DEBUGPRINT 7
#define DEBUGCHECK 8
#define MULT_UP 9
#define INTEGER 10
#define DATA 11
#define TRUEIMAGE 12
#define ALGPARAMETERS 13
#define ESTIMATEMARGINAL 14
#define INITIALIZE 15
#define LOGINTERVAL 16
#define LOGBINAER 17
#define READMARKEDPOINTS 18

int gInteger;
int gDebugPrint;
int gDebugCheck;

static void readInput(char * modelFileName, Input * input);

FILE * DFILE = stderr;
int main(int argc, char **argv)
{
  Input * input = new Input;
  Model * model;
  Simulation * sim;
  if (argc<=1 || argc >=3) {
    printf("Usage: %s <model-file>\n", argv[0]);
    exit(-1);
  }
  
  readInput(argv[1], input);

  gDebugPrint = input->debugPrint;
  gDebugCheck = input->debugCheck;

  model = new Model(input->beta, input->theta, input->sigma,
		    input->dim, input->nColour, input->nTheta,
		    input->secondOrderNeighbours,
		    input->areaInteraction, input->areaSize,
		    input->areaPenalty);

  //    if (input->seedFileName == NULL)
  sim = new Simulation(model, input);

  sim->simulation();

  sim->writeToFile();

  deleteVertices();

  delete model;
  delete sim;
  delete input;
}

static void readInput(char * modelFileName, Input * input) {
  int i, iComm, j;
  int dataRead;
  int trueImageRead;
  double sum;
  char ** endPtr = new char *;
  char **commands;
  struct COMMAND *sComm, *sCommands;
   
  commands = (char **) calloc(NCOMMANDS, sizeof(char *));
  for (i=0;i<NCOMMANDS;i++)
    commands[i] =  (char *) calloc(50, sizeof(char));
  strcpy(commands[NITER], "NITER");
  strcpy(commands[FILES], "FILES");
  strcpy(commands[SEED], "SEED");
  strcpy(commands[MODELPARAMETERS], "MODELPARAMETERS");
  strcpy(commands[ALGPARAMETERS], "ALGPARAMETERS");
  strcpy(commands[NCOLOUR], "NCOLOUR");
  strcpy(commands[DIMENSION], "DIMENSION");
  strcpy(commands[ACTIONPROBS], "ACTIONPROBS");
  strcpy(commands[DEBUGPRINT], "DEBUGPRINT");
  strcpy(commands[DEBUGCHECK], "DEBUGCHECK");
  strcpy(commands[MULT_UP], "MULT_UP");
  strcpy(commands[INTEGER], "INTEGER");
  strcpy(commands[DATA], "DATA");
  strcpy(commands[TRUEIMAGE], "TRUEIMAGE");
  strcpy(commands[ESTIMATEMARGINAL], "ESTIMATEMARGINAL");
  strcpy(commands[INITIALIZE], "INITIALIZE");
  strcpy(commands[LOGINTERVAL], "LOGINTERVAL");
  strcpy(commands[LOGBINAER], "LOGBINAER");
  strcpy(commands[READMARKEDPOINTS], "READMARKEDPOINTS");
   
  sCommands = spReadModelFile(modelFileName,
                              commands, NCOMMANDS);
   
  for (iComm=0;iComm<NCOMMANDS;iComm++) {
    sComm = &sCommands[iComm];

    switch (iComm) {
    case FILES:
      if (!sComm->bRead) {
        printf("Command FILES not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
        printf("Number of arguments for the FILES command != 1\n");
	exit(-1);
      }
      input->fileName = (char *) calloc(strlen(sComm->cppArgs[0])+1, sizeof(char));
      strcpy(input->fileName, sComm->cppArgs[0]);
       
      break;
    case READMARKEDPOINTS:
      if (!sComm->bRead) {
        input->readMarkedPoints = 0;
      }
      else if (sComm->nArgs != 1) {
        printf("Number of arguments for the READMARKEDPOINTS command != 1\n");
	exit(-1);
      }
      else {
	input->readMarkedPoints = 1;
	input->markedPointsFileName =
	  (char *) calloc(strlen(sComm->cppArgs[0])+1, sizeof(char));
	strcpy(input->markedPointsFileName, sComm->cppArgs[0]);
      }

      break;
    case SEED:
      if (!sComm->bRead) {
        printf("Command SEED not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
        printf("Number of arguments for the SEED command != 2\n");
 	exit(-1);
      }
      input->seed = new unsigned int;
      *(input->seed) = strtoul(sComm->cppArgs[0], endPtr, 10);
      if (strcmp(sComm->cppArgs[0], *endPtr) != 0) {
	input->seedFileName = NULL;
      }
      else {
	input->seedFileName = (char *) calloc(strlen(sComm->cppArgs[0])+1, sizeof(char));
	strcpy(input->seedFileName, sComm->cppArgs[0]);
      }
      break;
    case MODELPARAMETERS:
      if (!sComm->bRead) {
        printf("Command MODELPARAMETERS not specified\n");
	exit(-1);
      }      
      if (sComm->nArgs == 3) { // && strcmp(sComm->cppArgs[0], "area") == 0) {
	input->secondOrderNeighbours = 0; // dummy ??
	input->nTheta = 0;
	input->areaInteraction = 1;
	input->beta = atof(sComm->cppArgs[0]);
	input->areaSize = atof(sComm->cppArgs[1]);
	input->areaPenalty = atof(sComm->cppArgs[2]);
      }
      else {
	input->areaInteraction = 0;
	input->areaSize = input->areaPenalty = 0.0;
	if (sComm->nArgs == 2) {
	  input->secondOrderNeighbours = 0;
	  input->nTheta = 1;
	}
	else if (sComm->nArgs == 5) {
	  input->secondOrderNeighbours = 1;
	  input->nTheta = 4;
	}
	else {
	  printf("Number of arguments for the MODELPARAMETERS command != 2,3,5\n");
	  exit(-1);
	}
	input->beta = atof(sComm->cppArgs[0]);
	input->theta = (double *) malloc(input->nTheta*sizeof(double));
	for (i=0;i<input->nTheta;i++)
	  input->theta[i] = atof(sComm->cppArgs[i+1]);
      }
      break;
    case ALGPARAMETERS:
      if (!sComm->bRead) {
        printf("Command ALGPARAMETERS not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the ALGPARAMETERS command != 1\n");
	exit(-1);
      }
      input->sigmaMove = atof(sComm->cppArgs[0]);
      break;

    case DATA:
      if (!sComm->bRead) {
	input->nDataX = input->nDataY = 0;
	input->dataFileName = NULL;
	input->sigma = 0.0;
	dataRead = 0;
      }
      else {
	dataRead = 1;
	if (sComm->nArgs != 4) {
	  printf("Number of arguments for the DATA command != 5\n");
	  exit(-1);
	}
	input->nDataX = atoi(sComm->cppArgs[0]);
	input->nDataY = atoi(sComm->cppArgs[1]);
	input->dataFileName = (char *) calloc(strlen(sComm->cppArgs[2])+1, sizeof(char));
	strcpy(input->dataFileName, sComm->cppArgs[2]);
	input->sigma = atof(sComm->cppArgs[3]);
      }
      break;

    case TRUEIMAGE:
      if (!sComm->bRead) {
	trueImageRead = 0;
	input->trueFileName = NULL;
      }
      else {
	trueImageRead = 1;
	if (sComm->nArgs != 3) {
	  printf("Number of arguments for the TRUEIMAGE command != 3\n");
	  exit(-1);
	}
	input->trueFileName = (char *) calloc(strlen(sComm->cppArgs[2])+1, sizeof(char));
	strcpy(input->trueFileName, sComm->cppArgs[2]);
      }
      break;

    case INITIALIZE:
      if (!sComm->bRead) {
	input->init = 0;
      }
      else {
	input->init = 1;
	if (sComm->nArgs != 2) {
	  printf("Number of arguments for the TRUEIMAGE command != 2\n");
	  exit(-1);
	}
       input->initNPoints = atoi(sComm->cppArgs[0]);
       if (strcmp(sComm->cppArgs[1], "random") == 0)
	 input->initRandom = 1;
       else
	 input->initRandom = 0;
      }
      break;

    case MULT_UP:
      if (!sComm->bRead) {
        printf("Command MULT_UP not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the MULT_UP command != 1\n");
	exit(-1);
      }
      input->mult_up = atof(sComm->cppArgs[0]);
      break;

     case ACTIONPROBS:
      if (!sComm->bRead) {
        printf("Command ACTIONPROBS not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 3) {
	printf("Number of arguments for the ACTIONPROBS command != 3\n");
	exit(-1);
      }
      sum = input->probs[ADD] = atof(sComm->cppArgs[0]);
      sum += input->probs[MOVE] = atof(sComm->cppArgs[1]);
      sum += input->probs[REMOVE] = atof(sComm->cppArgs[2]);
      if (fabs(sum-1.0)>0.00000000001) {
	printf("Error: Probabilities do not sum to one.\n");
	exit(-1);
      }
      break;
    case NITER:
      if (!sComm->bRead) {
        printf("Command NITER not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the NITER command != 1\n");
	exit(-1);
      }
      input->nIter = atoi(sComm->cppArgs[0]);
      break;

    case INTEGER:
      if (!sComm->bRead) {
        gInteger = 0;
      }
      else
        gInteger = 1;

      if (sComm->nArgs != 0) {
	printf("Number of arguments for the INTEGER command != 0\n");
	exit(-1);
      }
      break;


    case LOGBINAER:
      if (!sComm->bRead) {
        input->logBinaer = 0;
      }
      else
        input->logBinaer = 1;

      if (sComm->nArgs != 0) {
	printf("Number of arguments for the LOGBINAER command != 0\n");
	exit(-1);
      }
      break;


    case ESTIMATEMARGINAL:
      if (!sComm->bRead) {
        input->estimateMarginalPosterior = 0;
      }
      else
        input->estimateMarginalPosterior = 1;

      if (sComm->nArgs != 0) {
	printf("Number of arguments for the ESTIMATEMARGINAL command != 0\n");
	exit(-1);
      }
      break;


    case DEBUGPRINT:
      if (!sComm->bRead) {
        printf("Command DEBUGPRINT not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the DEBUGPRINT command != 1\n");
	exit(-1);
      }
      input->debugPrint = atoi(sComm->cppArgs[0]);
      break;


    case DEBUGCHECK:
      if (!sComm->bRead) {
        printf("Command DEBUGCHECK not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the DEBUGCHECK command != 1\n");
	exit(-1);
      }
      input->debugCheck = atoi(sComm->cppArgs[0]);
      break;

    case NCOLOUR:
      if (!sComm->bRead) {
        printf("Command NCOLOUR not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the NCOLOUR command != 1\n");
	exit(-1);
      }
      input->nColour = atoi(sComm->cppArgs[0]);
      break;

    case LOGINTERVAL:
      if (!sComm->bRead) {
        input->logInterval = 0;
      }
      else if (sComm->nArgs != 1) {
	printf("Number of arguments for the NCOLOUR command != 1\n");
	exit(-1);
      }
      else
	input->logInterval = atoi(sComm->cppArgs[0]);

      break;

    case DIMENSION:
      if (!sComm->bRead) {
        printf("Command DIMENSION not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 1) {
	printf("Number of arguments for the DIMENSION command != 1\n");
	exit(-1);
      }
      input->dim = atoi(sComm->cppArgs[0]);
      break;
    }
  }

  if (trueImageRead != dataRead) {
    printf("Both true image and data image should be read!\n");
    exit(-1);
  }

  for (i=0;i<NCOMMANDS;i++)
    free(commands[i]);
  free(commands);

  

  for (iComm=0;iComm<NCOMMANDS;iComm++) {
    for (j=0;j<sCommands[iComm].nArgs;j++)
      free(sCommands[iComm].cppArgs[j]);
    free(sCommands[iComm].cppArgs);
  }
	
  free(sCommands);

  delete endPtr;

  return;
}
