/*
vorSim2D: C++-routines for simulation from a coloured Voronoi
tessellation model.

Version 1.0

Copyright (C)  2001 ivind Skare

  This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU
  General Public License as published by the Free Software Foundation; either ve
rsion 2 of the
  License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT AN
Y WARRANTY; without
  even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPO
SE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along with t
his program; if not,
  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Bost
on, MA 02111-1307,
  USA.


  The author's contact information:

 ivind Skare                Office tel.: (+47) 22 85 26 56
 Norwegian Computing Center         Fax:         (+47) 22 69 76 60 
 P.O.Box 114 Blindern               E-mail:      Oivind.Skare@nr.no 
 NO-0314 Oslo, Norway
*/
/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <assert.h>
//#include <iostream.h>
#define min(A,B) ((A) < (B) ? (A) : (B))
#define max(A,B) ((A) > (B) ? (A) : (B))

//extern  "C"
//{
#include "hull.h"
#include "hullOwn.h"
#include "vertex.h"
//}

#include "random.h"
#include "model.h"
#include "input.h"
#include "simulation.h"

#define  G_MODULUS  256*256*256*128 
#define  G_INVMOD  ( (double) 1 / ((double) G_MODULUS )) / ( (double) 2 ) 


#define PROGRAMMERINGGJENSTAAR 1

int gIt;
int gAccept;
int gLocalSimplexStructure;
point gPoints;
Vertex ** gVertex;
int gMaxPos=0;
static point gLastAddedPoint;
point * gDelNeigh;
int gnDelNeigh;
int gnMaxNeigh;
int gnEdges;
int gnDirEdges;
Edge * gFirstEdge;
Vertex * gFirstVertex;


int n40[1600];
int n20[400];
int n10[100];

simplex * gSimplex40[1600][10];
simplex * gSimplex20[400][40];
simplex * gSimplex10[100][160];
//simplex ** gSimplex5[25];

static short vd = DELAUNAY; /* Indicator for Delaunay triangulations */
static int gDim;
static int gnPoints;
static long s_num;
static long s_numDel;

void setNumDel(int n) {
  s_numDel = 0;
}

long site_numm(site pt) {

	long pos;

	if (vd && pt==infinity) return -1;
	if (!pt) return -2;
	pos = (pt - gPoints)/gDim;
	if (pos >= 0) /* && pos < gnPoints)*/
	  return pos;

	return -3;
}

long site_numm_neigh(site pt) {

	long pos;

	if (vd && pt==infinity) return -1;
	if (!pt) return -2;
	pos = (pt - gPoints)/gDim;
	if (pos >= 0) /* && pos < gnPoints)*/
	  return pos;

	return -3;
}
  /*static site (*get_site_n)(long);*/

static int sPos = 0;

site read_next_site(long iSite){
  /* Needs following global variables:
     - gnPoints
     - gPoints
     - gDim
     - mins/maxs (ok)*/
  point pt;
  int i=0;
  double tmp;

  if (iSite == gnPoints)
    return NULL;
  else if (iSite == (gnPoints - 1) && gLastAddedPoint)
    pt = gLastAddedPoint;
  else {
    while (gPoints[sPos*gDim] == REMOVED_MARK)
      sPos++;
    pt = &gPoints[sPos*gDim];
    sPos++;
  }
  //double * pLocal = (double *) calloc(gDim, sizeof(double));
  /* site = double * */

  assert(pt[0] != REMOVED_MARK);

  for (i=0; i<gDim; i++) {
    tmp = pt[i];
    mins[i] = (mins[i]<tmp) ? mins[i] : tmp;
    maxs[i] = (maxs[i]>tmp) ? maxs[i] : tmp;
  }

  return pt;
}
site get_next_site(void) {
  //	static long s_num = 0;
	site nextSite = read_next_site(s_num++);
	if (nextSite == NULL)
	  s_num--;

	return nextSite;
}



site read_next_neigh(long iSite){
  /* Needs following global variables:
     - gnDelNeigh
     - gDelNeigh
     - gDim
     */

  if (iSite == gnDelNeigh)
    return NULL;

  return gDelNeigh[iSite];
}
site get_next_neigh(void) {
  //	static long s_num = 0;
	site nextSite = read_next_neigh(s_numDel++);

	return nextSite;
}
// void make_output(simplex *root, void *(*visit_gen)(simplex*, visit_func* visit),
// 		visit_func* visit,
// 		out_func* out_funcp,
// 		FILE *F){

// 	out_funcp(0,0,F,-1);
// 	visit(0, out_funcp);
// 	visit_gen(root, visit); /*SKARE: Compilator warning !! */
// 	out_funcp(0,0,F,1);
// 	fclose(F);
// }


//} // extern C finished

//void findDirichletEdges();
//void addDirichletEdge(Vertex *v, int i1, int i2);
double distPoint(point pt1, point pt2);
int inTri2D(point * Tp, point pt);
//int Areasign(point a, point b, point c);
void sortDirichletEdges();
int truncateEdge(Edge * e, double * region);
int outside(point pt, double * region);


Simulation::Simulation(const Model * mod, Input * input)
{
  int i, j;
  int col;
  FILE * file;

  if (input->seedFileName == NULL)
    random = new RandomGenerator(*input->seed);
  else
    random = new RandomGenerator(input->seedFileName);

  logInterval = input->logInterval;
  logBinaer = input->logBinaer;

  model = mod;
  dim = mod->getDim();
  fileName = input->fileName;
  gnPoints = nPoints = 0;
  gLocalSimplexStructure = 0;
  nIter = input->nIter; 
  root = NULL;
  prob[ADD] = input->probs[ADD];
  prob[MOVE] = input->probs[MOVE];
  prob[REMOVE] = 1.0 -  prob[ADD] - prob[MOVE];

  init = input->init;
  initRandom = input->initRandom;
  initNPoints = input->initNPoints;

  estimateMarginalPosterior = input->estimateMarginalPosterior;

  sigmaMove = input->sigmaMove;

  mult_up = input->mult_up;

  region[XMIN] = 0.0;
  region[YMIN] = 0.0;
  region[XMAX] = 1.0;
  region[YMAX] = 1.0;

  for (i=0;i<NCLIQUES;i++)
    nClique[i] = 0;

  volRegion = 1.0;
  for (i=0;i<dim;i++)
    volRegion *= (region[2*i+1] - region[2*i]);

  multUpRegion(mult_up);

  gnMaxNeigh = 20;
  gDelNeigh = (site *) malloc(gnMaxNeigh * sizeof(site));

  readMarkedPoints = input->readMarkedPoints;
  markedPointsFileName = input->markedPointsFileName;

  simulateFromPosterior = (input->nDataX > 0);

  nDataX = input->nDataX;
  nDataY = input->nDataY;

  if (simulateFromPosterior) {
    file = fopen(input->dataFileName, "r");

    data = (Data **) malloc(nDataX * sizeof(Data *));
    for (i=0;i<nDataX;i++) {
      data[i] = (Data *) malloc(nDataY * sizeof(Data));
      for (j=0;j<nDataY;j++) {
	if (estimateMarginalPosterior)
	  data[i][j].sumProp = (int *) calloc(model->nColour, sizeof(int));
	fscanf(file, "%lf", &data[i][j].w);
	data[i][j].x = (point) malloc(2 * sizeof(double));
	data[i][j].x[0] = (((double) i) + 0.5 )* (region[XMAX]/((double) nDataX));
	data[i][j].x[1] = (((double) j) + 0.5 )* (region[YMAX]/((double) nDataY));
      }
    }
    fclose(file);


    file = fopen(input->trueFileName, "r");

    for (i=0;i<nDataX;i++) {
      for (j=0;j<nDataY;j++) {
	fscanf(file, "%d", &col);
	data[i][j].trueCol = col;	
      }
    }
    fclose(file);



  }


  return;
}

void Simulation::multUpRegion(double mult) {
  int i;

  mult_up = mult;

  for (i=0;i<4;i++)
    region[i] *= mult;

  for (i=0;i<dim;i++)
    volRegion *= (region[2*i+1] - region[2*i]);

  return;
}

Simulation::~Simulation() {
  int i, j;
  //  Vertex * v;

  
  if (root != NULL)
    free_hull_storage();

  deleteVertices();

  if (simulateFromPosterior) {
    for (i=0;i<nDataX;i++) {
      for (j=0;j<nDataY;j++) {
	free(data[i][j].x);
      }
      free(data[i]);
    }
    free(data);
  }

  free(gPoints);
  gPoints = NULL;
  gnPoints = 0;

  delete random;
}

void Simulation::simulation() {
  int i, j;
  int it;
  int n;
  int lastIt = 1;
  int sum;
  int initializedData = 0;
  int col;
  char * name;
  Vertex * v;
  FILE * logFile = NULL;
  FILE * logFile2 = NULL;
  FILE * logFile3 = NULL;
  //  FILE * ptsFile = NULL;

  float tmp[3];

  sumColDiff = 0;

  simplex_counter = 0;
  basis_s_counter = 0;

  if (logInterval == 0) {
    if (nIter>1000) {
      logInterval = nIter/1000;
    }
    else {
      logInterval = 1;
    }
  }

  name = (char *) malloc((strlen(fileName)+5)*sizeof(char));
  strcpy(name, fileName);
  logFile = efopen(strcat(name, ".log"), "w");  
  free(name);


  if (model->secondOrderNeighbours) {
    name = (char *) malloc((strlen(fileName)+6)*sizeof(char));
    strcpy(name, fileName);
    logFile2 = efopen(strcat(name, ".log2"), "w");  
    free(name);
  }

  if (estimateMarginalPosterior) {
    name = (char *) malloc((strlen(fileName)+6)*sizeof(char));
    strcpy(name, fileName);
    logFile3 = efopen(strcat(name, ".log3"), "w");  
    free(name);
  }

  //  name = (char *) malloc((strlen(fileName)+5)*sizeof(char));
  //  strcpy(name, fileName);
  //  ptsFile = efopen(strcat(name, ".tmp"), "w");  
  //  free(name);

  for (i=0;i<100;i++) {
    for (j=0;j<160;j++)
      gSimplex10[i][j] = NULL;
    n10[i] = 0;
  }

  for (i=0;i<400;i++) {
    for (j=0;j<40;j++)
    gSimplex20[i][j] = NULL;
    n20[i] = 0;
  }

  for (i=0;i<1600;i++) {
    for (j=0;j<10;j++)
      gSimplex40[i][j] = NULL;
    n40[i] = 0;
  }


  sumAlpha = 0.0;

  sumPotLikelihood = potLikelihood = 0.0;
  sumPixelDiff = pixelDiff = 0;

  sPos = 0;
  gDim = dim;

  point_size = dim * sizeof(double);

  gnEdges = gnDirEdges = 0;

  if (readMarkedPoints) {
    readMarkedPointsFile();

    return;
  }
      
    

  if (init) {
    initMH();
  }

  if (logBinaer) {
    n = nIter * 3/logInterval;
    fwrite(&n, sizeof(int), 1, logFile);
  }
  else {
    n = nIter * 6/logInterval;
    fprintf(logFile, "%d\n", n);
  }




  // ********** SIMULATION LOOP ************
  for (it=0; it<nIter; it++) {
    restart = 0;
    gIt = it;
    if (gDebugPrint >= 1)
      printf("Iteration %d\n", it);
    else if (gDebugPrint == 0)
      if (((it % 10000 ) == 0) && it>0)
	printf("Iteration %d\n", it);

    gAccept = 1;
    //choose action: ADD, REMOVE, MOVE
    drawAction();
    if (action == ADD) {
      addProposal();
    }
    else if (action == REMOVE) {
      removeProposal();
    }
    else { // MOVE
      moveProposal();
      action = MOVE;
    }

    if (restart)
      accept = 0;

    if (gDebugPrint >= 1) {
      for (i=0;i<model->nTheta;i++) {
	printf("%2d ", nClique[i]);
      }
      printf("\n");
    }
    if (gnPoints == 3 && !initializedData) {
      initializeDataInVertexStructure();
      initializedData = 1;
    }
    
    if (accept) {
      if (estimateMarginalPosterior && initializedData) {
	updateMarginalPosterior(it - lastIt);
	lastIt = it;
      }
    }
    
    if (model->secondOrderNeighbours && gnPoints >= 3 && gDebugCheck)
      checkCliques();

    if (root && gDebugCheck)
      checkDataInVertexStructure();

    if (nPoints >= 3 && gDebugCheck) {
      if (!check_ch_root(root)) {
	printf("Iteration %d: root != ch_root\n", it);
	restart = 1;
      }
      if (!checkAll(root))
	restart = 1;
      if (gDebugPrint >= 2)
	printAllNeigh();
    }

    if (restart) {
      resetVertices();

      printf("RESTART! Build up simplex structure\n");
      sPos = 0;
      gLastAddedPoint = NULL;
      buildConvexHull();
    }
    if ((it+1) % logInterval == 0) {
      if (model->secondOrderNeighbours) {
	for (i=0;i<NCLIQUES;i++)
	  fprintf(logFile2, "%d ", nClique[i]);
      
	fprintf(logFile2, "\n");
      }
      if (logBinaer) {
	tmp[0] = (float) nPoints;
	tmp[1] = (float) sumColDiff;
	tmp[2] = (float) sumPotLikelihood;

	fwrite(tmp, sizeof(float), 3, logFile);
      }
      else {
	fprintf(logFile, "%d %d %d %6.2f %d %6.2f\n", it, nPoints, sumColDiff,
		sumPotLikelihood, sumPixelDiff, sumAlpha/logInterval);
      }
      sumAlpha = 0.0;
      if (gDebugPrint >= 1) {
	printf("nPoints %d sumColDiff %d\n", nPoints, sumColDiff);

	printVertices();
	printAllNeigh();
      }
    }

    if (nPoints >=3 && !model->secondOrderNeighbours &&
	(gDebugCheck || !gAccept || restart || init)) {//Remove init later..
      sum = 0;
      v = gFirstVertex;
      while(v) {
	sum += v->colDiff;

	v = v->next;
      }
      sum /= 2;

      //      assert(sum == sumColDiff);
      if (sum != sumColDiff) {
	printf("%s %d: sum %d != sumColDiff %d (accept %d action %d)\n",
	       "Warning(simulation) at iter", it, sum, sumColDiff,
	       accept, action);
	printf("Reset sumColDiff\n");
	sumColDiff = sum;
      }

      v = gFirstVertex;
      while(v) {
	sum = 0;
	col = v->colour;
	for (i=0;i<v->nNeigh;i++) {
	  sum += (col != v->neigh[i]->colour);	    
	}
	if (v->colDiff != sum) {
	  printf("%s %d: sum %d != v->colDiff %d (accept %d action %d)\n",
		 "Warning(simulation) at iter", it, sum, v->colDiff,
		 accept, action);
	  printf("Reset v->colDiff\n");
	  v->colDiff = sum;
	}
	v = v->next;
      }
	
    }

    if ((((it+1) % 10000) == 0) && ((it+1) < nIter)) {
      resetVertices();
      if (gDebugPrint > 1) {
	printf("REBUILD: Deletes simplex structure\n");
	printf("Simplex counter %d\n", simplex_counter);
      }
      if (root != NULL)
	free_hull_storage();
      if (gDebugPrint > 1) {
	printf("REBUILD: Build up simplex structure\n");
      }
      simplex_counter = 0;
      basis_s_counter = 0;

      sPos = 0;
      gLastAddedPoint = NULL;
      buildConvexHull();

      if (gDebugPrint > 1) {
	printf("Simplex counter %d\n", simplex_counter);
      }

    }

  }

  if (estimateMarginalPosterior)
    writeMarginalPosterior(logFile3);

  //  v = gFirstVertex;
  //  while(v) {
  //    fprintf(ptsFile, "%.12g %.12g %d\n", v->pt[0], v->pt[1], v->colour);
    
  //    v = v->next;
  //  }
 

  //  fclose(ptsFile);

  if (gDebugPrint >= 2) {
    printf("\n\n --------- Final simplex structure --------\n");
    printVertices();
    printAllNeigh();
    printAll(root);
  }

  fclose(logFile);
  fclose(logFile2);
  if (estimateMarginalPosterior)
    fclose(logFile3);
  return;
}

void Simulation::initMH() {
  int i;
  Vertex * v;
  mPoint * x;

  for (i=0; i<initNPoints; i++) {
    //Draw a random number in R^{dim}    
    x = drawPoint();

    if (!initRandom)
      x->col = 0;

    v = addVoronoi(x);
  }

  buildConvexHull();

  v = gFirstVertex;
  sumColDiff = 0;
  while (v) {
    int col = v->colour;
    int sum = 0;
    for (i=0;i<v->nNeigh;i++) {
      sum += (col != v->neigh[i]->colour);
    }

    v->colDiff = (short) sum;

    sumColDiff += sum;

    v = v->next;
  }

  sumColDiff /= 2;

  return;
}


void Simulation::readMarkedPointsFile() {
  int col;
  Vertex * v;
  mPoint * x;
  FILE * file = fopen(markedPointsFileName, "r");

  x = (mPoint *) malloc(sizeof(mPoint));
  x->pt = (point) malloc(point_size);
  fscanf(file, "%lf %lf %d", &(x->pt[0]), &(x->pt[1]), &col);
  x->col = col;

  while (!feof(file)) {
    //Draw a random number in R^{dim}    

    v = addVoronoi(x);

    x = (mPoint *) malloc(sizeof(mPoint));
    x->pt = (point) malloc(point_size);

    fscanf(file, "%lf %lf %d", &(x->pt[0]), &(x->pt[1]), &col);
    x->col = col;
  }

  buildConvexHull();

  fclose(file);
  
  return;
}

// Assumes only 2 colours...
void Simulation::writeMarginalPosterior(FILE * file) {
  int i,j;
  int n  = data[0][0].sumProp[0] + data[0][0].sumProp[1];
  double d = 1.0/((double) n);


  for (i=0;i<nDataX;i++) {
    for (j=0;j<nDataY;j++) {
      fprintf(file, "%6.4f ", ((double) data[i][j].sumProp[0]) * d);
      assert(data[i][j].sumProp[0] + data[i][j].sumProp[1] == n);
    }
    fprintf(file, "\n");
  }

  return;
}


void Simulation::updateMarginalPosterior(int weight) {
  Vertex * v = gFirstVertex;

  while(v) {
    v->updateProp(weight);
    v = v->next;
  }

  return;
}

void Simulation::addProposal() {
  int resSum = 0;
  double pot;
  double alpha;
  Vertex * v;
  mPoint * x;

  //Draw a random number in R^{dim}    
  x = drawPoint();

  v = addVoronoi(x);
  if (gDebugPrint >= 1)
    printf("Proposal: new point (%6.2f %6.2f), with colour: %d\n",
	   x->pt[0], x->pt[1], x->col);
  if (nPoints==3 && !root) {
    buildConvexHull(); //buildConvexHull();
  }
  else if (nPoints > 3 && action == ADD) {
    gAccept = addPointToConvexHull(root);

    restart = (gAccept == -1);
    if (restart) {
      free(x->pt); // 1
      free(x);
      return;
    }
  }


  pixelDiff = 0;
  potLikelihood = 0.0;


  pot = acceptProb(v, &resSum);

  alpha = exp(-pot);

  accept = acceptance(alpha);

  if (accept) {
    removeDataFromVertices(v, 0); // Remove data from neighbours of v
    if (!model->secondOrderNeighbours) {
      sumColDiff += resSum;
    }
    updateColDiff(v);
    updateLogInfo();
  }
  else if (gAccept) { // No errors when adding a point to the Delaunay triang.
    
    removePointFromConvexHull(&root, v->pt, get_next_neigh,
 			      site_numm_neigh);

    if (gAccept)
      removeVoronoi(v->pt);
    else {
      removeDataFromVertices(v, 0); // Remove data from neighbours of v
      printf("Error(addProposal): Could not remove vertex\n");
      v->print();
      updateColDiff(v); // Added 05/07/99
      updateLogInfo();
    }
  }
  else {
    // The simplex structure was not updated, only have
    // to remove vertex from the vertex list
    removeVoronoi(v->pt);
  }

  free(x->pt);
  free(x);
}

void Simulation::removeProposal() {
  int resSum = 0;
  double pot;
  double alpha;
  Vertex * v;
  static Vertex * vCopy = new Vertex;
  point pt;

  pt = selectPoint();
  v = findVertex(pt);
  vCopy->nullify();
  copyVertex(v, vCopy);
  //      v = tmp;
  if (gDebugPrint >= 1)
    printf("Proposal: Delete point: (%6.2f %6.2f)\n", v->pt[0], v->pt[1]);

  assert (nPoints > 3);

  removePointFromConvexHull(&root, v->pt, get_next_neigh, site_numm_neigh);
  assert(((v->nNeigh == 0) && gAccept) || (v->nNeigh != 0) && !gAccept);

  pixelDiff = 0;
  potLikelihood = 0.0;


  pot = acceptProb(vCopy, &resSum);

  alpha = exp(-pot);

  accept = acceptance(alpha);

  if (accept) {
    if (!model->secondOrderNeighbours) {
      sumColDiff -= resSum;
    }

    removeVoronoi(v->pt);

    updateColDiff(vCopy);
    updateLogInfo();
  }
  else if (gAccept) {
    addVoronoi(v->pt); // Resets input structure: next site is v->pt
    gAccept = addPointToConvexHull(root);
    if (gAccept == -1)
      restart = 1; // 2
    else if (!gAccept) {
      printf("Error(removeProposal): Could not add vertex\n");
      v->print();
      removeVoronoi(v->pt);
      updateLogInfo();
    }
    else {
      removeDataFromVertices(v, 1);
    }
  }
  else {
    // The simplex structure was not updated, do nothing
  }

  return;
}

void Simulation::moveProposal() {
  int resSumDel = 0;
  int resSumAdd = 0;
  double pot;
  double alpha;
  Vertex * v;
  static Vertex * vCopy = new Vertex;
  point pt;
  mPoint * x;

  //Draw a random number in R^{dim}
  pt = selectPoint();
  x = movePoint(pt);

  if (!inside(x->pt)) {
    gAccept = 0;
    alpha = 0.0;

    free(x->pt);
    free(x);

    // The simplex structure was not updated, do nothing
    return;
  }
  else {
    if (gDebugPrint >= 1)
      printf("Proposal: Move point: (%6.2f %6.2f) to (%6.2f %6.2f)",
	     v->pt[0], v->pt[1], x->pt[0], x->pt[1]);

    v = findVertex(pt);
    vCopy->nullify();
    copyVertex(v, vCopy);

    assert(nPoints >= 4);

    removePointFromConvexHull(&root, vCopy->pt, get_next_neigh, site_numm_neigh);
    assert(((v->nNeigh == 0) && gAccept) || (v->nNeigh != 0) && !gAccept);


    // Check if OK
    if (!gAccept) {
      // The simplex structure was not updated, do nothing

      free(x->pt);
      free(x);

      return;
    }


    pixelDiff = 0;
    potLikelihood = 0.0;

    action = MOVE_REMOVE;
    pot = acceptProb(vCopy, &resSumDel);
      

    v = addVoronoi(x);
    gAccept = addPointToConvexHull(root);
    if (gAccept == -1) {
      restart = 1; //3

      free(x->pt);
      free(x);

      return;
    }

    action = MOVE_ADD;
    pot += acceptProb(v, &resSumAdd);

    alpha = exp(-pot);

  }

  accept = acceptance(alpha);

  if (accept) {
    if (!model->secondOrderNeighbours) {
      // Update sumColDiff
      sumColDiff -= resSumDel;
      sumColDiff += resSumAdd;
    }

    updateColDiff(v); // NBNB Er dette mulig?	
    updateColDiff(vCopy);

    updateLogInfo();

    removeVoronoi(vCopy->pt);

    removeDataFromVertices(v, 0); // Remove data from neighbours of v
  }
  else if (gAccept) {
    removePointFromConvexHull(&root, v->pt, get_next_neigh,
			      site_numm_neigh);
    if (!gAccept) {
      printf("Error(moveProposal): Could not remove vertex\n");
      v->print();

      // Must accept the move operation
      if (!model->secondOrderNeighbours) {
	sumColDiff -= resSumDel;
	sumColDiff += resSumAdd;
      }

      updateColDiff(v); // NBNB Er dette mulig?	
      updateColDiff(vCopy);

      updateLogInfo();

      removeVoronoi(vCopy->pt);

      removeDataFromVertices(v, 0); // Remove data from neighbours of v
    }
    else {
      removeVoronoi(v->pt);

      addVoronoi(vCopy->pt); // Resets input structure: next site is v->pt
      gAccept = addPointToConvexHull(root);
      if (gAccept == -1) {
	restart = 1; // 4

	free(x->pt);
	free(x);

	return;
      }
      else if (!gAccept) {
	printf("Error(moveProposal): Could not add vertex\n");
	vCopy->print();
	removeVoronoi(vCopy->pt);
      }
      else {
	removeDataFromVertices(vCopy, 1);
      }
    }
  }
  else { // The last add operation was aborted;
    // Have to add removed vertex in simplex structure
    addVoronoi(vCopy->pt); // Resets input structure: next site is v->pt
    gAccept = addPointToConvexHull(root);
    if (gAccept == -1) {
      restart = 1; // 5
	
      return;
    }
    else if (!gAccept) {
      printf("Error(moveProposal): Could not add vertex\n");
      vCopy->print();
      removeVoronoi(vCopy->pt);
    }
    else {
      removeDataFromVertices(vCopy, 1);
    }

  }

  free(x->pt);
  free(x);

  return;
}

void Simulation::updateLogInfo() {
  sumPotLikelihood += potLikelihood;
  sumPixelDiff += pixelDiff;

  return;
}

int Simulation::acceptance(double alpha) {
  double u;

  if (gDebugPrint >= 1)
    printf("%8.2f ", alpha);

  if (!gAccept)
    return 0;

  if (alpha > 1.0)
    alpha = 1.0;

  u = random->unif01();
  accept = (u<alpha);

  sumAlpha += alpha;
    
  return accept;
}
  


void Simulation::drawAction() {
  double u;

  if (nPoints <= 3) {
    action = ADD;
  }
  else {
    u = random->unif01();
    if (u<prob[ADD])
      action = ADD;
    else if (u<prob[MOVE]+prob[ADD])
      action = MOVE;
    else
      action = REMOVE;
  }

  if (gDebugPrint >= 2) {
    if (action == ADD)
      printf("Action: ADD\n");
    else if (action == REMOVE)
      printf("Action: REMOVE\n");
    else
      printf("Action: MOVE\n");
  }

  return;
}

// draws point uniformely over region S
mPoint * Simulation::drawPoint() {
  mPoint * x = (mPoint *) malloc(sizeof(mPoint));
  int i;
  int acc = 0;
  double d;
  double sum;
  double tol = 0.00001;
  Vertex * v;

  x->pt = (point) malloc(point_size);

  while (!acc) {
    for (i=0; i<dim; i++) {
      x->pt[i] = random->unif01();
      x->pt[i] += (random->unif01()-0.5) * G_INVMOD;
      x->pt[i] *= mult_up;
      if (gInteger)
	x->pt[i] = floor(x->pt[i]+0.5); /* Necessary? */
    }
    x->col = random->iUnif(0,model->nColour-1);
    
    if (gDebugCheck) {
      acc = 1;
      v = gFirstVertex;
      while (v) {
	sum = 0.0;
	for (i=0;i<dim;i++) {
	  d = v->pt[i] - x->pt[i];
	  sum += d*d;
	}
	if (sum < tol) {
	  printf("Warning(drawPoint): sum = %f\n", sum);
	  acc = 0;
	  break;
	}
	v = v->next;
      }
    }
    else {
      acc = 1;
    }
  }
  return x;
}


mPoint * Simulation::movePoint(point pt) {
  int i;
  double d;
  mPoint * x = (mPoint *) malloc(sizeof(mPoint));

  x->pt = (point) malloc(point_size);

  movePot = (dim/2.0)*log(2.0 * G_PI) + dim * log(sigmaMove);

  for (i=0;i<dim;i++) {
    x->pt[i] = random->normal(pt[i], sigmaMove);

    // Compute the move potential
    d = (x->pt[i] - pt[i])/sigmaMove;
    movePot += d*d/2.0;
  }

  x->col = random->iUnif(0,model->nColour-1);

  return x;
}


void Simulation::addVoronoi(point pt) {
  gLastAddedPoint = pt;
  s_num--;
}

Vertex * Simulation::addVoronoi(mPoint * x) {
  int i;
  int index;
  int minBatch=10000; /* Must be large enough to contain all points generated */

  if (nPoints == 0) {
    gPoints = (point) calloc(minBatch, point_size);
    gVertex = (Vertex **) calloc(minBatch, sizeof(Vertex *));
  }
  else if ((nPoints % minBatch) == 0) {
    printf("WARNING, # points exceeds %d\n", minBatch);
    gPoints = (point) realloc(gPoints, (nPoints +minBatch) * point_size);
    gVertex = (Vertex **) realloc(gVertex, (nPoints +minBatch) * sizeof(Vertex *));
  }

  // Find free position ; kan optimaliseres ved en seperat liste over fjernet
  // punkts plassering i gPoints
  for (i=0;i<nPoints;i++) {
    if (gPoints[i*gDim] == REMOVED_MARK) {
      break;
    }
  }
 
  index = i;

  if (index > gMaxPos)
    gMaxPos = index;

  for (i=0;i<dim;i++) {
    gPoints[index*gDim + i] = x->pt[i];
  }
  nPoints++;
  gnPoints = nPoints;
  gLastAddedPoint = &gPoints[index*gDim];

  addVertex(&gFirstVertex, gLastAddedPoint, x->col);

  gVertex[index] = gFirstVertex; // NEW

  return gFirstVertex;
}


point Simulation::selectPoint() {
  int i, k;
  int pos;
  point pt;

  // Draw point to be deleted
  pos = random->iUnif(0, nPoints-1);
  k = 0;
  i = 0;
  while (1) {
    if (gPoints[k*gDim] != REMOVED_MARK) {
      if (i==pos)
	break;
      i++;
    }
    k++;
  }

  pos = k;

  pt = &gPoints[pos*dim];

  return pt;
}

void Simulation::removeVoronoi(point pt) {
  int i;

  deleteVertex(&gFirstVertex, pt);
  
  nPoints--;
  gnPoints = nPoints;
  s_num--;
  
  for (i=0;i<pdim;i++)
    pt[i] = REMOVED_MARK;
  
  return;
}

void Simulation::moveVoronoi(mPoint * x) {

  return;
}

double Simulation::acceptProb(Vertex * v, int * resSum) {
  double pot = 0.0;
  int n;
  //  DataList * dl;
  Vertex * w;

  if (!gAccept)
    return 0.0;

  if (gIt >= 3 && simulateFromPosterior && gDebugPrint > 0) {
    n = 0;
    w = gFirstVertex;
    printf("FOER *************\n");
    while (w) {
      w->print();
      w->printData();
      //      if (w->pt != v->pt)
      n += w->nData;
      w = w->next;
    }
//     if (action == REMOVE)
//       n -= v->nData;

    assert(n == nDataX*nDataY);
  }

  if (simulateFromPosterior && gIt >= 3) {
    pot = potentialLikelihood(v);
    potLikelihood += pot;
  }

  pot += acceptProbPrior(v, resSum);

  if (gIt >= 3 && simulateFromPosterior && gDebugPrint > 0) {
    n = 0;
    w = gFirstVertex;
    printf("ETTER ************* Sjekk-1\n");
    while (w) {
      w->print();
      w->printData();
      if (w->pt != v->pt)
	n += w->nData;
      w = w->next;
    }
//     if (action == REMOVE)
//       n -= v->nData;

    assert(n == nDataX*nDataY);

//     if (action == ADD) {
//       n = 0;
//       w = gFirstVertex;
//       printf("ETTER ************* Sjekk-2\n");
//       while (w) {
// 	if (w->pt != v->pt) {
// 	  dl = w->dataList;
// 	  while (dl) {
// 	    n += (!dl->removed);
// 	    dl = dl->next;
// 	  }
// 	}
// 	w = w->next;
//       }

//       n += v->nData;

//       assert(n == nDataX*nDataY);
//     }
  }

  return pot;
}

double Simulation::potentialLikelihood(Vertex * v) {
  double pot;
  
  if (action == ADD || action == MOVE_ADD)
    pot = potentialLikelihoodAdd(v);
  else
    pot = potentialLikelihoodRemove(v);

  return pot;
}
double Simulation::potentialLikelihoodAdd(Vertex * v) {
  int i;
  double d1, d2;
  double pot = 0.0;
  DataList * dl;
  Vertex * w;

  for (i=0;i<v->nNeigh;i++) {
    w = v->neigh[i];
    dl = w->dataList;
    while (dl) {
      // Check if data point is closer to v than w
      d1 = distPoint(dl->d->x, v->pt); 
      d2 = distPoint(dl->d->x, w->pt);
      dl->removed = 1;
      if (d1 < d2) {
	pot += updatePotentialLikelihood(v, w, dl);
      }
      else
	dl->removed = 0;

      dl = dl->next;
    }
  }
    
  return pot;    
}

double Simulation::potentialLikelihoodRemove(Vertex * v) {
  int i;
  double d2;
  double distMin = 100000000.0;;
  double pot = 0.0;
  DataList * dl;
  Vertex * w;

  for (i=0;i<v->nNeigh;i++) {
    w = v->neigh[i];
    w->nDataLastAdded = 0;
    if (w->dataList) {
      w->dataList->removed = 0;
    }
  }

  dl = v->dataList;
  while (dl) {
    for (i=0;i<v->nNeigh;i++) {
      d2 = distPoint(dl->d->x, v->neigh[i]->pt); 
      if (d2 < distMin) {
	w = v->neigh[i];
	distMin = d2;
      }
    }
    pot += updatePotentialLikelihood(v, w, dl);

    dl = dl->next;
  }
    
  //  if (action == REMOVE || action == MOVE_REMOVE) {
  pot *= -1;
    //  }

  return pot;    
}

// Single Likelihood term
double Simulation::updatePotentialLikelihood(Vertex * v, Vertex * w,
				       DataList * dl) {  
  double pot = 0.0;
  double s;
  //  double sigma = 1.0;

  s = (dl->d->w - w->colour)/model->sigma;
  pot -= 0.5*s*s; // Assumes sigma to be constant

  s = (dl->d->w - v->colour)/model->sigma;
  pot += 0.5*s*s; // Assumes sigma to be constant

  if (action == ADD || action == MOVE_ADD) {
    pixelDiff += (v->colour != dl->d->trueCol) - (w->colour != dl->d->trueCol);
    v->insertData(dl->d);
  }	  
  else {
    pixelDiff += (w->colour != dl->d->trueCol) - (v->colour != dl->d->trueCol);
    w->insertData(dl->d);
  }

  return pot;
}

double Simulation::updateSecondOrder(Vertex * v) {
  int i, j, k;
  short col;
  short colNew=0;
  int nOfSameColour;
  int sum;
  int diff;
  int prevInf = 0;
  int first;
  int finished;
  int nDiffClique[NCLIQUES];
  double pot;
  point pt;
  point pts[2];
  point wpt;
  Vertex * w;
  simplex * s;
  simplex * t;
  simplex * sPrev;

  if (nPoints <= 3) {
    // Update clique info
    if (nPoints == 3) {
      w = gFirstVertex;
      while (w) {
	col = w->colour;
	sum = 0;
	for (i=0;i<w->nNeigh;i++) {	
	  sum += (col != w->neigh[i]->colour);
	}

	if (sum == 0)
	  w->clique = HOMOG;
	else if (sum == 1)
	  w->clique = DISCONT;
	else
	  w->clique = SINGLE;

	// Initialize nClique
	nClique[w->clique]++;

	w = w->next;
      }
    }
    pot = 0.0;
  }
  else {
    for (i=0;i<NCLIQUES;i++)
      nDiffClique[i] = 0;
      
    for (i=-1;i<v->nNeigh;i++) {
	
      if (i == -1)
	w = v;
      else
	w = v->neigh[i]; // Assume for the moment that w->infinity != 0

      //      ww = wFirst = w->neigh[0]; // Assume for the moment that ww->infinity != 0
      wpt = w->pt;
      s = w->simp[0];
      j = 1;
      while (s->infinity)
	s = w->simp[j++];

      counter--;
      s->visit = counter;
      //	printTriangle(s);
      first = 1;
      nOfSameColour = 0;
      prevInf = 0;
      diff = 0;
      finished = 0;
      sPrev = NULL;
      while (!finished) {
	// Find next simplex
	//	  col = ww->col;
	for (j=0;j<cdim;j++) {
	  if (s->neigh[j].vert != wpt && s->neigh[j].simp != sPrev) {
	    //	      pt = s->neigh[j].vert;
	    t = s->neigh[j].simp;
	    if (t->visit == counter)
	      finished = 1;
	    else
	      t->visit = counter;
	    break;
	  }
	}

	if (!s->infinity) {
	    
	  for (k=0;k<cdim;k++) {
	    pt = s->neigh[k].vert;
	    if (pt != wpt && s->neigh[k].simp == t)
	      pts[0] = pt;
	    else if (pt != wpt)
	      pts[1] = pt;
	  }

	  col = findVertex(pts[0])->colour;
	  if (prevInf) {
	    prevInf = 0;
	    diff += (col != colNew);
	  }
	    
	  colNew = findVertex(pts[1])->colour;

	  //	    printTriangle(t);
	
	  nOfSameColour += (col == w->colour);
	  diff += (col != colNew);
	}
	else if (finished) {
	  for (k=0;k<cdim;k++) {
	    pt = s->neigh[k].vert;
	    if (pt != wpt && pt != infinity)
	      break;
	  }
	  col = findVertex(pt)->colour;
	    
	  diff += (col != colNew);
	}
	else
	  prevInf = 1;

	if (!finished) {
	  sPrev = s;
	  s = t;
	}
      }

      assert(diff % 2 == 0);
      if (diff == 0) {
	if (col != w->colour) {
	  w->newClique = SINGLE;
	}
	else {
	  w->newClique = HOMOG;
	}
      }
      else if (diff == 2 && nOfSameColour > 1)
	w->newClique = CONT;
      else
	w->newClique = DISCONT;
    
      if (w != v || (action == REMOVE || action == MOVE_REMOVE)) {
	assert(w->clique >= 0 && w->clique < NCLIQUES);
	nDiffClique[w->clique]--;
      }
      if  (w != v || (action == ADD || action == MOVE_ADD)) {
	assert(w->newClique >= 0 && w->newClique < NCLIQUES);
	nDiffClique[w->newClique]++;
      }
    }
    
    pot = 0.0;
    for (i=0;i<model->nTheta;i++) {
      pot += model->theta[i] * ((double) nDiffClique[i]);
    }
    if (gDebugPrint >= 1) {
      for (i=0;i<model->nTheta;i++) {
	printf("%2d ", nDiffClique[i]);
      }
    }
    if (action == REMOVE || action == MOVE_REMOVE)
      pot *= -1.0;

    if (gDebugPrint >= 1) {
      printf("%8.2f ", pot);
    }
  }
  
  return pot;
}

double Simulation::updateFirstOrder(Vertex * v, int * resSum) {
  int i, j;
  short col;
  //  int resSum;
  int sum;
  double pot = 0.0;
  Vertex * w;


  if (nPoints <= 3) {
    // Update colour info
    if (nPoints == 3) {
      w = gFirstVertex;
      while (w) {
	col = w->colour;
	sum = 0;
	for (i=0;i<w->nNeigh;i++) {	
	  sum += (col != w->neigh[i]->colour);
	}

	w->colDiffNew = sum;

	sumColDiff += sum;
      
	w = w->next;
      }
      sumColDiff /=2;
    }
  }
  else {
    if (action == ADD || action == MOVE_ADD) {
      col = v->colour;
      sum = 0;
      for (i=0;i<v->nNeigh;i++) {
	sum += (col != v->neigh[i]->colour);
      }

      v->colDiffNew = (short) sum;
      *resSum = v->colDiffNew;
    }
    else
      *resSum = v->colDiff;

    // Update neighbours colDiff
    if (gDebugPrint >= 1) {
      printf("v: ");
      v->print();
      v->printNeigh();
    }
    for (i=0;i<v->nNeigh;i++) {
      w = v->neigh[i];
      col = w->colour;
      sum = 0;
      for (j=0;j<w->nNeigh;j++) {	
	sum += (col != w->neigh[j]->colour);
      }
      if (action == ADD || (action == MOVE_ADD && w->visit != gIt))
	*resSum += (sum - w->colDiff);
      else if (action == MOVE_ADD)
	*resSum += (sum - w->colDiffNew);
      else
	*resSum -= (sum - w->colDiff);

      w->visit = gIt;

      if (gDebugPrint >= 1) {
	printf("w:");
	w->print();
	printf("sum %d w->colDiff %d\n", sum, w->colDiff);
      }
      w->colDiffNew = sum;
    }

    if (*resSum % 2 != 0) {
      printf("ERROR(Simulation::acceptProbPrior): resSum %d\n", *resSum);
    }

    *resSum /= 2;

    pot = model->theta[0] * ((double) *resSum);
  }

  return pot;
}

simplex * nextSimplexInCycle(simplex * s, simplex * sPrev, point pt) {
  int i;
  neighbor * t;

  for (i=0;i<cdim;i++) {
    t = &(s->neigh[i]);
    if (t->simp != sPrev && t->vert != pt)
      break;
  }

  return t->simp;
}

double Simulation::findAreaComponent(Vertex * v) {
  double area = 0.0;
#ifdef NOTINUSE
  int i;
  Vertex *w;
  int colour = v->colour;

  static Vertex **st;
  static long ss = MAXDIM;
  long tms = 0;

  area = 0.0;
  counter--;
  initStackMemory();
  push(v);
  v->visit = counter;
  while(tms && area < model->areaSize) {
    stackMemory();
    pop(w);

    for (i=0;i<w->nNeigh; i++)
      if (w->visit != counter && w->colour == colour)
	push(w->neigh[i]);

    area += w->areaCell;
  }

  area = min(model->areaSize, area);
#endif
  // Remains to update all other vertices belonging to the same
  // component. Must remember to update the potential somewhere.

  return area;
}



double Simulation::findAreaCell(Vertex * v) {
  int i;
  double area;
  point pt, ptPrev;
  simplex *s;
  simplex *sNew;
  simplex *sPrev;

  sPrev = NULL;
  s = v->simp[0];

  pt = NULL;

  area = 0.0;

  for (i=0;i <= v->nSimp;i++) {
    ptPrev = pt;
    pt = s->centerpoint;

    if (ptPrev)
      area += (ptPrev[0]+pt[0])*(pt[1]-ptPrev[0]);
    
    sNew = nextSimplexInCycle(s, sPrev, v->pt);

    sPrev = s;
    s = sNew;
  }

  return area;
}


double Simulation::updateAreaInteraction(Vertex * v) {
  int i;
  double diff;
  double pot = 0.0;
  Vertex * w;

  if (nPoints < 3) {
    // Do nothing??
  }
  else if (nPoints == 3) {
    v = gFirstVertex;
    while (v) {
      v->areaCell = findAreaCell(v);
      v = v->next;
    }
  }
  else {

    // Should expand each of the neighbours (taking into account that
    // same belong to same cluster may speed up the search) so that the
    // same colour area is found up to a size 'areaSize'

    // Start with finding new areas for v and its neighbours
    v->newAreaCell = findAreaCell(v);
    for (i=0;i<v->nNeigh;i++) {
      w = v->neigh[i];

      w->newAreaCell = findAreaCell(w);
    }

    v->newAreaComponent = findAreaComponent(v);
    for (i=0;i<v->nNeigh;i++) {
      w = v->neigh[i];
      if (w->colour == v->colour)
	w->newAreaComponent = v->newAreaComponent;
      else
	w->newAreaComponent = findAreaComponent(w);
    }

    // Then scan Voronoi tesselation for the connected component of v
    // exit when max area size is exceeded or when the
    // connected component is found. Use its area in updating
    // the potential

    diff = (w->newAreaComponent < model->areaSize) - (w->areaComponent < model->areaSize);
    pot += diff * model->areaPenalty;
  }

  return pot;
}

double Simulation::acceptProbPrior(Vertex * v, int * resSum) {
  int n;
  double r;
  double pot;

  if (model->secondOrderNeighbours)
    pot = updateSecondOrder(v);
  else if (model->areaInteraction)
    pot = updateAreaInteraction(v);
  else
    pot = updateFirstOrder(v, resSum);

  //     if (action == ADD || action == MOVE_ADD) //NBNBMOVE FEIL
  //       n = nPoints-1;
  //     else
  if (nPoints > 3) {
    n = nPoints;

    if (action == MOVE)
      n--;

    r = model->beta * prob[REMOVE]/prob[ADD];
    r /= (double) n;

    if (action == MOVE_ADD || action == MOVE_REMOVE) {
      r /= exp(-movePot);
    }
    else {
      r *= volRegion;
    }

    pot -= log(r);

    if (action == ADD || action == MOVE_ADD) {
      //      alpha = r;
    }
    else {
      pot *= -1;
    }
    if (gDebugPrint >=  1)
      printf("resSum, r: %d, %6.2f\n", *resSum, r);
  }
  else
    pot = 0.0;

  return pot;
}


void Simulation::updateColDiff(Vertex *v) {
  int i;
  Vertex * w;
  
  if (model->areaInteraction) {
    for (i=0;i<v->nNeigh;i++) {
      w = v->neigh[i];
      w->areaCell = w->newAreaCell;
      w->areaComponent = w->newAreaComponent;
    }
    v->areaCell = v->newAreaCell; // dummy if remove
    v->areaComponent = v->newAreaComponent; // dummy if remove
  }
  if (model->secondOrderNeighbours && gIt>2) {
    for (i=0;i<v->nNeigh;i++) {
      w = v->neigh[i];
      nClique[w->clique]--;
      nClique[w->newClique]++;
      w->clique = w->newClique;
    }
    if (action == REMOVE || action == MOVE_REMOVE)
      nClique[v->clique]--;
    if (action == ADD || action == MOVE_ADD) {
      nClique[v->newClique]++;
      v->clique = v->newClique;
    }
  }
  else if (nPoints >= 3){
    for (i=0;i<v->nNeigh;i++) {
      w = v->neigh[i];
      w->colDiff = w->colDiffNew;
    }
    v->colDiff = v->colDiffNew; //Dummy statement if action == REMOVE
  }
  return;
}

void addNeigh(Vertex * v, Vertex *vNew) {
int i;
for (i=0;i<v->nNeigh;i++) { // NBNB Lite effektivt
if (v->neigh[i] == vNew) {
break;
}
}
if (i == v->nNeigh) {
  if (v->nNeigh >= 10) {
    v->neigh = (Vertex **) realloc(v->neigh, (v->nNeigh+1)*sizeof(Vertex *));
    v->mark = (short *) realloc(v->mark, (v->nNeigh+1)*sizeof(short *));
  }
  v->mark[i] = 0;
  v->neigh[i] = vNew;
  v->nNeigh++;
}
else {
  v->mark[i]++;
}

return;
}

void removeNeigh(Vertex * v, Vertex * vDel) {
  int i, j;

  for (i=0;i<v->nNeigh;i++) { // NBNB Lite effektivt
    if (v->neigh[i] == vDel) {
      break;
    }
  }
  if (i == v->nNeigh)
    assert(i < v->nNeigh);

  if (v->mark[i])
    v->mark[i]--;
  else {
    v->nNeigh--;
    for (j=i;j<v->nNeigh;j++) {
      v->neigh[j] = v->neigh[j+1];
      v->mark[j] = v->mark[j+1];
    }
  }

  return;
}

void copyVertex(Vertex * from, Vertex * to) {
  int i;
  DataList * dlfrom;

  to->nSimp = from->nSimp;
  to->nNeigh = from->nNeigh;
  if (!to->simp)
    to->simp = (simplex **) malloc(100*sizeof(simplex *));
  if (!to->neigh)
    to->neigh = (Vertex **) malloc(100*sizeof(Vertex *));
  if (!to->mark)
    to->mark = (short *) malloc(100*sizeof(short));
  for (i=0;i<to->nNeigh;i++)
    to->neigh[i] = from->neigh[i];
  for (i=0;i<to->nSimp;i++)
    to->simp[i] = from->simp[i];
  for (i=0;i<to->nNeigh;i++)
    to->mark[i] = from->mark[i];
  to->pt = from->pt;
  to->colour = from->colour;
  to->clique = from->clique;
  to->pixelDiff = from->pixelDiff;
  to->it = from->it;
  to->colDiff = from->colDiff;

  to->nData = 0;
  to->dataList = NULL;
  dlfrom = from->dataList;
  while (dlfrom) {
    to->insertData(dlfrom->d);
    dlfrom = dlfrom->next;
  }


  return;
}
  


Vertex * findVertex(point pt) {
  int index;
  Vertex * v;
//  Vertex * v = gFirstVertex;

//   while (v) {
//     if (v->pt == pt)
//       break;
//     v = v->next;
//   }

  index = site_numm(pt);

  assert(index >= 0);

  v = gVertex[index];

  assert(v);

  return v;
}

void addVertex(Vertex ** vList, point vert, short colour) {
  Vertex * v = new Vertex(vert, colour);

  v->next = (*vList);
  if (*vList)
    (*vList)->prev = v;
  (*vList) = v;

  v->prev = NULL;
  

  return;
}
void deleteVertex(Vertex ** vList, point pt) {
  Vertex * v;
  Vertex * vPrev;

//   v = gFirstVertex;
//   while (v) {
//     if (v->pt == pt)
//       break;
//     vPrev = v;
//     v = v->next;
//   }
//   if (!v)
//     assert(v);

  v = findVertex(pt);

  vPrev = v->prev;

  if (vPrev)
    vPrev->next = v->next;
  else
    (*vList) = v->next;

  if (v->next)
    v->next->prev = vPrev;

  delete v;

  return;
}

void resetVertices() {
  Vertex * v;

  v = gFirstVertex;
  while (v) {
    v->reset();

    v = v->next;
  }

  return;
}

void deleteVertices() {
  Vertex * v;
  Vertex * vPrev = NULL;

  v = gFirstVertex;
  while (v) {
    vPrev = v;
    v = v->next;

    delete vPrev;
  }

  gFirstVertex = NULL;

  return;
}
 

void freeEdges(Edge ** first) {
  Edge * e = *first;
  Edge * eNext;

  while(e) {
    eNext = e->next;
    if (e->pt1) {
      free(e->pt1);
      free(e->pt2);
    }
    free(e);
    e = eNext;
  }
  *first = NULL;
}


void Simulation::buildConvexHull() {
  int i;

  //Initialize variables
  s_num = 0;
  for (i=0; i<MAXDIM; i++) {
    mins[i] = DBL_MAX;
    maxs[i] = -DBL_MAX;
  }

  root = build_convex_hull(get_next_site, site_numm, dim, DELAUNAY);

  return;
}

void Simulation::writeToFile() {
  int i, j;
  char *name;
  Vertex * v;
  FILE * outFilePts;

  name = (char *) malloc((strlen(fileName)+5)*sizeof(char));
  strcpy(name, fileName);
  outFilePts = fopen(strcat(name, ".pts"), "w");
  free(name);

  //  out_func* mof = & vlist_out;
  //  visit_func *pr =  facets_print;

  //make_output(root, visit_hull, pr, mof, OUTFILE);


  findDirichletEdges();
  sortDirichletEdges();
  truncateDirichletEdges();

  writeToSFile(NULL, fileName);

  for (i=0;i<nPoints;i++) {
    if (gPoints[i*dim] != REMOVED_MARK) {
      for (j=0;j<dim;j++) {
	fprintf(outFilePts, "%.12g ", gPoints[i*dim + j]);
      }
      v = findVertex(&gPoints[i*dim]);
      fprintf(outFilePts, "%d ", v->colour);
      fprintf(outFilePts, "\n");
    }
  }
  //  fclose(OUTFILE);
  fclose(outFilePts);

  return;
}


void writeToSFile(simplex * root, char * fileName) {
  //  simplex *s;
  int i, j, k;
  int ind;
  int vInd;
  char * name;
  Edge * e;
  Vertex * v;
  simplex * s;
  simplex * t;
  simplex * start;
  Deldir * deldir;
  Deldir::Delsgs *del;
  Deldir::Dirsgs *dir;
  FILE * file;

  static simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  fflush(NULL);

  name = (char *) malloc((strlen(fileName)+6)*sizeof(char));
  strcpy(name, fileName);
  file = fopen(strcat(name, ".sres"), "w");
  free(name);

  gnEdges = 0;

  if (!root) {
    v = gFirstVertex;
    while (v) {
      for (i=0;i<v->nSimp;i++) {
	s = v->simp[i];
	for (j=0;j<2;j++)
	  for (k=j+1;k<3;k++)
	    if (s->neigh[j].vert != infinity && s->neigh[k].vert != infinity)
	      addEdge(&gFirstEdge, &gnEdges, s->neigh[j].vert, s->neigh[k].vert);
      }
      v = v->next;
    }
  }
  else {
    counter--;
    start = findFirst(root);
    counter--;
    initStackMemory();
    push(start);
    start->visit = counter;
    while(tms) {
      stackMemory();
      pop(s);

      for (j=0;j<2;j++)
	for (k=j+1;k<3;k++)
	  if (s->neigh[j].vert != infinity && s->neigh[k].vert != infinity)
	    addEdge(&gFirstEdge, &gnEdges, s->neigh[j].vert, s->neigh[k].vert);
      for (j=0;j<cdim;j++) {
	t = s->neigh[j].simp;
	if (t->visit != counter) {
	  push(t);
	  t->visit = counter;
	}
      }
    }

  }

  // NBNB: temporary region = 0,100,0,100
  deldir = new Deldir(gnEdges, gnPoints, 0, 100, 0, 100);
  
  // Make a Deldir object
  // Delaunay triangulations

  // One method: Recursively span all simplices
  //             and then record the edges
  //  findDelaunay(root);
  del = &(deldir->delsgs);
  dir = &(deldir->dirsgs);

  e = gFirstEdge;
  for (i=0;i<gnEdges;i++) {
    del->x1[i] = gPoints[e->p1*gDim];
    del->y1[i] = gPoints[e->p1*gDim+1];
    del->x2[i] = gPoints[e->p2*gDim];
    del->y2[i] = gPoints[e->p2*gDim+1];
    e = e->next;
  }

  freeEdges(&gFirstEdge);

  i = 0;
  ind = 0;
  while(ind<gnPoints) {
    if (gPoints[i*gDim] != REMOVED_MARK) {
      deldir->x[ind] = gPoints[i*gDim];
      deldir->y[ind] = gPoints[i*gDim+1];
      ind++;
    }
    i++;
  }

  if (!root) {
    vInd = 0;
    ind = 0;
    v = gFirstVertex;
    while (v) {
      e = v->dirEdges;
      while (e) {
	if (ind > 2*deldir->nEdges-1) {
	  dir->cell = (short *) realloc(dir->cell, (ind+1)*sizeof(short));
	  dir->x1 = (double *) realloc(dir->x1, (ind+1)*sizeof(double));
	  dir->x2 = (double *) realloc(dir->x2, (ind+1)*sizeof(double));
	  dir->y1 = (double *) realloc(dir->y1, (ind+1)*sizeof(double));
	  dir->y2 = (double *) realloc(dir->y2, (ind+1)*sizeof(double));
	  dir->colour = (short *) realloc(dir->colour, (ind+1)*sizeof(short));
	}
	dir->cell[ind] = vInd;
	dir->x1[ind] = e->pt1[0];
	dir->x2[ind] = e->pt2[0];
	dir->y1[ind] = e->pt1[1];
	dir->y2[ind] = e->pt2[1];
	dir->colour[ind] = v->colour;
	ind++;
	gnDirEdges++;
	e = e->next;
      }
      v = v->next;
      vInd++;
    }
  }    

  // Write a deldir object, so that it can be read by source
  // creating an object to be used by plot.deldir

  //Header
  fprintf(file, "\"x\"<-\n");
  fprintf(file, "structure(.Data = list(");

  //Write $delsgs (Needs the x1, y1, x2, y2 columns)
  fprintf(file, "delsgs = structure(.Data = c(");
  // print x1
  for (i=0;i<gnEdges;i++)
    fprintf(file, "%8.2f, ", del->x1[i]);
  // print y1
  for (i=0;i<gnEdges;i++)
    fprintf(file, "%8.2f, ", del->y1[i]);
  // print x2
  for (i=0;i<gnEdges;i++)
    fprintf(file, "%8.2f, ", del->x2[i]);
  // print y2
  for (i=0;i<gnEdges;i++)
    fprintf(file, "%8.2f, ", del->y2[i]);
  // print ind 1
  // print ind 2
  for (i=0;i<2*gnEdges-1;i++)
    fprintf(file, "0, ");

  fprintf(file, "0), .Dim = c(");
  // print dimensions
  fprintf(file, "%d, %d), ", gnEdges, 6);

  fprintf(file, ".Dimnames = list(character(0), c(\"x1\", \"y1\", \"x2\", \"y2\", \"ind1\", \"ind2\"))), ");

  //Write $dirsgs (Needs the cell, x1, y1, x2, y2, colour columns)
  if (gnDirEdges > 0) {
    fprintf(file, "dirsgs = structure(.Data = list(");
    //print cell
    fprintf(file, "cell = c(");
    for (i=0; i<gnDirEdges-1; i++)
      fprintf(file, "%d, ", dir->cell[i]);
    fprintf(file, "%d", dir->cell[gnDirEdges-1]);
    fprintf(file, "), ");

    //print x1
     fprintf(file, "x1 = c(");
    for (i=0; i<gnDirEdges-1; i++)
      fprintf(file, "%8.2f, ", dir->x1[i]);
    fprintf(file, "%8.2f", dir->x1[gnDirEdges-1]);
    fprintf(file, "), ");

    //print y1
    fprintf(file, "y1 = c(");
    for (i=0; i<gnDirEdges-1; i++)
      fprintf(file, "%8.2f, ", dir->y1[i]);
    fprintf(file, "%8.2f", dir->y1[gnDirEdges-1]);

    fprintf(file, "), ");
    //print x2
    fprintf(file, "x2 = c(");
    for (i=0; i<gnDirEdges-1; i++)
      fprintf(file, "%8.2f, ", dir->x2[i]);
    fprintf(file, "%8.2f", dir->x2[gnDirEdges-1]);

    fprintf(file, "), ");
    //print y2
    fprintf(file, "y2 = c(");
    for (i=0; i<gnDirEdges-1; i++)
      fprintf(file, "%8.2f, ", dir->y2[i]);
    fprintf(file, "%8.2f", dir->y2[gnDirEdges-1]);
    fprintf(file, "), ");


    //print colour
    fprintf(file, "colour = c(");
    for (i=0; i<gnDirEdges-1; i++)
      fprintf(file, "%d, ", dir->colour[i]);
    fprintf(file, "%d", dir->colour[gnDirEdges-1]);
    fprintf(file, ")");

  //print ind1
  //print ind2
  //print bp1
  //print bp2
    fprintf(file, "), row.names = c(");
    for (i=0;i<gnDirEdges-1; i++)
      fprintf(file, "\"%d\", ", i);
    fprintf(file, "\"%d\"), ", gnDirEdges-1);
    fprintf(file, " class = \"data.frame\")");
  }
  // Write $summary (needs only the x,y column)
  fprintf(file, ", summary = structure(.Data = c(\n");
  // write x
  for (i=0;i<gnPoints;i++)
    fprintf(file, "%8.2f, ", deldir->x[i]);
  // write y
  for (i=0;i<gnPoints-1;i++)
    fprintf(file, "%8.2f, ", deldir->y[i]);
  fprintf(file, "%8.2f)", deldir->y[i]);
  //write dim
  fprintf(file, ", .Dim = c(");
  fprintf(file, "%d, %d)", gnPoints, 2);
  fprintf(file, ", .Dimnames = list(character(0), c(\"x\", \"y\")))");

  // Write $n.data
  fprintf(file, ", n.data = %d\n", gnPoints);
  // Write $rw
  fprintf(file, ", rw = c(%8.2f, %8.2f, %8.2f, %8.2f))",
	  deldir->rw[0], deldir->rw[1], deldir->rw[2], deldir->rw[3]);
  fprintf(file, ", class = \"deldir\")\n");

  delete deldir;

  fclose(file);

  return;
}


void Simulation::findDirichletEdges() {
  int i, j, k;
  Vertex * v = gFirstVertex;
  short * foundNeighbour;
  simplex *s;
  simplex *t;


  while (v) {
    foundNeighbour = (short *) calloc(v->nSimp, sizeof(short));
    for (i=0;i<v->nSimp-1;i++) {
      s = v->simp[i];
      if (gDebugPrint >= 2)
	printTriangle(s);
      
      for (j=i+1;j<v->nSimp;j++) {
	t = v->simp[j];
	for (k=0;k<cdim;k++) {
	  if (s == t->neigh[k].simp) {
	    foundNeighbour[i] = foundNeighbour[j] = 1;
	    if (!s->infinity || !t->infinity) {
	      addDirichletEdge(v, i, j);
	    }
	  }
	}
      }
    }
    for (i=0;i<v->nSimp;i++) {
      if (!foundNeighbour[i])
	break;
    }
    
    if (!(i == v->nSimp))
      assert(i == v->nSimp);

    free(foundNeighbour);

    v = v->next;
  }


  return;
}

double distPoint(point pt1, point pt2) {
  int i;
  double h;
  double dist = 0.0;
  for (i=0;i<pdim;i++) {
    h = pt2[i] - pt1[i];
    dist += h*h;
  }

  return sqrt(dist);
}

void Simulation::addDirichletEdge(Vertex *v, int i1, int i2) {
  int i, j;
  int n;
  int index;
  int index1, index2;
  int sign;
  int newEdge = 1;
  int count;
  int out;
  int area0, area1;
  int first;
  int ind;
  int inds[4];
  double s;
  double smin[4];
  double r[4];
  double pos[4];
  double mult;

  //  point d;
  point pt = (point) malloc(point_size);
  longPoint ptN = (longPoint) malloc(point_size);
  point N;
  longPoint * Tp = (longPoint *) malloc(3*sizeof(longPoint));
  Edge * e = (Edge *) malloc(sizeof(Edge));
  simplex *s1 = v->simp[i1];
  simplex *s2 = v->simp[i2];
  simplex *h;

  e->pt1 = (point) malloc(point_size);
  e->pt2 = (point) malloc(point_size);
  
  if (s1->infinity) {
    h = s2;
    s2 = s1;
    s1 = h;
  }

  if (s2->infinity) {
    // s2 outside convex 2D hull, sharing a common
    // edge with the hull
    for (i=0;i<pdim;i++)
      pt[i] = s1->centerpoint[i];
    N = s2->direction;

    shareEdge(s2, s1, Tp, &index1, &index2);

    if (fabs(N[0])>fabs(N[1]))
      index = 0;
    else
      index = 1;
    
    mult = 1.0/fabs(N[index]);

    for (i=0;i<pdim;i++)
      ptN[i] = s2->centerpoint[i]+ mult*N[i];

    area0 = Areasign(Tp[0], Tp[1], Tp[2]);
    area1 = Areasign(Tp[0], Tp[1], ptN);
    
    if (area0 == area1)
      sign = -1;
    else
      sign = 1;
      

    out = outside(pt, region);

    if (out) {
      newEdge = 0;
      if (pt[out-1] < region[(out-1)*2])
	newEdge = (sign * N[out-1] > 0);
      else if (pt[out-1] > region[(out-1)*2+1])
	newEdge = (sign * N[out-1] < 0);
    }
    if (!newEdge) {
      free(e->pt1);
      free(e->pt2);
      free(e);
      free(Tp);
      free(pt);
      free(ptN);

      return;
    }

    newEdge = 0;
    count = 0;
    n = 0;
    for (index=0;index<pdim;index++) {
      for (i=0;i<2;i++) {
	s = (region[count] - pt[index])/(sign*N[index]);
	if (s>0.0) {
	  for (j=0;j<n;j++)
	    if (s<smin[j])
	      break;
	  ind = j;

	  for (j=n;j>ind;j--) {
	    smin[j] = smin[j-1];
	    r[j] = r[j-1];
	    pos[j] = pos[j-1];
	    inds[j] = inds[j-1];
	  }
	  r[ind] = region[count];
	  smin[ind] = s;
	  pos[ind] = pt[1-index] + smin[ind] * (sign*N[1-index]);
	  inds[ind] = index;
	  n++;
	}
	count++;
      }
    }

    index = 0;
    if (out) {
      first = 1;
      for (i=0;i<n;i++) {
	if (pos[i] > region[2*inds[i]] && pos[i] <= region[2*inds[i]+1]) {
	  if (first)
	    first = 0;
	  else
	    break;
	}
      }
      index = i;

      if (index < n)
	newEdge = index;
    }
    else {
      newEdge = (n>0);
      index = 0;
    }

    if (newEdge) {
      for (i=0;i<pdim;i++) {
	pt[i] += smin[index] * (sign*N[i]);
      }
      pt[inds[index]] = r[index];

      for (i=0;i<pdim;i++) {
	e->pt1[i] = s1->centerpoint[i];
	e->pt2[i] = pt[i];
      }
    }

  }
  else {

    for (i=0;i<pdim;i++) {
      e->pt1[i] = s1->centerpoint[i];
      e->pt2[i] = s2->centerpoint[i];
      
    }
  }

  free(ptN);
  free(Tp);
    
  free(pt);

  if (newEdge) {
    e->next = v->dirEdges;
    v->dirEdges = e;
    
    v->nDirEdges++;
  }
  else {
    free(e->pt1);
    free(e->pt2);
    free(e);
  }

  return;
}

int outside(point pt, double * region) {
  int index;
  int count = 0;
  int out=0;

  for (index=0;index<pdim;index++) {
    if (pt[index] >= region[count+1] ||
	pt[index] <= region[count]) {
      out = index+1;
    }
    count += 2;
  }
  
  return out;
}


int truncateEdge(Edge * e, double * region, int iEdge) {
  int i, j;
  int out;
  int sign;
  int count;
  int index;
  int truncate;
  int insideRegion;
  double r;
  double s, smax;
  double tmp;
  double eps = 0.0000000001;
  point d = (point) malloc(point_size);
  point pt;

  for (i=0;i<pdim;i++) {
    d[i] = e->pt2[i] - e->pt1[i];
  }

  if (iEdge == 1) {
    out = outside(e->pt1, region);
    pt = e->pt2;
    sign = -1;
  }
  else { // iEdge == 2
    out = outside(e->pt2, region);
    pt = e->pt1;
    sign = 1;
  }

  if (!out) {
    free(d);
    return out;
  }  

  truncate = 0;
  count = 0;
  for (index=0;index<pdim;index++) {
    for (i=0;i<2;i++) { // Min and Max
      s = (region[count] - pt[index])/(sign * d[index]);
      if (s>0.0 && s<=1.0) {
	// Check if crosspoint is inside region
	insideRegion = 1;
	for (j=0;j<pdim;j++) {
	  if (j != index) {
	    tmp = pt[j]+s*sign*d[j];
	    if (tmp < region[j*2]-eps || tmp > region[j*2+1]+eps) { // (tmp<min || tmp> max)
	      insideRegion = 0;
	      break;
	    }
	  }
	}
	      
	if (insideRegion) {
	  if (!truncate) {
	    truncate = index+1;
	    r = region[count];
	    smax = s;
	  }
	  else {
	    if (s > smax) {
	      smax = s;
	      r = region[count];
	      truncate = index+1;
	    }
	  }
	}
      }
      count++;
    }
  }

  if (truncate) {

    if (iEdge == 1) {
      for (i=0;i<pdim;i++) {
	e->pt1[i] = pt[i] + smax*sign*d[i];
      }
      e->pt1[truncate-1] = r;
    }
    else {
      for (i=0;i<pdim;i++) {
	e->pt2[i] = pt[i] + smax*sign*d[i];
      }
      e->pt2[truncate-1] = r;
    }

  // Compute correct index for outside
    out = truncate;
  }
  else
    out = -1;

  free(d);

  return out;
}

void Simulation::truncateDirichletEdges() {
  int i;
  int index;
  int out, out2, outFirst;
  double val;
  Vertex * v = gFirstVertex;
  point d = (point) malloc(point_size);
  Edge * e;
  Edge * ePrev;
  Edge * newE;
  Edge * newE2;
  Edge * newE3;
 
  while (v) {
    e = v->dirEdges;

    outFirst = truncateEdge(e, region, 1);
    while (outFirst == -1) {
      ePrev = e;
      e = e->next;
      free(ePrev->pt1);
      free(ePrev->pt2);
      free(ePrev);
      outFirst = truncateEdge(e, region, 1);
    }
    v->dirEdges = e;

    while (1) {
      out = truncateEdge(e, region, 2); // always >= 0

      if (!e->next)
	break;
      if (out) {
	out2 = truncateEdge(e->next, region, 1);
	assert(out2 != 0);

	if (out2 > 0) {
	  newE = (Edge *) malloc(sizeof(Edge));
	  newE->pt1 = (point) malloc(point_size);
	  newE->pt2 = (point) malloc(point_size);
	  if (out2 == out) {
	    for (i=0;i<pdim;i++) {
	      newE->pt1[i] = e->pt2[i];
	      newE->pt2[i] = e->next->pt1[i];
	    }
	    newE->next = e->next;
	    e->next = newE;
	    e = newE->next;
	  }
	  else { // Two new edges added
	    newE2 = (Edge *) malloc(sizeof(Edge));
	    newE2->pt1 = (point) malloc(point_size);
	    newE2->pt2 = (point) malloc(point_size);

	    newE->pt1[0] = e->pt2[0];
	    newE->pt1[1] = e->pt2[1];
	    newE->pt2[out-1] = e->pt2[out-1];
	    newE->pt2[out2-1] = e->next->pt1[out2-1];

	    newE2->pt1[out-1] = e->pt2[out-1];
	    newE2->pt1[out2-1] = e->next->pt1[out2-1];
	    newE2->pt2[0] = e->next->pt1[0];
	    newE2->pt2[1] = e->next->pt1[1];

	    newE2->next = e->next;
	    newE->next = newE2;
	    e->next = newE;

	    e = newE2->next;
	  }
	}
	else {
	  ePrev = e->next;
	  e->next = e->next->next;
	  free(ePrev->pt1);
	  free(ePrev->pt2);
	  free(ePrev);
	}
      }
      else {
	e = e->next;
      }

    }

    assert(!((outFirst == 0 && out > 0) || (outFirst > 0 && out == 0)));

    if (out) {
      newE = (Edge *) malloc(sizeof(Edge));
      newE->pt1 = (point) malloc(point_size);
      newE->pt2 = (point) malloc(point_size);
      if (outFirst == out) {
	if (e->pt2[out-1] == v->dirEdges->pt1[outFirst-1]) {
	  for (i=0;i<pdim;i++) {
	    newE->pt1[i] = e->pt2[i];
	    newE->pt2[i] = v->dirEdges->pt1[i];
	  }
	  newE->next = NULL;
	  e->next = newE;
	}
	else {
	   // Three new edges added
	  for (i=0;i<pdim;i++)
	    d[i] = e->pt2[i] - e->pt1[i];
	  index = 1- (out-1); // Assuming pdim == 2
	  if (d[index] < 0)
	    val = region[2*index];
	  else
	    val = region[2*index+1];

	  newE2 = (Edge *) malloc(sizeof(Edge));
	  newE2->pt1 = (point) malloc(point_size);
	  newE2->pt2 = (point) malloc(point_size);

	  newE3 = (Edge *) malloc(sizeof(Edge));
	  newE3->pt1 = (point) malloc(point_size);
	  newE3->pt2 = (point) malloc(point_size);

	  newE->pt1[0] = e->pt2[0];
	  newE->pt1[1] = e->pt2[1];
	  //
	  newE->pt2[out-1] = e->pt2[out-1];
	  newE->pt2[index] = val;

	  newE2->pt1[out-1] = e->pt2[out-1];
	  newE2->pt1[index] = val;
	  newE2->pt2[index] = val;
	  newE2->pt2[out-1] = v->dirEdges->pt1[out-1];


	  newE3->pt1[out-1] = v->dirEdges->pt1[out-1];
	  newE3->pt1[index] = val;
	  //
	  newE3->pt2[0] = v->dirEdges->pt1[0];
	  newE3->pt2[1] = v->dirEdges->pt1[1];

	  newE3->next = e->next;
	  newE2->next = newE3;
	  newE->next = newE2;
	  e->next = newE;

	  e = newE3->next;
	}	  
      }
      else { // Two new edges added
	newE2 = (Edge *) malloc(sizeof(Edge));
	newE2->pt1 = (point) malloc(point_size);
	newE2->pt2 = (point) malloc(point_size);

	newE->pt1[0] = e->pt2[0];
	newE->pt1[1] = e->pt2[1];
	newE->pt2[out-1] = e->pt2[out-1];
	newE->pt2[outFirst-1] = v->dirEdges->pt1[outFirst-1];

	newE2->pt1[out-1] = e->pt2[out-1];
	newE2->pt1[outFirst-1] = v->dirEdges->pt1[outFirst-1];
	newE2->pt2[0] = v->dirEdges->pt1[0];
	newE2->pt2[1] = v->dirEdges->pt1[1];

	newE2->next = e->next;
	newE->next = newE2;
	e->next = newE;

	e = newE2->next;
      }
    }
    v = v->next;
  }

  free(d);

  return;
}



void sortDirichletEdges() {
  int i;
  int n;
  int stop;
  double h;
  Vertex * v = gFirstVertex;
  Edge * e;
  Edge * ee;
  Edge * ePrev;
  Edge * eList;
  Edge * neigh;
  Edge * newE;
  Edge * last;

  while (v) {
    e = v->dirEdges;

    assert(v->nDirEdges);

    if (v->nDirEdges > 1) {
      // find edge with only one neighbour
      stop = 0;
      while (!stop) {
	ee = v->dirEdges;
	n = 0;
	while (ee) {
	  if (((ee->pt1[0] == e->pt1[0] && ee->pt1[1] == e->pt1[1]) ||
	       (ee->pt1[0] == e->pt2[0] && ee->pt1[1] == e->pt2[1]) ||
	       (ee->pt2[0] == e->pt1[0] && ee->pt2[1] == e->pt1[1]) ||
	       (ee->pt2[0] == e->pt2[0] && ee->pt2[1] == e->pt2[1]))
	      && ee != e) {
	    if (n == 0)
	      neigh = ee;
	    n++;
	  }
	  ee = ee->next;
	}
	if (!(n>=1 && n<=2)) {
	  assert(n>=1 && n<=2);
	}
	if (n==1)
	  break;

	if (e->next)
	  e = e->next;
	else
	  stop = 1;
      }

      if ((e->pt1[0] == neigh->pt2[0]) && (e->pt1[1] == neigh->pt2[1]) ||
	  ((e->pt1[0] == neigh->pt1[0]) &&(e->pt1[1] == neigh->pt1[1]))) {
	// Swap e's points
	for (i=0;i<2;i++) {
	  h = e->pt1[i];
	  e->pt1[i] = e->pt2[i];
	  e->pt2[i] = h;
	}
      }
    }
    
    ePrev = NULL;

    eList = (Edge *) malloc(sizeof(Edge));
    eList->pt1 = (point) malloc(point_size);
    eList->pt2 = (point) malloc(point_size);
    for (i=0;i<pdim;i++) {
      eList->pt1[i] = e->pt1[i];
      eList->pt2[i] = e->pt2[i];
    }
    eList->next = NULL;
    last = eList;

    n = 1;
    //    addEdge(&eList, &n, e->pt1, e->pt2);
    while (n<v->nDirEdges) {
      
      // find neighbour to e
      ee = v->dirEdges;
      while (ee) {
	if (((ee->pt1[0] == e->pt1[0] && ee->pt1[1] == e->pt1[1]) ||
	     (ee->pt1[0] == e->pt2[0] && ee->pt1[1] == e->pt2[1]) ||
	     (ee->pt2[0] == e->pt1[0] && ee->pt2[1] == e->pt1[1]) ||
	     (ee->pt2[0] == e->pt2[0] && ee->pt2[1] == e->pt2[1])) &&
	    ee != ePrev && ee != e) {
	  break;
	}
	ee = ee->next;
      }

      assert(ee);

      ePrev = e;
      e = ee;

      assert(!((e->pt2[0] == ePrev->pt1[0]) && (e->pt2[1] == ePrev->pt1[1])));

      if ((e->pt2[0] == ePrev->pt2[0]) && (e->pt2[1] == ePrev->pt2[1])) {
	for (i=0;i<2;i++) {
	  h = e->pt1[i];
	  e->pt1[i] = e->pt2[i];
	  e->pt2[i] = h;
	}

      }
      //      addEdge(&eList, &n, e->pt1, e->pt2);
      newE = (Edge *) malloc(sizeof(Edge));
      newE->pt1 = (point) malloc(point_size);
      newE->pt2 = (point) malloc(point_size);
      for (i=0;i<pdim;i++) {
	newE->pt1[i] = e->pt1[i];
	newE->pt2[i] = e->pt2[i];
      }
      newE->next = NULL;
      last->next = newE;
      last = newE;

      n++;

    }


    freeEdges(&(v->dirEdges));

    v->dirEdges = eList; // NB: memory leakage

    v = v->next;
  }


  return;
}



int inTri2D(longPoint * Tp, longPoint pt) {
  int area0, area1, area2;
  int inside = 1;

  area0 = Areasign(pt, Tp[0], Tp[1]);
  area1 = Areasign(pt, Tp[1], Tp[2]);
  area2 = Areasign(pt, Tp[2], Tp[0]);

  if ((area0<0 && area1<0 && area2<0) || (area0>0 && area1>0 && area2>0))
    inside = -1;

  return inside;
}

int Areasign(longPoint a, longPoint b, longPoint c) {
  int sign;
  //  long double tol = 0.0000000001;
  //  long double area2;
  double tol = 0.000000000000001; // 10^-16
  double area2;

  area2 = (b[0]-a[0])*(c[1]-a[1]) - (c[0]-a[0])*(b[1]-a[1]);

  if (area2 < tol && area2 > - tol) {
    printf("Warning(Areasign): area = %.12g, it = %d\n", area2, gIt);
    printf("a: %.12g %.12g  b: %.12g %.12g  c: %.12g %.12g\n",
	   a[0], a[1], b[0], b[1], c[0], c[1]);
    sign = 0;
  }
  else
    sign = (area2<0) ? -1 : 1;


  return sign;
}

void Simulation::checkCliques() {
  int i;
  int n[NCLIQUES];
  Vertex *  v = gFirstVertex;

  for (i=0;i<NCLIQUES;i++)
    n[i] = 0;

  while (v) {
    n[v->clique]++;
    v = v->next;
  }

  for (i=0;i<NCLIQUES;i++) {
    if (nClique[i] != n[i])
      printf("Error(Simulation::checkCliques): nClique[i] %d != n[i] %d\n",
	     nClique[i], n[i]);
  }
  return;
}

void Simulation::checkDataInVertexStructure() {
  int n = 0;
  int nDiff = 0;
  Vertex *  v = gFirstVertex;

  while (v) {
    n += v->nData;
    nDiff += v->findPixelDiff();

    v = v->next;
  }
  
  assert(n == nDataX*nDataY);

  if (nDiff != sumPixelDiff) {
    printf("Error(checkDataInVertexStructure): nDiff %d != sumPixelDiff %d\n",
	   nDiff, sumPixelDiff);
    sumPixelDiff = nDiff;
  }
  return;
}


void Simulation::initializeDataInVertexStructure() {
  int i, j;
  point pt;
  double dist, distMin;
  Vertex * v;
  Vertex * vMin;

  for (i=0;i<nDataX; i++) {
    for (j=0;j<nDataY; j++) {
      pt = data[i][j].x;

      v = gFirstVertex;
      distMin = distPoint(v->pt, pt);
      vMin = v;
      while (v) {
	dist = distPoint(v->pt, pt);
	if (dist < distMin) {
	  distMin = dist;
	  vMin = v;
	}
	v = v->next;  
      }

      // Insert data[i][j] into vMin
      sumPixelDiff += (vMin->colour != data[i][j].trueCol);
      vMin->insertData(&data[i][j]);
    }
  }

  return;
}

void Simulation::removeDataFromVertices(Vertex * v, int first) {
  int i;
  Vertex * w;

  for (i=0;i<v->nNeigh;i++) {
    w = v->neigh[i];
    if (w->nData > 0) {
      if (first)
	w->removeFirstData();
      else
	w->removeData();
    }
  }

  return;
}
