#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <errno.h>
#include <assert.h>

#include "random.h"  //CHANGE$$ position...
#include "basic.h"
#include "point.h"
#include "vertex.h"
#include "simplex.h"
#include "edge.h"
#include "wells.h"
#include "delaunay.h"


Delaunay::Delaunay(int debug, int volAnn, int nMeanVertices,
		   RandomGenerator * random, double simRegion[]) {
  sList = new SimplexList(nMeanVertices, random, simRegion);
  vList = new VertexList(nMeanVertices);
  eList = new EdgeList(vList, nMeanVertices);
  nPoints = 0;

  Point pt = new double[DIM];
  mPt.pt = pt;
  lastCpy = new Vertex(pt, 0, 0);
  last = NULL;

  int i;
  for (i=0;i<2*DIM;i++)
    region[i] = simRegion[i];

  debugCheck = debug;

  volumeAnnealing = volAnn;

  imageData = 0;

  nSumSearch = 0;
  nCallSearch = 0;

  nFlipEdges[0][0] = 0;
  nFlipEdges[0][1] = 0;
  nFlipEdges[1][0] = 0;
  nFlipEdges[1][1] = 0;

  nFlipTriangles[0][0] = 0;
  nFlipTriangles[0][1] = 0;
  nFlipTriangles[1][0] = 0;
  nFlipTriangles[1][1] = 0;

  it = -1;

  mark = 0;

  gridDim[0] = 1;
  for (i=1; i<NGRIDDIM;i++)
    gridDim[i] = 2*gridDim[i-1];

  for (i=0; i<NGRIDDIM;i++)
  {
    int n = gridDim[i];
    if (DIM == 3)
      n *= n*n;
    else if (DIM == 2)
      n *= n;
    nSimpArray[i] = new int[n];
    posSimp[i] = new int[n];
    posSimpInv[i] = new int[n];
    simpArray[i] = new Simplex **[n];
    for (int j=0; j<n;j++)
    {
      simpArray[i][j] = new Simplex * [2];

      nSimpArray[i][j] = 0;
      simpArray[i][j][0] = NULL;
      simpArray[i][j][1] = NULL;

    }
    setPosSimp(posSimp[i], posSimpInv[i], i);
  }

  if (DIM == 3)
  {
    for (i=0;i<DIM+1;i++)
      for (int j=0;j<DIM;j++)
	perm[i][j] = Simplex::perm[i][j];

    permT[0][0] = 0;
    permT[0][1] = 1;
    permT[0][2] = 2;
    permT[0][3] = 3;

    permT[1][0] = 0;
    permT[1][1] = 2;
    permT[1][2] = 1;
    permT[1][3] = 4;

    indexT[0] = 4;
    indexT[1] = 3;

    indexOldT[0][0] = 1;
    indexOldT[0][1] = 2;
    indexOldT[0][2] = 0;

    indexOldT[1][0] = 1;
    indexOldT[1][1] = 0;
    indexOldT[1][2] = 2;


    permE[0][0] = 0;
    permE[0][1] = 2;
    permE[0][2] = 3;
    permE[0][3] = 4;

    permE[1][0] = 0;
    permE[1][1] = 3;
    permE[1][2] = 1;
    permE[1][3] = 4;

    permE[2][0] = 1;
    permE[2][1] = 2;
    permE[2][2] = 4;
    permE[2][3] = 3;

    indexETop[0][0] = 3;
    indexETop[0][1] = 2;

    indexETop[1][0] = 3;
    indexETop[1][1] = 1;

    indexETop[2][0] = 2;
    indexETop[2][1] = 3;

    indexNeigh[0][0] = 1;
    indexNeigh[0][1] = 1;

    indexNeigh[1][0] = 2;
    indexNeigh[1][1] = 2;

    indexNeigh[2][0] = 0;
    indexNeigh[2][1] = 0;

    indexE[0][0] = 0;
    indexE[0][1] = 1;

    indexE[1][0] = 0;
    indexE[1][1] = 2;

    indexE[2][0] = 0;
    indexE[2][1] = 1;

    indexNewNeigh[0][0] = 2;
    indexNewNeigh[0][1] = 1;

    indexNewNeigh[1][0] = 2;
    indexNewNeigh[1][1] = 0;

    indexNewNeigh[2][0] = 0;
    indexNewNeigh[2][1] = 1;
  }
  else if (DIM == 2)
  {
    perm[0][0] = 1;
    perm[0][1] = 2;

    perm[1][0] = 2;
    perm[1][1] = 0;

    perm[2][0] = 0;
    perm[2][1] = 1;

    permE[0][0] = 0;
    permE[0][1] = 3;
    permE[0][2] = 2;

    permE[1][0] = 1;
    permE[1][1] = 2;
    permE[1][2] = 3;

    indexETop[0][0] = 1;
    indexETop[0][1] = 2;

    indexETop[1][0] = 2;
    indexETop[1][1] = 1;

    indexNeigh[0][0] = 1;
    indexNeigh[0][1] = 1;

    indexNeigh[1][0] = 0;
    indexNeigh[1][1] = 0;

    indexE[0][0] = 0;

    indexE[1][0] = 0;

    indexNewNeigh[0][0] = 1;

    indexNewNeigh[1][0] = 0;
  }
}

Delaunay::~Delaunay() {
  delete eList;
  delete sList;
  delete vList;

  lastCpy->nSegment = 0; // NBNB Et lite hack... for at lastCpy->segment ikke skal deallokeres
  delete lastCpy;
  delete [] mPt.pt;

  for (int i=0; i<NGRIDDIM;i++) {
    int n = gridDim[i];
    if (DIM == 3)
      n *= n*n;
    else if (DIM == 2)
      n *= n;

    delete [] nSimpArray[i];
    delete [] posSimp[i];
    delete [] posSimpInv[i];
    for (int j=0; j<n;j++)
    {
      delete [] simpArray[i][j];
    }
    delete [] simpArray[i];
  }

}

void Delaunay::setPosSimp(int * posS, int * posSInv, int level)
{
  int * dim1 = new int[level+1]; // One too much but avoids new int[0]...
  int * dim2 = new int[DIM];
  int * dim3 = new int[level+1]; // One too much but avoids new int[0]...
  int * dim4 = new int[DIM];

  int dimFactor;
  if (DIM == 2)
    dimFactor = 4;
  else
    dimFactor = 8;
  int i;
  int n = 1;
  int factor=1;
  for (i=level-1;i>=0;i--) { // 64, 8, 1
    dim1[i] = n;
    n *= dimFactor;
    factor *=2;
  }
  n = 1;
  for (i=DIM-1;i>=0;i--) {//4, 2, 1
    dim2[i] = n;
    n *= 2;
  }
  n = 1;
  for (i=level-1;i>=0;i--) { // 4, 2, 1
    dim3[i] = n;
    n *= 2;
  }
  n = 1;
  for (i=DIM-1;i>=0;i--) {//64, 8, 1 OR 16, 4, 1 OR 4, 2, 1
    dim4[i] = n;
    n *= factor;
  }

  int N = 1;
  for (i=0;i<level;i++)
    N *= dimFactor;

  for (i=0;i<N;i++) {
    int sumPos[DIM];

    int j;
    for (j=0;j<DIM;j++)
      sumPos[j] = 0;

    int rest = i;
    for (j=0;j<level;j++) {
      int ind = (int) (rest / dim1[j]); // Index of grid block at level j
      rest = rest % dim1[j];
      int pos[DIM];
      
      int subrest = ind;
      for (int k=0; k<DIM;k++) {
	pos[k] = (int) (subrest/dim2[k]);
	subrest = subrest % dim2[k];
	sumPos[k] += pos[k] * dim3[j];
      }
    }
    int index = 0;
    for (int k=0; k<DIM;k++)
      index += sumPos[k] * dim4[k]; // (sumPos[0], ..., sumPos[DIM]) grid index

    posS[index] = i;
    posSInv[i] = index;
  }

  delete [] dim1;
  delete [] dim2;
  delete [] dim3;
  delete [] dim4;
}

void Delaunay::firstSimplex(mPoint * t, Wells * wells)
{
  Vertex * v[DIM+1];

  nPoints = DIM+1;

  Simplex * s = sList->New();

  int i;
  for (i=0;i<DIM+1;i++)
    v[i] = vList->insert(t[i], it);

  s->initialise(v, 0);

  s->findSphere();

  insertSimp(s);

  for (i=0;i<DIM+1;i++)
    v[i]->insertNeigh(mark);

  if (wells) {
    v[0]->initialiseSegments(wells);
    for (i=1; i<DIM+1;i++) {
      insertData(v[i], wells);
    }
  }
  
}

void Delaunay::checkSearchTable()
{
  for (int level=0;level<NGRIDDIM;level++) {
    for (int i=0;i<gridDim[level];i++) {
      Simplex * s = simpArray[level][i][0];
      if (s)
	assert(!s->removed);
      s = simpArray[level][i][1];
      if (s)
	assert(!s->removed);
    }
  }
}

int Delaunay::findStartIndex(int lowerLevel, int upperLevel, int posUpper)
{
  int n;

  if (DIM == 3)
  {
    n = 8 * (upperLevel - lowerLevel);
  }
  else if (DIM == 2)
  {
    n = 4 * (upperLevel - lowerLevel);
  }

  int pos = posSimp[upperLevel][posUpper];

  int start = n * ((int) pos / n);

  return start;
}

int Delaunay::insert(mPoint & p, Wells * wells) {
  nPoints++;
 
  last = vList->insert(p, it);

  Simplex * s = initialSearch(last);

  assert(s);
  assert(!s->removed);

  nSearch = 0;
  Simplex * sFirst = search(s); // Find tetraheder that contain last
  nSumSearch += nSearch;
  nCallSearch++;

  if (0) {
    printf("# search: %d\n", nSearch);
    printf("Average search: %6.4f\n", meanNSearch());
  }
  assert(sFirst);
  assert(!sFirst->removed);
  
  connect(sFirst); // Triangelize

  int error = flip(); // Restore Delaunay triangulation by flipping operations

  if (error)
    return 1;

  if (wells)
    insertData(wells);

  if (imageData)
    insertImage();

  vList->insertMarg(last);

  if (volumeAnnealing)
    volFractionAdd();

  if (debugCheck)
    checkConsistency(wells);

  lastAction = ADD; //TO BE REMOVED?

  return 0;
}


int Delaunay::move(Vertex * v, mPoint & x, Wells * wells)
{
  int error = remove(v, wells);
  if (error)
    return 1;

  //  vList->checkConsistency();

  error = insert(x, wells);
  if (error)
    return 1;

  //  vList->checkConsistency();

  lastAction = MOVE;

  return 0;
}

void Delaunay::insertData(Wells * wells)
{
  insertData(last, wells);
}


void Delaunay::insertData(Vertex * vNew, Wells * wells)
{
  int nNeigh = vNew->nNeigh;
  //  int newSegment[3];

  int i;
  for (i = 0;i<nNeigh;i++) {
    Vertex * v = vNew->neigh[i];

    int n = v->nSegment;
    
    for (int j=0;j<n;j++) {      
      // Check that vNew intersects this well segment
      wells->intersect(vNew, v, v->segment[j]);
    }
  }
  vNew->clearSegment();
  for (i = 0;i<nNeigh;i++) {
    Vertex * v = vNew->neigh[i];
    v->clearSegment();
  }
}


void Delaunay::removeData(Wells * wells)
{
  Vertex * vOld = last;
  Vertex * vOldCpy = lastCpy;
  int nNeigh = vOldCpy->nNeigh;
  Vertex ** neigh = vOldCpy->neigh;

  int fullSegment[3];
  
  int n = vOld->nSegment;

  int i;
  for (i=0;i<n;i++) {
    int * segment = (int *) vOld->segment[i];

    fullSegment[0] = segment[0];
    fullSegment[1] = segment[1];
    fullSegment[2] = wells->findIndex(fullSegment, vOld);

    wells->updateSegment(fullSegment, vOld, neigh, nNeigh);
  }
  vOld->clearSegment();
  for (i=0;i<nNeigh;i++)
    neigh[i]->clearSegment();
}

void Delaunay::connect(Simplex * sOld)
{
  Simplex * sNew[DIM+1];

  int i;
  for (i=0;i<DIM+1;i++) {
    sNew[i] = sList->New();
  }

  // Set points for the new simplices
  // Update the neighbour relations between simplices
  //perm[0][] = 1,3,2
  //perm[1][] = 0,2,3
  //perm[2][] = 0,3,1
  //perm[3][] = 0,1,2
  for (i=0;i<DIM+1;i++)
  {
    for (int j=0;j<DIM;j++)
    {
      sNew[i]->v[j] = sOld->v[perm[i][j]];
      sNew[i]->neigh[j] = sNew[perm[i][j]];
    }
    sNew[i]->v[DIM] = last;
    sNew[i]->neigh[DIM] = sOld->neigh[i];
    Simplex * t = sOld->neigh[i];
    if (t)
      t->setNeigh(sOld, sNew[i]); //NBNBNB How to this efficiently?

    // Compute centre points and radii of surrounding spheres
    sNew[i]->initialise(it);
    sNew[i]->findSphere();

    insertSimp(sNew[i]);
  }

  last->insertNeigh(sOld->getVertex(), mark);

  removeSimp(sOld);
  sList->Free(sOld);
}

void Delaunay::disConnect()
{
  Simplex * sNew = sList->New();

  int nSimp;
  Simplex ** sOld;
  last->getSimp(sOld, nSimp);

  // Find vertices of sNew - correctly oriented
  int index = sOld[0]->getNeigh(last);
  for (int j=0;j<DIM;j++) // First three indices
    sNew->v[j] = sOld[0]->v[perm[index][j]];

  index = sOld[1]->getRemainingVertex(sOld[0]->v);
  sNew->v[DIM] = sOld[1]->getVertex(index);

  int i;
  for (i=0;i<nSimp;i++) {
    // Find index that contains last
    Simplex * t = sOld[i]->getNeigh(last, index);
    int newIndex = sNew->getRemainingVertex(sOld[i]->v, index);
    sNew->neigh[newIndex] = t;
    if (t) {
      int tIndex = t->getRemainingVertex(sOld[i]->v, index);
      t->neigh[tIndex] = sNew;
    }
  }

  sNew->initialise(it);
  sNew->findSphere();
  insertSimp(sNew);

  last->removeNeigh();

  for (i=0;i<nSimp;i++) {
    removeSimp(sOld[i]);
    sList->Free(sOld[i]);
  }

  delete [] sOld;
}


void Delaunay::insertSimpInSearch(Simplex * s)
{
  Point p = s->getMidPoint();
  int pos[NGRIDDIM];

#ifdef PRINTX
  printf("pos: ");
#endif
  for (int i=0;i<NGRIDDIM;i++)
  {
    pos[i] = position(p, i);
#ifdef PRINTX
    printf("%d ", pos[i]);
#endif
  }
#ifdef PRINTX
  printf("\n");
  fflush(stdout);
#endif

  insertSimpInSearch(s, pos, NGRIDDIM);
}

void Delaunay::insertSimpInSearch(Simplex * s, int * pos, int n)
{
  int m = pos[n-1];

  if (nSimpArray[n-1][m] == 1)
  {
    simpArray[n-1][m][nSimpArray[n-1][m]] = s;
    nSimpArray[n-1][m]++;
    s->inserted = 1;
  }
  else
  {
    int index = n-1;
    m = pos[index];
    while (index >= 0 && nSimpArray[index][m] == 0)
    {
      simpArray[index][m][nSimpArray[index][m]] = s;
      nSimpArray[index][m]++;
      index--;
      if (index>=0)
	m = pos[index];
    }
    if (index<n-1)
      s->inserted = 1;
  }

}

void Delaunay::removeSimpFromSearch(Simplex * s, int * pos, int n)
{
  int found;
  int m = pos[n-1];

  int index = n-1;
  do
  {
    m = pos[index];
    if (simpArray[index][m][0] == s) {
      simpArray[index][m][0] = simpArray[index][m][1];
      simpArray[index][m][1] = NULL;
      nSimpArray[index][m]--;
      found = 1;
      index--;
    }
    else if (simpArray[index][m][1] == s) {
      simpArray[index][m][1] = NULL;
      nSimpArray[index][m]--;
      found = 1;
      index--;
    }
    else
      found = 0;
  }  while (index >= 0 && found);

  // If removed at an upper(most?) level, then re-instore if possible
  if (index < n-2)
  {
    int p;
    if (DIM == 2)
      p = 4;
    else if (DIM == 3)
      p = 8;
    for (int k=n-2; k>=index+1; k--)
    {
      int start = findStartIndex(k, k+1, pos[k+1]);
      
      Simplex * sFound = NULL;
      for (int l=start; l<start+p; l++) {
	int I = posSimpInv[k+1][l];
	if (nSimpArray[k+1][I] > 0) {
	  sFound = simpArray[k+1][I][0];
	  break;
	}
      }
      if (sFound) {
	simpArray[k][pos[k]][0] = sFound;
	nSimpArray[k][pos[k]]++;
      }
    }
    
  }
}


int Delaunay::flip2D()
{
  int i;
  // Keep a stack of incident simplices
  static Simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  int nSimp;
  Simplex ** simp;

  Simplex * sNew[2];

  Simplex * s;

  int nRemoved = 0;
  int nMaxRemoved = 1000;  //NBNB Hope this is enough...
  Simplex ** sRemoved = new Simplex *[nMaxRemoved];
  
  last->getSimp(simp, nSimp);

  Point p = last->getPt();

  initStackMemory();
  for (i=0;i<nSimp;i++)
    push(simp[i]);

  delete [] simp;

  mark++;
 
  while (tms) {
    pop(s);

    if (!s->removed) {
      // Check if s is locally delaunay

      int index;
      Simplex * t = s->getNeigh(last, index);
      if (t) {
	if (!t->isDelaunay(last)) {
	  if (t->convex(s, p)) {
	    flipTriangle(s, t, sNew, sRemoved, nRemoved);
	    if (sNew[0]->isMember(last)) {
	      push(sNew[0]);
	    }
	    if (sNew[1]->isMember(last)) {
	      push(sNew[1]);
	    }
	  }
	}
      }
      s->visit = mark;
    }
  }

  assert(nRemoved <= nMaxRemoved);

  for (i=0;i<nRemoved;i++)
    sList->Free(sRemoved[i]);

  delete [] sRemoved;

  return 0;
}

int Delaunay::flip3D()
{
  int i;
  // Keep a stack of incident simplices
  static Simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  int nSimp;
  Simplex ** simp;

  Simplex * sNew[3];

  Simplex * s;

  int nRemoved = 0;
  int nMaxRemoved = 1000;  //NBNB Hope this is enough...
  Simplex ** sRemoved = new Simplex *[nMaxRemoved];
  
  last->getSimp(simp, nSimp);

  Point p = last->getPt();

  initStackMemory();
  for (i=0;i<nSimp;i++)
    push(simp[i]);

  delete [] simp;

  mark++;
 
  while (tms) {
    pop(s);

    if (!s->removed) {
      // Check if s is locally delaunay

      int index;
      Simplex * t = s->getNeigh(last, index);
      if (t) {
	if (!t->isDelaunay(last)) {
	  Edge * edge = NULL;
	  int found = 0;
	  for (i=0; i<NEDGE_T; i++) {
	    edge = s->getEdge(index, i);
	    if (edge->nSimp == 3 && !edge->boundary) {
	      found = 1;
	      break;
	    }
	  }
	  if (found) {
	    int flipped = flipEdge(edge, sNew, sRemoved, nRemoved);
	    if (flipped) {
	      if (sNew[0]->isMember(last)) {
		push(sNew[0]);
	      }
	      if (sNew[1]->isMember(last)) {
		push(sNew[1]);
	      }
	    }
	    else if (flipped == -1)
	      return 1; // error in flipEdge
	  }
	  else if (t->convex(s, p)) {
	    flipTriangle(s, t, sNew, sRemoved, nRemoved);
	    for (i=0;i<3;i++)
	      push(sNew[i]);
	  }
	}
      }
      s->visit = mark;
    }
  }

  assert(nRemoved <= nMaxRemoved);

  for (i=0;i<nRemoved;i++)
    sList->Free(sRemoved[i]);

  delete [] sRemoved;

  return 0;
}

void Delaunay::getNeighbourVertices(Edge * e, Vertex * neighV[], Simplex * sOld[])
{
  Vertex * vNew[2];
  int nNeighV = 0;

  assert(e->nSimp == 3);

  for (int iSimp=0;iSimp<2;iSimp++) {    
    e->neigh[iSimp]->getRemainingVertices(e->v, vNew);

    int jIndex = -1;
    int found[2];
    found[0] = 0;
    found[1] = 0;
    for (int i=0;i<nNeighV;i++) {
      for (int j=0;j<2;j++) {
	if (neighV[i] == vNew[j]) {
	  assert(found[i] == 0);
	  found[i] = 1; // nNeighV = 0 or 2 ...
	  jIndex = (j+1) % 2;
	}
      }
    }
    if (iSimp == 0) {
      for (int j=0;j<2;j++) {
	if (!found[j])
	  neighV[nNeighV++] = vNew[j];
      }
      sOld[iSimp] = e->neigh[iSimp];
    }
    else {
      neighV[nNeighV++] = vNew[jIndex];
      if (found[0]) {
	sOld[1] = e->neigh[2];
	sOld[2] = e->neigh[1];
      }
      else {
	sOld[1] = e->neigh[1];
	sOld[2] = e->neigh[2];
      }
    }
  }

}


inline int Delaunay::oriented(Vertex * v[]) //NBNB Same function as in Simplex
{
  int vol = 0;

  Vertex * w[DIM];
  for (int i=0;i<DIM;i++)
    w[i] = v[perm[0][i]];

  Point pt = v[perm[0][DIM]]->getPt();
  
  vol = volumeSign(w, pt);

  return (vol<0);
}


int Delaunay::volumeSign(Vertex * v[], Point newP)
{
  int volSign;
  double vol;

  if (DIM == 2) {
    double ba0 = (*v[1])[0] - (*v[0])[0];
    double ba1 = (*v[1])[1] - (*v[0])[1];
    double ca0 = newP[0] - (*v[0])[0];
    double ca1 = newP[1] - (*v[0])[1];

    vol = ba0*ca1 - ca0*ba1;
  }
  else {
    double ad0 = (*v[0])[0] - newP[0];
    double ad1 = (*v[0])[1] - newP[1];
    double ad2 = (*v[0])[2] - newP[2];

    double bd0 = (*v[1])[0] - newP[0];
    double bd1 = (*v[1])[1] - newP[1];
    double bd2 = (*v[1])[2] - newP[2];

    double cd0 = (*v[2])[0] - newP[0];
    double cd1 = (*v[2])[1] - newP[1];
    double cd2 = (*v[2])[2] - newP[2];

    vol =  - ad2 * bd1 * cd0 + ad1 * bd2 * cd0;
    vol +=   ad2 * bd0 * cd1 - ad0 * bd2 * cd1;
    vol += - ad1 * bd0 * cd2 + ad0 * bd1 * cd2;
  }

  if (vol > 0.0)
    volSign = 1;
  else if (vol < 0.0)
    volSign = -1;
  else
  {
    volSign = 0;
    printf("WARNING(volumeSign): volSign == 0\n");
  }

  return volSign;
}

int Delaunay::flipEdge(Edge * edge, Simplex ** sNew, Simplex * sRemoved[],
		       int & nRemoved, int flipType)
{
  int i;
  Vertex * v[DIM+2];
  Simplex * sOld[3];
  int top;

  getNeighbourVertices(edge, v, sOld);
  v[3] = edge->v[0];
  v[4] = edge->v[1];
  
  int volSign[2];
  volSign[0] = volumeSign(v, v[3]->getPt());
  volSign[1] = -volumeSign(v, v[4]->getPt()); //NBNB Litt grisete progr..
  if (flipType == FLIP_REMOVE) {
    // Check if the edge is transformable
    if (volSign[0] != volSign[1]) {
      nFlipEdges[0][FLIP_REMOVE]++;
      return 0;
    }
  }
  assert(volSign[0] == volSign[1]);

  if (volSign[0] > 0) {
    Vertex * tmp = v[3];
    v[3] = v[4];
    v[4] = tmp;
  }

  for (i=0;i<2;i++) // Create two new simplices...
    sNew[i] = sList->New();

  // Assumes that simplex sOld[i] contains v[i], v[i+1] and v[DIM], v[DIM+1]
  // permT[0][] = 0,1,2,3 indexT[0] = 4
  // permT[1][] = 0,2,1,4 indexT[1] = 3
  // indexOldT[0][] = 1,2,0  indexOldT[1][] = 1,0,2

  Simplex * t[2][DIM];

  for (i=0;i<2;i++)
  {
    int j;
    // Set points and neighbour relations....
    for (j=0;j<DIM+1;j++)
      sNew[i]->v[j] = v[permT[i][j]];

    for (j=0;j<DIM;j++) {
      Simplex * s = sOld[indexOldT[i][j]];
      t[i][j] = s->getNeigh(v[indexT[i]], top);
      sNew[i]->neigh[j] = t[i][j];
    }
    sNew[i]->neigh[DIM] = sNew[(i+1) % 2]; 

    int error = !sNew[i]->oriented(perm, 1);
    if (error) { //CHANGE$$
      printf("Error(Delaunay::flipEdge): sNew not oriented\n");
      return -1; // error code
    }

    // Compute centre points and radii of surrounding spheres
    sNew[i]->initialise(it);
    sNew[i]->findSphere();

  }

  int delaunay = 1;

  if (flipType == FLIP_REMOVE) {
    Simplex * s;
    if (sNew[0]->isMember(last)) {
      s = sNew[1];
      sNew[0]->visit = mark;
    }
    else {
      s = sNew[0];
      sNew[1]->visit = mark;
    }

    assert(!s->isMember(last));

    delaunay = s->isDelaunay(last->neigh, last->nNeigh);
#ifdef OUT
    for (i=0;i<DIM+1;i++) {
      Simplex * ss = s->neigh[i];
      if (ss && ss->visit != mark)
	if (!ss->isDelaunay(s->v[i])) {
	  delaunay = 0;
	  break;
	}
    }
#endif

  }

  nFlipEdges[delaunay][flipType]++;

  if (delaunay) {
    // First set neighbour relations of incident simplices
    for (i=0;i<2;i++)
      for (int j=0;j<DIM;j++)
	if (t[i][j]) {
	  Simplex * s = sOld[indexOldT[i][j]];
	  t[i][j]->setNeigh(s, sNew[i]);
	}

    for (i=0;i<2;i++)
      insertSimp(sNew[i]);

    edge->v[0]->removeNeigh(edge->v[1]);
    edge->v[1]->removeNeigh(edge->v[0]);

    for (i=0;i<3;i++) {
      removeSimp(sOld[i]);
      sRemoved[nRemoved++] = sOld[i];
    }

  }
  else {
    sList->Free(sNew[0]);
    sList->Free(sNew[1]);
  }

  return delaunay;
}

int Delaunay::flipTriangle(Simplex * s, Simplex * t, Simplex ** sNew,
			   Simplex * sRemoved[], int & nRemoved,
			   int flipType)
{
  int i;
  Vertex * v[DIM+2];

  Simplex * sOld[] = {s, t};

  const int nNew = DIM;
  const int nOld = 2;
  const int nRem = DIM-1;
  int top[nOld]; //for sOld[0], sOld[1]
  int indexOld[nOld][DIM];
#ifdef DIM2D
  int permO[DIM+1][DIM] = {{1,2},{2,0},{0,1}};
#else
  int permO[DIM+1][DIM] = {{1,3,2},{0,2,3},{0,3,1},{0,1,2}}; // For orientation
#endif
  for (int iOld=0;iOld<2;iOld++)
    top[iOld] = sOld[iOld]->getNeigh(sOld[(iOld+1) % 2]);

  // Find vertices of edge to be flipped....

  for (i=0;i<DIM;i++) {
    v[i] = s->v[permO[top[0]][i]];
    indexOld[0][i] = permO[top[0]][i];
  }
  v[DIM] = s->v[top[0]];
  v[DIM+1] = t->v[top[1]];

  for (i=0;i<DIM;i++) {
    int j;
    for (j=0;j<DIM+1;j++) {
      if (v[i] == t->v[j])
	break;
    }
    assert(j<DIM+1);
    indexOld[1][i] = j;
  }

  // 2D
  // Assumes that simplex sOld[0], sOld[1] contains 012, 013
  // permE[0][] = 0,3,2 indexETop[0][] = 1,2 indexNeigh[0][] = 1,1
  // permE[1][] = 1,2,3 indexETop[1][] = 2,1 indexNeigh[1][] = 0.0
  // indexE[0][] = 0  indexNewNeigh[0][] = 1
  // indexE[1][] = 0  indexNewNeigh[1][] = 0
 
  // 3D
  // Assumes that simplex sOld[0], sOld[1] contains 0123, 0124
  // permE[0][] = 0,2,3,4 indexETop[0][] = 3,2 indexNeigh[0][] = 1,1
  // permE[1][] = 0,3,1,4 indexETop[1][] = 3,1 indexNeigh[1][] = 2,2
  // permE[2][] = 1,2,4,3 indexETop[2][] = 2,3 indexNeigh[2][] = 0,0
  // indexE[0][] = 0,1  indexNewNeigh[0][] = 2,1
  // indexE[1][] = 0,2  indexNewNeigh[1][] = 2,0
  // indexE[2][] = 0,1  indexNewNeigh[2][] = 0,1


  for (i=0;i<nNew;i++) // Create three new simplices...
    sNew[i] = sList->New();


  Simplex * tt[nNew][nOld];

  for (i=0;i<nNew;i++)
  {
    int j;
    // Set points and neighbour relations....
    for (j=0;j<DIM+1;j++)
      sNew[i]->v[j] = v[permE[i][j]];

    for (j=0;j<nOld;j++) { // Neighbours of sOld
      Simplex * ss = sOld[j];
      tt[i][j] = ss->neigh[indexOld[j][indexNeigh[i][j]]];
      sNew[i]->neigh[indexETop[i][j]] = tt[i][j];

    }
    for (j=0;j<nRem;j++) { // Neighbours of sNew
      sNew[i]->neigh[indexE[i][j]] = sNew[indexNewNeigh[i][j]];
    }

    // Compute centre points and radii of surrounding spheres
    sNew[i]->initialise(it);
    sNew[i]->findSphere();
  }
  
  int delaunay = 1;

  if (flipType == FLIP_REMOVE) {
    for (i=0;i<nNew;i++)
      if (!sNew[i]->isMember(last))
	break;
    
    assert(i<nNew);

    delaunay = sNew[i]->isDelaunay(last->neigh, last->nNeigh);
  }

  nFlipTriangles[delaunay][flipType]++;

  if (delaunay) {
    for (i=0;i<nNew;i++) {
      insertSimp(sNew[i]);

      for (int j=0;j<nOld;j++) { // Neighbours of sOld
	Simplex * ss = sOld[j];
	if (tt[i][j])
	  tt[i][j]->setNeigh(ss, sNew[i]);
      }
    }
    for (i=0;i<nOld;i++) {
      removeSimp(sOld[i]);
      sRemoved[nRemoved++] = sOld[i];
    }
    
    v[DIM]->insertNeigh(v[DIM+1]);
    v[DIM+1]->insertNeigh(v[DIM]);
    if (DIM == 2) {
      v[0]->removeNeigh(v[1]);
      v[1]->removeNeigh(v[0]);
    }
    if (flipType == FLIP)
      assert(v[DIM] == last);
  }
  else {
    for (i=0;i<nNew;i++) {
      sList->Free(sNew[i]);
    }
  }


  return delaunay;
}

int Delaunay::flipTriangle_old(Simplex * s, Simplex * t, Simplex ** sNew,
			       Simplex * sRemoved[], int & nRemoved,
			       int flipType)
{
  int i;
  Vertex * v[DIM+2];

  Simplex * sOld[] = {s, t};

  int top[2]; //for sOld[0], sOld[1]
  int indexOld[2][DIM];
#ifdef DIM2D
  int permO[DIM+1][DIM];
  assert(0); // NBNB MISSING
#else
  int permO[DIM+1][DIM] = {{1,3,2},{0,2,3},{0,3,1},{0,1,2}}; // For orientation
#endif
  for (int iOld=0;iOld<2;iOld++)
    top[iOld] = sOld[iOld]->getNeigh(sOld[(iOld+1) % 2]);


  for (i=0;i<DIM;i++) {
    v[i] = s->v[permO[top[0]][i]];
    indexOld[0][i] = permO[top[0]][i];
  }
  v[DIM] = s->v[top[0]];
  v[DIM+1] = t->v[top[1]];

  for (i=0;i<DIM;i++) {
    int j;
    for (j=0;j<DIM+1;j++) {
      if (v[i] == t->v[j])
	break;
    }
    assert(j<DIM+1);
    indexOld[1][i] = j;
  }

  
  // Assumes that simplex sOld[0], sOld[1] contains 0123, 0124
  // permE[0][] = 0,2,3,4 indexETop[0][] = 3,2 indexNeigh[0][] = 1,1
  // permE[1][] = 0,3,1,4 indexETop[1][] = 3,1 indexNeigh[1][] = 2,2
  // permE[2][] = 1,2,4,3 indexETop[2][] = 2,3 indexNeigh[2][] = 0,0
  // indexE[0][] = 0,1  indexNewNeigh[0][] = 2,1
  // indexE[1][] = 0,2  indexNewNeigh[1][] = 2,0
  // indexE[2][] = 0,1  indexNewNeigh[2][] = 0,1


  for (i=0;i<3;i++) // Create three new simplices...
    sNew[i] = sList->New();


  Simplex * tt[3][2];

  for (i=0;i<3;i++)
  {
    int j;
    // Set points and neighbour relations....
    for (j=0;j<DIM+1;j++)
      sNew[i]->v[j] = v[permE[i][j]];

    for (j=0;j<2;j++) { // Neighbours of sOld
      Simplex * ss = sOld[j];
      tt[i][j] = ss->neigh[indexOld[j][indexNeigh[i][j]]];
      sNew[i]->neigh[indexETop[i][j]] = tt[i][j];

    }
    for (j=0;j<2;j++) { // Neighbours of sNew
      sNew[i]->neigh[indexE[i][j]] = sNew[indexNewNeigh[i][j]];
    }

    // Compute centre points and radii of surrounding spheres
    sNew[i]->initialise(it);
    sNew[i]->findSphere();
  }
  
  int delaunay = 1;

  if (flipType == FLIP_REMOVE) {
    for (i=0;i<3;i++)
      if (!sNew[i]->isMember(last))
	break;
    
    assert(i<3);

    delaunay = sNew[i]->isDelaunay(last->neigh, last->nNeigh);
  }

  nFlipTriangles[delaunay][flipType]++;

  if (delaunay) {
    for (i=0;i<3;i++) {
      insertSimp(sNew[i]);

      for (int j=0;j<2;j++) { // Neighbours of sOld
	Simplex * ss = sOld[j];
	if (tt[i][j])
	  tt[i][j]->setNeigh(ss, sNew[i]);
      }
    }
    for (i=0;i<2;i++) {
      removeSimp(sOld[i]);
      sRemoved[nRemoved++] = sOld[i];
    }

    v[DIM]->insertNeigh(v[DIM+1]);
    v[DIM+1]->insertNeigh(v[DIM]);
    if (flipType == FLIP)
      assert(v[DIM] == last);
  }
  else {
    for (i=0;i<3;i++) {
      sList->Free(sNew[i]);
    }
  }


  return delaunay;
}


Simplex * Delaunay::search(Simplex * sFirst)
{
  Simplex * s;
  Simplex * sFound = NULL;
  Point pt = last->getPt();

  // Keep a stack of incident simplices
  static Simplex **st;
  static long ss = (long) MAXDIM;
  long tms = 0;

  mark++;

  initStackMemory();

  push(sFirst);

 
  while (tms) {
    pop(s);

    if (!s->isVisited(mark)) {

      s->visit = mark;

      nSearch++;

      if (0) {
	s->print();

	s->printDistance(pt);
      }

      if (s->inside(pt)) {
	sFound = s;
	break;
      }
      else {
	int index = -1;
	double minDist = DBL_MAX;
	for (int i=0;i<DIM+1;i++) {
	  if (s->neigh[i] && s->neigh[i]->visit != mark) {
	    double dist = distPoint(pt, s->neigh[i]->midPoint);
	    if (dist < minDist) {
	      minDist = dist;
	      index = i;
	    }
	  }
	}


	if (index>=0) {
	  // Then search the other simplices
	  for (int i=0;i<DIM+1;i++) {
	    if (s->neigh[i] && s->neigh[i]->visit != mark && i != index) {
	      push(s->neigh[i]);
	    }
	  }
	  // Search first the simplex nearest to last
	  push(s->neigh[index]);
	}
      }
    }    
  }

  assert(sFound);

  return sFound;
}

void Delaunay::checkConsistency(Wells * wells) {
  checkSearchTable();
  checkMarginalPosterior();
  vList->checkConsistency();
  sList->checkConsistency(vList, eList, perm);
  sList->checkIfDelaunay(vList);
  if (wells)
    wells->checkConsistency(vList);
}


Simplex * Delaunay::initialSearch(Vertex * v)
{
  Simplex * s;
  // Find initial simplex near 'v'
  int n = NGRIDDIM-1;
  do {
    s = findSimplex(v, n);
    n--;
  }
  while (!s && n>=0);

  assert(s);

  if (0) {
    printf("Initial search: Level %d, s->midPoint: ", n+1);
    for (int i=0;i<DIM;i++)
      printf("%6.4f ", s->midPoint[i]);
    printf("\n");
  }

  return s;
}

Simplex * Delaunay::findSimplex(Vertex * v, int n)
{
  int pos = position(v->getPt(), n);
  
  return simpArray[n][pos][0];
}

int Delaunay::position(Point pt, int n)
{
  int i;
  int pos[DIM];

  for (i=0;i<DIM;i++)
  {
    pos[i] = (int) ((pt[i]/region[2*i+1])*gridDim[n]);
    if (pos[i]<0)
      pos[i] = 0;
    else if (pos[i]>=gridDim[n])
      pos[i] = gridDim[n]-1;
  }
  
  int index = pos[0];
  int mult = 1;
  for (i=1; i<DIM; i++) {
    mult *= gridDim[n];
    index += mult*pos[i];
  }

  return index;
}

int Delaunay::remove(Vertex * v, Wells * wells) {
  nPoints--;

  last = v;

  last->copy(lastCpy);
  lastAction = REMOVE;

  int error = flipRemove(); // Reduce number of neighbouring simplices to DIM+1
  if (error) // Error in flipRemove(), must restore the delaunay triangulation
    return 1;

  disConnect(); // Remove neighbouring simplices and add one

  if (wells)
    removeData(wells);

  if (imageData)
    removeImage();

  vList->removeMarg(lastCpy);

  if (volumeAnnealing)
    volFractionRemove();

  vList->remove(last);

  if (debugCheck)
    checkConsistency(wells);


  return 0;
}

int Delaunay::flipRemove2D()
{
  int i;
  // Keep a stack of incident simplices
  static Simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  int nSimp;
  Simplex ** simp;

  const int nNew = 2;
  Simplex * sNew[nNew];

  Simplex * s;

  int nRemoved = 0;
  int nMaxRemoved = 1000;  //NBNB Hope this is enough...
  Simplex ** sRemoved = new Simplex *[nMaxRemoved];

  last->getSimp(simp, nSimp);

  mark++; // Mark all neighbouring simplices of last

  initStackMemory();
  for (i=0;i<nSimp;i++) {
    push(simp[i]);
    simp[i]->visit = mark;
  }

  delete [] simp;

  // Perform a serie of flipEdge and flipTriangle operations
  while (last->getnSimp()>3 && tms) { // OR last->nNeigh > 3
    pop(s);

    if (!s->removed) {

      int index = s->getNeigh(last);

      int flipped = 0;

      Simplex * t[DIM];
      int top[DIM];
      s->getNeigh(index, top, t);
      for (i=0; i<DIM; i++) {
	Point pt = s->getPt(top[i]);
	if (t[i]->convex(s, pt)) {
	  flipped = flipTriangle(s, t[i], sNew, sRemoved, nRemoved, FLIP_REMOVE);
	  if (flipped) {
	    for (int j=0;j<nNew;j++)
	      if (sNew[j]->isMember(last)) {
		sNew[j]->visit = mark;
		push(sNew[j]);
	      }
	    break;
	  }
	}
      }
    }
  }

  if (last->getnSimp() != 3) {
    printf("ERROR(Delaunay::flipRemove): last->getnSimp() != 3\n");
    last->printAll();
    last->printNeigh();
    last->printNeigh(vList);

    assert(last->getnSimp() == 3);
  }

  assert(nRemoved <= nMaxRemoved);

  for (i=0;i<nRemoved;i++)
    sList->Free(sRemoved[i]);

  delete [] sRemoved;

  return 0;
}

int Delaunay::flipRemove3D()
{
  int i;
  // Keep a stack of incident simplices
  static Simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  int nSimp;
  Simplex ** simp;

  Simplex * sNew[3];

  Simplex * s;

  int nRemoved = 0;
  int nMaxRemoved = 1000;  //NBNB Hope this is enough...
  Simplex ** sRemoved = new Simplex *[nMaxRemoved];

  last->getSimp(simp, nSimp);

  mark++; // Mark all neighbouring simplices of last

  initStackMemory();
  for (i=0;i<nSimp;i++) {
    push(simp[i]);
    simp[i]->visit = mark;
  }

  delete [] simp;

  // Perform a serie of flipEdge and flipTriangle operations
  while (last->getnSimp()>4 && tms) { // OR last->nNeigh > 4
    pop(s);

    if (!s->removed) {

      int index = s->getNeigh(last);

      Edge * edge = NULL;
      int flipped = 0;
      for (i=0; i<NEDGE_V; i++) {
	edge = s->getEdge2(index, i);
	assert(!edge->boundary);
	if (edge->nSimp == 3) {
	  flipped = flipEdge(edge, sNew, sRemoved, nRemoved, FLIP_REMOVE);

	  if (flipped == 1) {
	    // Kun de(n) som inneholder last
	    if (sNew[0]->isMember(last)) {
	      sNew[0]->visit = mark;
	      push(sNew[0]);
	    }
	    if (sNew[1]->isMember(last)) {
	      sNew[1]->visit = mark;
	      push(sNew[1]);
	    }
	  }
	  else if (flipped == -1) {
	    delete [] sRemoved;
	    return 1; // return error
	  }
	  break;
	}
      }

      if (!flipped) {
	Simplex * t[3];
	int top[3];
	s->getNeigh(index, top, t);
	for (i=0; i<3; i++) {
	  Point pt = s->getPt(top[i]);
	  if (t[i]->convex(s, pt)) {
	    flipped = flipTriangle(s, t[i], sNew, sRemoved, nRemoved, FLIP_REMOVE);
	    if (flipped) {
	      for (int j=0;j<3;j++)
		if (sNew[j]->isMember(last)) {
		  sNew[j]->visit = mark;
		  push(sNew[j]);
		}
	      break;
	    }
	  }
	}
      }
    }
  }

  if (last->getnSimp() != 4) {
    printf("ERROR(Delaunay::flipRemove): last->getnSimp() != 4\n");
    last->printAll();
    last->printNeigh();
    last->printNeigh(vList);

    delete [] sRemoved;
    return 1; // error
  }

  assert(nRemoved <= nMaxRemoved);

  for (i=0;i<nRemoved;i++)
    sList->Free(sRemoved[i]);

  delete [] sRemoved;

  return 0;
}

void Delaunay::rejectLastUpdate(Wells * wells)
{
  int action = lastAction;
  if (action == ADD) {
    //    removeData(wells);
    remove(last, wells);
    lastCpy->checkColDiff();  // Check if colDiff == colDiffNew
  }
  else if (action == REMOVE) {
    lastCpy->getMPt(&mPt);
    insert(mPt, wells);
    //    insertData(wells);
    last->checkColDiff();
    last->updateColDiff(); // NBNB Litt grisete programmering
  }
  else { // MOVE
    lastCpy->getMPt(&mPt);

    remove(last, wells); // NBNB Destroys lastCpy

    if (debugCheck)
      vList->checkConsistency();

    insert(mPt, wells);

    if (debugCheck)
      vList->checkConsistency();

    last->updateColDiff(); // NBNB Litt grisete programmering
  }
  if (debugCheck)
    vList->checkAcceptedConsistency(); // Check if colDiff == colDiffNew
}

void Delaunay::acceptLastUpdate()
{
  if (lastAction == ADD)
    last->updateColDiff();
  else if (lastAction == REMOVE)
    lastCpy->updateColDiff();
  else {
    lastCpy->updateColDiff();
    last->updateColDiff(); //NBNB Check if this work..
  }
  if (debugCheck)
    vList->checkAcceptedConsistency(); // Check if colDiff == colDiffNew
}

int Delaunay::sumFirstOrderColDiff(int action)
{
  int sum=0;

  if (nPoints > DIM+1)
  {
    if (action == ADD)
      sum = last->sumFirstOrderColDiff(action);
    else if (action == REMOVE)
      sum = -lastCpy->sumFirstOrderColDiff(action);
    else { // action == MOVE
      mark++;
      sum = lastCpy->sumFirstOrderColDiff(REMOVE, mark);
      sum += last->sumFirstOrderColDiff(ADD, mark);//NBNB Check if this work..
    }
  }

  if (sum % 2 != 0) {
    printf("ERROR(Delaunay::sumFirstOrderColDiff): sum %d\n", sum);
    assert(sum % 2 == 0);
  }
  
  sum /= 2;

  return sum;
}

void Delaunay::printLogInfo(FILE * logFile)
{
  fprintf(logFile, "# iterations %d\n\n", it+1);
  fprintf(logFile, "Total # flips %d\n",
	  nFlipTriangles[0][FLIP] + nFlipTriangles[0][FLIP_REMOVE] +
	  nFlipTriangles[1][FLIP] + nFlipTriangles[1][FLIP_REMOVE] +
	  nFlipEdges[0][FLIP] + nFlipEdges[0][FLIP_REMOVE] +
	  nFlipEdges[1][FLIP] + nFlipEdges[1][FLIP_REMOVE]);
  fprintf(logFile, "Total # accepted flips %d\n",
	  nFlipTriangles[1][FLIP] + nFlipTriangles[1][FLIP_REMOVE] +
	  nFlipEdges[1][FLIP] + nFlipEdges[1][FLIP_REMOVE]);


  fprintf(logFile, "# accepted triangle flips: %d (insert) %d (remove) = %d (total)\n",
	  nFlipTriangles[1][FLIP], nFlipTriangles[1][FLIP_REMOVE],
	  nFlipTriangles[1][FLIP] + nFlipTriangles[1][FLIP_REMOVE]);
  fprintf(logFile, "# accepted edge flips:     %d (insert) %d (remove) = %d (total)\n\n",
	  nFlipEdges[1][FLIP], nFlipEdges[1][FLIP_REMOVE],
	  nFlipEdges[1][FLIP] + nFlipEdges[1][FLIP_REMOVE]);

  fprintf(logFile, "# triangle flips: %d (insert) %d (remove) = %d (total)\n",
	  nFlipTriangles[0][FLIP] + nFlipTriangles[1][FLIP],
	  nFlipTriangles[0][FLIP_REMOVE] + nFlipTriangles[1][FLIP_REMOVE],
	  nFlipTriangles[0][FLIP] + nFlipTriangles[0][FLIP_REMOVE]+
	  nFlipTriangles[1][FLIP] + nFlipTriangles[1][FLIP_REMOVE]);
  fprintf(logFile, "# edge flips:     %d (insert) %d (remove) = %d (total)\n\n",
	  nFlipEdges[0][FLIP]+ nFlipEdges[1][FLIP],
	  nFlipEdges[0][FLIP_REMOVE] + nFlipEdges[1][FLIP_REMOVE],
	  nFlipEdges[0][FLIP] + nFlipEdges[0][FLIP_REMOVE]+
	  nFlipEdges[1][FLIP] + nFlipEdges[1][FLIP_REMOVE]);

  fprintf(logFile, "Average search length %6.1f\n\n", meanNSearch());

  fprintf(logFile, "# vertices %d\n",
	  vList->getNumberOfVertices());
  fprintf(logFile, "# simplices %d (Total generated %d)\n",
	  sList->getNumberOfSimplices(), sList->getId()+1);
  fprintf(logFile, "# edges %d (Total generated %d)\n",
	  eList->getNumberOfEdges(), eList->getId()+1);

  vList->printLogInfo(logFile, it + 1);

}

void Delaunay::printRes(FILE * resFile)
{
  // Result file
  vList->printRes(resFile);
}

void Delaunay::printMarginalPosterior(FILE * margFile, int nIter)
{
  vList->printMarginalPosterior(margFile, nIter);
}


void Delaunay::sample3D(short * colour, int dim[])
{
  Point pt = new double[DIM];
  double dx = 1.0/dim[0];
  double dy = 1.0/dim[1];
  double dz = 1.0/dim[2];

  Vertex *v; // Nearest vertes to pt

  int index = 0;

  for (int k=0;k<dim[2]; k++) {
    pt[2] = (((double) k) + 0.5) * dz;
    for (int j=0;j<dim[1]; j++) {
      pt[1] = (((double) j) + 0.5) * dy;
      for (int i=0;i<dim[0]; i++) {    
	pt[0] = (((double) i) + 0.5) * dx;

	v = NULL;
	vList->findNearestVertex(pt, v);

	colour[index++] = v->colour;
      }
    }
  }
}

void Delaunay::restore(int nMeanVertices, RandomGenerator * random,
		       Wells * wells, double simRegion[]) //CHANGE$$
{
  int i;
  int nPointsRestore;
  
  printf("Restore delaunay triangulation\n");

  assert(lastAction != MOVE);

  if (lastAction == ADD)
    nPointsRestore = nPoints-1;
  else if (lastAction == REMOVE)
    nPointsRestore = nPoints+1;

  mPoint * mPt = new mPoint[nPointsRestore];
  for (i=0;i<nPointsRestore;i++)
    mPt[i].pt = new double[DIM];

  if (lastAction == ADD)
    vList->getMarkedPoints(mPt, last);
  else
    vList->getMarkedPoints(mPt, NULL); // 'last' not removed from vList

  delete eList;
  delete sList;
  delete vList;

  lastCpy->nSegment = 0;

  sList = new SimplexList(nMeanVertices, random, simRegion);
  vList = new VertexList(nMeanVertices);
  eList = new EdgeList(vList, nMeanVertices);
  nPoints = 0;
  for (i=0; i<NGRIDDIM;i++)
  {
    int n = gridDim[i];
    if (DIM == 3)
      n *= n*n;
    else if (DIM == 2)
      n *= n;
    for (int j=0; j<n;j++)
    {
      nSimpArray[i][j] = 0;
      simpArray[i][j][0] = NULL;
      simpArray[i][j][1] = NULL;
    }
  }

  firstSimplex(mPt, wells);
  for (i=DIM+1;i<nPointsRestore;i++)
    insert(mPt[i], wells);

  vList->initColDiff();

  for (i=0;i<nPointsRestore;i++)
    delete [] mPt[i].pt;
  delete [] mPt;
}


void Delaunay::restart(FILE * file, mPoint * mPt, int nMPoints, Wells * wells,
		       int estimateMarginalPosterior) //CHANGE$$
{
  int i;
 
  for (i=0;i<nMPoints;i++)
    insert(mPt[i], wells);

  vList->initColDiff();

  if (estimateMarginalPosterior)
    vList->restartMarg(file);
}


void Delaunay::printTessellation(FILE * file, int it) //CHANGE$$
{
  fprintf(file, "%d\n", it);
  fprintf(file, "%d\n", nPoints-DIM-1);
  
  vList->printTessellation(file);
}
