#ifndef DELAUNAY__
#define DELAUNAY__ 1

#define NGRIDDIM 4

class Simplex;
class SimplexList;
class VertexList;
class EdgeList;
class Edge;
class Wells;


class Delaunay {
private:
  int nPoints;

  SimplexList * sList;
  VertexList * vList;
  EdgeList * eList;

  Vertex * last;
  Vertex * lastCpy;
  mPoint mPt;

  int lastAction;
  int debugCheck;

  int imageData;
  int volumeAnnealing;

  double region[2*DIM];

  Simplex * initialSearch(Vertex * v);
  Simplex * search(Simplex * sFirst);
  Simplex * findSimplex(Vertex * v, int n);
  int position(Point pt, int n);
  void connect(Simplex * sOld);
  void disConnect();
  int flip();
  int flip2D();
  int flip3D();
  int flipRemove();
  int flipRemove2D();
  int flipRemove3D();
  void flipRemove2(Simplex * sFirst);
  int flipEdge(Edge * e, Simplex ** sNew, Simplex * sRemoved[],
	       int & nRemoved, int flipType = FLIP);
  int flipTriangle(Simplex * s, Simplex * t, Simplex ** sNew,
		   Simplex * sRemoved[], int & nRemoved, int flipType = FLIP);
  int flipTriangle_old(Simplex * s, Simplex * t, Simplex ** sNew,
		       Simplex * sRemoved[], int & nRemoved, int flipType = FLIP);
  
  int oriented(Vertex * v[]);

  void insertSimp(Simplex * s);
  void removeSimp(Simplex * s);

  void insertSimpInEdges(Simplex *s);
  void removeSimpFromEdges(Simplex *s);

  void insertSimpInVertices(Simplex *s);
  void removeSimpFromVertices(Simplex *s);

  void insertSimpInSearch(Simplex * s);
  void insertSimpInSearch(Simplex * s, int * pos, int n);
  void removeSimpFromSearch(Simplex * s);
  void removeSimpFromSearch(Simplex * s, int * pos, int n);

  void checkSearchTable();

  int findStartIndex(int upperLevel, int lowerLevel, int posLower);
  void setPosSimp(int * posS, int * posSInv, int level);

  void getNeighbourVertices(Edge * e, Vertex * neighV[], Simplex * sOld[]);
  int volumeSign(Vertex * v[], Point newP);

  int gridDim[NGRIDDIM];
  int * nSimpArray[NGRIDDIM];
  Simplex *** simpArray[NGRIDDIM];
  int mark; // = Number of updates

  int * posSimp[NGRIDDIM];
  int * posSimpInv[NGRIDDIM];


  // Used in connection with flipping edges
  int permT[2][DIM+1];
  int indexT[2];
  int indexOldT[2][3];

  // Used in connection with flipping triangles
  int permE[3][DIM+1];
  int indexE[3][2];
  int indexETop[3][2];
  int indexNeigh[3][2];
  int indexNewNeigh[3][2];

  int perm[DIM+1][DIM];

  int it;

  //****** Log information *******
  
  int nSearch;
  int nSumSearch;
  int nCallSearch;

  int nFlipTriangles[2][2]; //Index 1: !Accept/Accept Index 2: insert/remove
  int nFlipEdges[2][2];

  //  int nInserted;
  //  int nRemoved;
  //  int nMoved;
  

public:

  Delaunay(int debug, int volAnn, int nMeanVertices,
	   RandomGenerator * random, double simRegion[]);
  ~Delaunay();

  void checkConsistency(Wells * wells);

  void firstSimplex(mPoint * t, Wells * wells);

  int insert(mPoint & x, Wells * wells);
  void insertData(Wells * wells);
  void insertData(Vertex * vNew, Wells * wells);

  int remove(Vertex * v, Wells * wells);
  void removeData(Wells * wells);

  void restore(int nMeanVertices, RandomGenerator * random,
	       Wells * wells, double simRegion[]);
  void restart(FILE * file, mPoint * mPt, int nMPoints, Wells * wells,
	       int estimateMarginalPosterior); //CHANGE$$

  void initialiseMarginalPosterior(int nColour, int dim[], int bIn);
  void initialiseMarginalPosterior();
  void updateMarginalPosteriorFinal();
  void checkMarginalPosterior();

  void initialiseVolFraction(int nColour, int dim[], int bIn);
  void initialiseVolFraction();

  void initialiseImageData(char * dataFileName, char * trueFileName,
			   int * dim, double region[]);
  void initialiseImageData();

  void insertImage();
  void removeImage();
  double SumColDiff();

  int move(Vertex * v, mPoint & x, Wells * wells);

  void acceptLastUpdate();
  void rejectLastUpdate(Wells * wells);

  void setIt(int iter);

  double meanNSearch();

  int sumFirstOrderColDiff(int action);
  int initColDiff();

  Vertex * selectVertex(RandomGenerator & random);
  int getNPoints();
  void updateMarginalPosterior(int action);

  void sample3D(short * colour, int dim[]);

  void testPoint(Point pt, double u); // CHANGE$$

  void volFractionAdd();
  void volFractionRemove();
  const double * volFraction() const;

  void printLogInfo(FILE * logFile);
  void printMarginalPosterior(FILE * margFile, int nIter);
  void printRes(FILE * resFile);
  void printTessellation(FILE * file, int it);
};

inline double Delaunay::meanNSearch()
{
  return nSumSearch/((double) nCallSearch);
}

inline void Delaunay::setIt(int iter)
{
  it = iter;
}

inline int Delaunay::getNPoints()
{
  return nPoints;
}


inline Vertex * Delaunay::selectVertex(RandomGenerator & random)
{
  int pos;
  int n = vList->getN();

  do {
    pos = random.iUnif(DIM+1, n-1);
  }
  while (vList->isRemoved(pos));

  return &(*vList)[pos];
}

inline void Delaunay::insertImage()
{
  vList->insertImage(last);

  if (debugCheck)
    vList->checkImage();
}

inline void Delaunay::removeImage()
{
  vList->removeImage(last, lastCpy);

  if (debugCheck)
    vList->checkImage();
}

inline void Delaunay::insertSimp(Simplex * s)
{
  insertSimpInVertices(s);

  insertSimpInEdges(s);

  insertSimpInSearch(s);
}

inline void Delaunay::insertSimpInVertices(Simplex * s)
{
  for (int i=0;i<DIM+1;i++)
  {
    Vertex * v = s->v[i];
    v->insert(s);
  }
}

inline void Delaunay::insertSimpInEdges(Simplex * s)
{
  Vertex ** v = s->getVertex();

  for (int j=0;j<DIM;j++) {
    for (int k=j+1;k<DIM+1;k++) {
      Edge * e = eList->insert(v[j], v[k]);
      e->insert(s);
      s->setEdge(j, k, e);
    }
  }
}

inline void Delaunay::removeSimp(Simplex * s)
{
  s->removed = 1;
  if (s->inserted)
    removeSimpFromSearch(s);
  removeSimpFromEdges(s);
  removeSimpFromVertices(s);
  //  sList->Free(s);
}

inline void Delaunay::removeSimpFromVertices(Simplex * s)
{
  for (int i=0;i<DIM+1;i++)
  {
    Vertex * v = s->v[i];
    v->remove(s);
  }
}

inline void Delaunay::removeSimpFromEdges(Simplex * s)
{
  for (int j=0;j<NEDGES;j++) {
    Edge * e = s->edge[j];
    e->remove(s);

    if (e->nSimp == 0)
      eList->remove(e);
  }
}


inline void Delaunay::removeSimpFromSearch(Simplex * s)
{
  Point p = s->getMidPoint();
  int pos[NGRIDDIM];

  for (int i=0;i<NGRIDDIM;i++)
  {
    pos[i] = position(p, i);
  }

  removeSimpFromSearch(s, pos, NGRIDDIM);
}


inline int Delaunay::flip()
{
  int error;

  if (DIM == 2)
    error = flip2D();
  else
    error = flip3D();

  return error;
}


inline int Delaunay::flipRemove()
{
  int error;

  if (DIM == 2)
    error = flipRemove2D();
  else
    error = flipRemove3D();

  return error;
}


inline int Delaunay::initColDiff()
{
  return vList->initColDiff();
}

inline void Delaunay::updateMarginalPosterior(int action)
{
  if (action == ADD) {
    vList->updateMarginalPosterior(last, it);
  }
  else if (action == REMOVE) {
    vList->updateMarginalPosterior(lastCpy, it); //NBNB Sjekk dette...
  }
  else {
    vList->updateMarginalPosterior(last, it); //NBNB Sjekk dette...
    vList->updateMarginalPosterior(lastCpy, it); //NBNB Sjekk dette...
  }
}

inline void Delaunay::initialiseMarginalPosterior(int nColour,
						  int margGridDim[],
						  int burnIn)
{
  vList->initialiseMarginalPosterior(nColour, margGridDim, burnIn,
				     region);
}

inline void Delaunay::initialiseMarginalPosterior()
{
  vList->initialiseMarginalPosterior();
}

inline void Delaunay::updateMarginalPosteriorFinal()
{
  vList->updateMarginalPosteriorFinal(it);
}

inline void Delaunay::checkMarginalPosterior()
{
  vList->checkMarginalPosterior();
}

inline void Delaunay::initialiseVolFraction(int nColour, int margGridDim[],
				     int burnIn)
{
  vList->initialiseVolFraction(nColour, margGridDim, burnIn,
			       region);
}

inline void Delaunay::initialiseVolFraction()
{
  vList->initialiseVolFraction();
}

inline void Delaunay::initialiseImageData(char * dataFileName,
					  char * trueFileName,
					  int * dim, double region[])
{
  imageData = 1;
  vList->initialiseImageData(dataFileName, trueFileName, dim, region);
}

inline void Delaunay::initialiseImageData()
{
  vList->initialiseImageData();
}

inline double Delaunay::SumColDiff()
{
  return vList->SumColDiff();
}

inline void Delaunay::volFractionAdd()
{
  vList->insertVol(last);

  if (volumeAnnealing && debugCheck)
    vList->checkVolumeFraction();
}

inline void Delaunay::volFractionRemove()
{
  vList->removeVol(last, lastCpy);

  if (volumeAnnealing && debugCheck)
    vList->checkVolumeFraction();
}

inline const double * Delaunay::volFraction() const
{
  return vList->volumeFraction();
}

inline void Delaunay::testPoint(Point pt, double u)
{
  double t[DIM];
  t[0] = u;
  t[1] = 0.5*(1.0-u);
  t[2] = 0.5*(1.0-u);//CHANGE$$ only for 3D
  for (int i=0;i<DIM;i++) {
    pt[i] = 0.0;
    int j=0;
    int n=0;
    while (n<DIM) {
      int pos = j+DIM+1;
      Vertex * v = &((*vList)[pos]);
      if (!v->removed) {
	pt[i] += t[n]*v->pt[i];
	n++;
      }
      j++;
    }
  }
}
#endif
