#include <stdio.h>
#include <math.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include "random.h"

#define  EPSILON 0.00000001

#define  G_MULTIPLIER  69069 
#define  G_SHIFT           1 
#define  G_MODULUS  256*256*256*128 
#define  G_INVMOD  ( (double) 1 / ((double) G_MODULUS )) / ( (double) 2 ) 
#define PI  3.14159265358979323846

const double pi =  PI;

RandomGenerator::RandomGenerator(unsigned  int iseed) : RandomSeed()
{
    setSeed(iseed);
}

RandomGenerator::RandomGenerator(char *filename) : RandomSeed()
{ 
   setSeed(filename);
}


/*F:RandomGenerator::unif01*
________________________________________________________________
		RandomGenerator::unif01
________________________________________________________________
Name:		RandomGenerator::unif01
Syntax:		@RandomGenerator::unif01-syntax
Description:
              Generates a uniform[0,1] variable using the congruential
              generator given in ripley: stochastic simulation, p. 46.
              The start-seed should be either negative, or big and positive.
              The seed is changed in this function.
              NOTE: Requires that a  int is 32 bits because the
              algorithm requires that modulus 2^32 is taken.
	      This is done implicit using 32bits integer.

Side effects:
              The start-seed should be either negative, or big and positive.
              The seed is changed in this function.

Return value: The pseudo-random number in [0,1], both inclusive.
Example: l_unif = lib_ran_unif01()

Author:		Oddvar Lia, Arne Skorstad NR
Date: Dec 94
Source file: 
________________________________________________________________*/
/*<RandomGenerator::unif01-syntax: */
double RandomGenerator::unif01(void) 
/*>RandomGenerator::unif01-syntax: */
{
 double l_ran;

 seed = G_MULTIPLIER * seed + G_SHIFT; /* Modulus 2^32 implicit */
 l_ran = (double) seed * G_INVMOD;
 return(l_ran);
}	/* end of RandomGenerator::unif01 */





/*F:RandomGenerator::unif*
________________________________________________________________
		RandomGenerator::unif
________________________________________________________________
Name:		RandomGenerator::unif
Syntax:		@RandomGenerator::unif-syntax
Description:
              Generates a uniform[min,max] variable using 
	      transformation of a uniform[0,1] variable

Side effects:

Return value: The pseudo-random number in [min,max], both inclusive.
Example: l_unif = lib_ran_unif01()

Author:		Haakon Tjelmeland, NR
Date: Feb 95
Source file: 
________________________________________________________________*/
/*<RandomGenerator::unif-syntax: */
double RandomGenerator::unif(double min,double max) 
/*>RandomGenerator::unif-syntax: */
{
  return min + (max - min) * unif01();

}	/* end of RandomGenerator::unif */





/*F:RandomGenerator::iUnif*
________________________________________________________________
		RandomGenerator::iUnif
________________________________________________________________
Name:		RandomGenerator::iUnif
Syntax:		@RandomGenerator::iUnif-syntax
Description:
              Generates a variable from uniform distribution for integers 
	      in [min,max]  by transformation of a uniform[0,1] variable

Side effects:

Return value: The pseudo-random number in [min,max], both inclusive.
Example: l_unif = lib_ran_unif01()

Author:		Haakon Tjelmeland NR
Date: Feb 95
Source file: 
________________________________________________________________*/
/*<RandomGenerator::iUnif-syntax: */
int RandomGenerator::iUnif(int min,int max) 
/*>RandomGenerator::iUnif-syntax: */
{
  assert(max >= min);

  double ran = unif01();
  int range = max - min + 1;
  int value = (int) (ran * range) + min;
  if (value == max + 1) value -= 1;

  return value;

}	/* end of RandomGenerator::iUnif */

/*F:RandomGenerator::normal01*
________________________________________________________________
		RandomGenerator::normal01
________________________________________________________________
Name:		RandomGenerator::normal01
Syntax:		@RandomGenerator::normal01-syntax
Description:
              Generate two independent normal(0,1)-distributed numbers,
              using the Box-Muller method.  Reference: Ripley p.54.
	      The start-seed should be either negative, or big and positive.

Example: lib_ran_norm01(double &o_x1, double &o_x2)

Author:		Oddvar Lia, Arne Skorstad NR
Date: Dec 94 
Source file: 
________________________________________________________________*/
/*<RandomGenerator::normal01-syntax: */
void RandomGenerator::normal01(double &o_x1, double &o_x2)
/*>RandomGenerator::normal01-syntax: */
{
  double l_u1, l_u2;

/* First generate two uniform(0,1)-numbers: */
  l_u1 = unif01();
  l_u2 = unif01();

/* Then compute two normal(0,1)-numbers: */
  o_x1 = sqrt(-2.0*log(l_u1)) * cos(2.0*pi*l_u2); 
  o_x2 = sqrt(-2.0*log(l_u1)) * sin(2.0*pi*l_u2);  
  return;
}

RandomSeed::RandomSeed(void)
{
   strcpy(errorMsg, "");
}

RandomSeed::~RandomSeed(void)
{
}


void RandomSeed::setSeed(unsigned int iseed)
{
   seed=iseed;
}

int RandomSeed::setSeed(const char * seedfile)
{
   FILE * file = fopen(seedfile, "r");
   if (!file)
   {
      printf("Could not open seed file %s for reading",seedfile);
      return 1;
   }

   fscanf(file, "%u", &seed);

   return 0;
}


unsigned int RandomSeed::getSeed(void) const
{
   return seed;
}

// Definition and initialization of static member
unsigned int RandomSeed::seed = 140567; // Default seed
