/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <assert.h>

#include "model_scan.h"
#include "basic.h"
#include "point.h"
#include "sampleGrid.h"

#define NCOMMANDS 5
#define FILES 0
#define DIMENSION 1
#define CONSTANT 2
#define OBJECTS 3
#define WELLS 4

int main(int argc, char **argv)
{
  if (argc<=1 || argc >=3) {
    printf("Usage: %s <model-file>\n", argv[0]);
    exit(-1);
  }

  SampleGrid * sGrid = new SampleGrid(argv[1]);

  sGrid->read();

  sGrid->transformWellCoord();

  sGrid->sample();

  sGrid->write();

  delete sGrid;
}


SampleGrid::SampleGrid(char * modelFileName)
{
  readInput(modelFileName);

  for (int i=0;i<nWells;i++) {
    //    wellPt[i] = 
  }
  
}

SampleGrid::~SampleGrid()
{
  int i;
  for (i=0;i<nWells;i++) {
    delete [] wellName[i];
    delete [] wellPt[i];
  }
  delete [] wellName;
  delete [] wellPt;

  for (i=0;i<nVertexPts;i++) {
    delete [] vertexPts[i];
    delete [] neigh[i];
  }

  delete [] neigh;
  delete [] vertexIdInv;
  delete [] vertexPts;
  delete [] colour;
  delete [] nNeigh;

  delete [] inputFileName;
  delete [] gridFileName;
  
  delete [] grid;
}

void SampleGrid::readInput(char * modelFileName)
{
  int i, j;
  int index;
  int iComm;
  char **commands;
  struct COMMAND *sComm, *sCommands;
   
  commands = (char **) calloc(NCOMMANDS, sizeof(char *));
  for (i=0;i<NCOMMANDS;i++)
    commands[i] =  (char *) calloc(50, sizeof(char));

  strcpy(commands[FILES], "FILES");
  strcpy(commands[DIMENSION], "DIMENSION");
  strcpy(commands[CONSTANT], "CONSTANT");
  strcpy(commands[OBJECTS], "OBJECTS");
  strcpy(commands[WELLS], "WELLS");

  sCommands = spReadModelFile(modelFileName, commands, NCOMMANDS);
   
  for (j=0;j<DIM;j++) {
    sampleXYZ[j] = 1;
    ptFixed[j] = 0.0;
  }

  for (iComm=0;iComm<NCOMMANDS;iComm++) {
    sComm = &sCommands[iComm];

    switch (iComm) {
    case FILES:
      if (!sComm->bRead) {
        printf("Command FILES not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != 2) {
        printf("Number of arguments for the FILES command = %d != 2\n", sComm->nArgs);
	exit(-1);
      }
      inputFileName = new char[strlen(sComm->cppArgs[0])+1];
      strcpy(inputFileName, sComm->cppArgs[0]);

      gridFileName = new char[strlen(sComm->cppArgs[1])+1];
      strcpy(gridFileName, sComm->cppArgs[1]);
       
      break;

    case DIMENSION:
      if (!sComm->bRead) {
        printf("Command DIMENSION not specified\n");
	exit(-1);
      }
      if (sComm->nArgs != DIM) {
        printf("Number of arguments for the DIMENSION command != DIM\n");
	exit(-1);
      }
      for (j=0;j<DIM;j++)
	gridDim[j] = atoi(sComm->cppArgs[j]);

      break;

    case OBJECTS:
      objects = sComm->bRead;

      break;

    case CONSTANT:
      if (!sComm->bRead) {
	break;
      }
      if (sComm->nArgs != 2) {
        printf("Number of arguments for the CONSTANT command != 2\n");
	exit(-1);
      }

      index = atoi(sComm->cppArgs[0]);
      if (index<0 || index>=DIM) {
	printf("Wrong index %d for the CONSTANT command\n", index);
	exit(-1);
      }
      sampleXYZ[index] = 0;
      ptFixed[index] = atof(sComm->cppArgs[1]);

      break;

    case WELLS:
      if (!sComm->bRead) {
	break;
      }
      if (((sComm->nArgs % 3) != 0) || (sComm->nArgs == 0)) {
        printf("Bad number of arguments for the WELLS command\n");
	exit(-1);
      }
      nWells = sComm->nArgs/3;

      wellName = new char *[nWells];
      wellPt = new double * [nWells];

      index = 0;
      for (i=0;i<nWells;i++) {
	wellName[i] = new char[strlen(sComm->cppArgs[index])+1];
	strcpy(wellName[i], sComm->cppArgs[index++]);

	wellPt[i] = new double[2];

	wellPt[i][0] = atof(sComm->cppArgs[index++]);

	wellPt[i][1] = atof(sComm->cppArgs[index++]);
      }

      break;
    }
  }


  for (i=0;i<NCOMMANDS;i++)
    free(commands[i]);
  free(commands);

  

  for (iComm=0;iComm<NCOMMANDS;iComm++) {
    for (j=0;j<sCommands[iComm].nArgs;j++)
      free(sCommands[iComm].cppArgs[j]);
    free(sCommands[iComm].cppArgs);
  }
	
  free(sCommands);
}

void SampleGrid::sampleAlongWells()
{
  int i;
  int k;
  Point pt = new double[DIM];
  double d[DIM];

  for (i=0;i<DIM;i++)
    d[i] = region[2*i+1]-region[2*i];

  //  double dx = d[0]/gridDim[0];
  //  double dy = d[1]/gridDim[1];
  double dz = d[2]/gridDim[2];

  double * dxW = new double[nWells-1];
  double * dyW = new double[nWells-1];

  int * nGrid = new int[nWells-1];

  // Find total length along given path through wells
  double totalDist = 0.0;
  int iWell;
  double xLength;
  double yLength;
  double * dXY = new double[nWells];

  for (iWell=0;iWell<nWells-1;iWell++) {
    xLength =  wellPt[iWell+1][0] - wellPt[iWell][0];
    yLength =  wellPt[iWell+1][1] - wellPt[iWell][1];

    dXY[iWell] = sqrt(xLength*xLength + yLength*yLength);

    totalDist += dXY[iWell];
  }

  // Find step length;
  double dStepXY = totalDist/gridDim[0];

  int newGridDimXY = 0;
  for (iWell=0;iWell<nWells-1;iWell++) {
    xLength =  wellPt[iWell+1][0] - wellPt[iWell][0];
    yLength =  wellPt[iWell+1][1] - wellPt[iWell][1];

    nGrid[iWell] = max(1, (int) ((dXY[iWell])/dStepXY + 0.5));

    newGridDimXY += nGrid[iWell];

    dxW[iWell] = xLength/((double) nGrid[iWell]);

    dyW[iWell] = yLength/((double) nGrid[iWell]);
  }

  delete [] dXY;
  dXY = NULL;

  gridDim[0] = newGridDimXY;
  gridDim[1] = 1;
  int n = newGridDimXY * gridDim[2];
  grid = new short[n];
  

  int index = 0;

  pt[2] = region[4] - 0.5*dz;
  for (k=0;k<gridDim[2]; k++) {
    pt[2] += dz;

    int j;

    int vIndex=-1; // Nearest vertex to pt

    for (iWell=0;iWell<nWells-1;iWell++) {
      pt[0] = wellPt[iWell][0] - 0.5*dxW[iWell];
      pt[1] = wellPt[iWell][1] - 0.5*dyW[iWell];
      for (j=0;j<nGrid[iWell];j++) {
	// Find x and y coordinates
	pt[0] += dxW[iWell];
	pt[1] += dyW[iWell];
      
	// Transform x and y coordinates into indices..
	findNearestVertex(pt, vIndex);

	grid[index++] = colour[vIndex];
      }
    }
  }
  
  delete [] nGrid;
  delete [] dxW;
  delete [] dyW;
  delete [] pt;
  
}

void SampleGrid::sample()
{
  if (nWells > 0)
    sampleAlongWells();
  else
    sampleNormal();
}


void SampleGrid::sampleNormal()
{
  int i;
  int k;
  Point pt = new double[DIM];
  double d[DIM];
  for (i=0;i<DIM;i++)
    d[i] = region[2*i+1]-region[2*i];

  double dx = d[0]/gridDim[0];
  double dy = d[1]/gridDim[1];
  double dz = d[2]/gridDim[2];

  int vIndex; // Nearest vertex to pt

  int index = 0;
  int * vertexFound = NULL;
  if (objects) {
    vertexFound = new int[nVertexPts];
  }

  int n = gridDim[0] * gridDim[1] * gridDim[2];
  grid = new short[n];

  pt[2] = region[4] - 0.5*dz;
  for (k=0;k<gridDim[2]; k++) {
    if (sampleXYZ[2])
      pt[2] += dz;
    else
      pt[2] = ptFixed[2];

    pt[1] = region[2] - 0.5*dy;
    for (int j=0;j<gridDim[1]; j++) {
      if (sampleXYZ[1])
	pt[1] += dy;
      else
	pt[1] = ptFixed[1];

      pt[0] = region[0] - 0.5*dx;
      vIndex = -1;
      for (i=0;i<gridDim[0]; i++) {    
	if (sampleXYZ[0])
	  pt[0] += dx;
	else
	  pt[0] = ptFixed[0];

	findNearestVertex(pt, vIndex);

	if (objects) {
	  grid[index++] = vIndex;
	  vertexFound[vIndex] = 1;
	}
	else
	  grid[index++] = colour[vIndex];
      }
    }
  }
  if (objects) {
    index = 0;
    for (k=0;k<nVertexPts;k++) {
      if (vertexFound[k])
	vertexFound[k] = index++;
    }
    for (k=0;k<n;k++)
      grid[k] = vertexFound[grid[k]];

    delete [] vertexFound;
  }

  delete [] pt;
}

void SampleGrid::findNearestVertexSlow(Point pt, int & vIndex)
{
  double distMin=10000000000.0;
  int vMin = -1;


  for (int i=0;i<nVertexPts;i++) {
    Point vPt = vertexPts[i];

    double dist = distPoint(vPt, pt);
      
    if (dist<distMin) {
      vMin = i;
      distMin = dist;
    }
  }

  assert(vMin>=0);

  vIndex = vMin;
}


void SampleGrid::findNearestVertex(Point pt, int & vIndex)
{
  if (vIndex<0) {
    findNearestVertexSlow(pt, vIndex);
    return;
  }

  int found = 0;
  double distMin=distPoint(vertexPts[vIndex], pt);
  while (!found) {
    int vMin = -1;

    for (int i=0;i<nNeigh[vIndex];i++) {
      int I = vertexIdInv[neigh[vIndex][i]];
      Point vPt = vertexPts[I];

      double dist = distPoint(vPt, pt);
      
      if (dist<distMin) {
	vMin = I;
	distMin = dist;
      }
    }
    if (vMin>=0)
      vIndex = vMin;
    else
      break;
  }
}

void SampleGrid::transformWellCoord()
{
  for (int i=0;i<nWells;i++) {
    wellPt[i][0] -= regionOrig[XMIN];
    wellPt[i][1] -= regionOrig[YMIN];
  }
}

void SampleGrid::read()
{
  FILE * file = fopen(inputFileName, "r");
  assert(file);

  mPoint x;
  x.pt = new double[DIM];

  int i;
  for (i=0;i<2*DIM;i++)
    fscanf(file, "%lf", &region[i]);
  for (i=0;i<2*DIM;i++)
    fscanf(file, "%lf", &regionOrig[i]);

  fscanf(file, "%lf", &zAnisotropy);

  //  region[ZMAX] *= zAnisotropy;

  fscanf(file, "%d ", &nMaxId);
  fscanf(file, "%d ", &nVertexPts);

  
  vertexIdInv = new int [nMaxId];
  vertexPts = new Point [nVertexPts];
  colour = new short [nVertexPts];
  nNeigh = new short [nVertexPts];
  neigh = new int * [nVertexPts];

  for (i=0;i<nVertexPts;i++) {
    int id;
    fscanf(file, "%d", &id);
    vertexIdInv[id] = i;
    vertexPts[i] = new double[DIM];
    Point pt = vertexPts[i];
    fscanf(file, "%lf %lf %lf", &pt[0], &pt[1], &pt[2]);
    
    fscanf(file, "%hd", &colour[i]);

    fscanf(file, "%hd", &nNeigh[i]);

    neigh[i] = new int [nNeigh[i]];

    for (int j=0;j<nNeigh[i];j++) {
      fscanf(file, "%d", &neigh[i][j]);
    }
  }

  delete [] x.pt;

  fclose(file);
}

void SampleGrid::write()
{
  int index = 0;
  FILE * file = fopen(gridFileName, "w");
  int i;
  for (i=0;i<DIM; i++)
    fprintf(file, "%d ", gridDim[i]);
  fprintf(file, "\n");
  for (int k=0;k<gridDim[2]; k++) {
    for (int j=0;j<gridDim[1]; j++) {
      for (i=0;i<gridDim[0]; i++) { 
	fprintf(file, "%d ", grid[index++]);
      }
      fprintf(file, "\n");
    }
  }
  fclose(file);
}
