/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <errno.h>
#include <assert.h>

#include "basic.h"
#include "random.h"
#include "point.h"
#include "vertex.h"
#include "edge.h"
#include "simplex.h"

int Simplex::edgeMatrix[DIM+1][DIM+1];
#ifdef DIM2D
int Simplex::edgeIndex[DIM+1][NEDGE_T] = {{2}, {1}, {0}};
int Simplex::edgeIndex2[DIM+1][NEDGE_V] = {{0,1}, {0,2}, {1,2}};
int Simplex::perm[DIM+1][DIM] = {{1,2}, {2,0}, {0,1}};
#else
int Simplex::edgeIndex[DIM+1][NEDGE_T] = {{3,4,5}, {1,2,5}, {0,2,4}, {0,1,3}};
int Simplex::edgeIndex2[DIM+1][NEDGE_V] = {{0,1,2}, {0,3,4}, {1,3,5}, {2,4,5}};
int Simplex::perm[DIM+1][DIM] = {{1,3,2}, {0,2,3}, {0,3,1}, {0,1,2}};
#endif

RandomGenerator * Simplex::random;
double Simplex::pertEps;
double Simplex::equalLimit;

void Simplex::print()
{
  printf("Id: %d It: %d Neigh: ", id, it);

  int i;
  for (i=0; i<DIM+1;i++) {
    if (neigh[i])
      printf("%d ", neigh[i]->id);
    else
      printf("NULL ");
  }
  printf("\n");

  for (i=0; i<DIM+1;i++) {
    for (int j=0; j<DIM;j++)
      printf("%6.4f ", (*v[i])[j]);
    printf("\n");
  }
}

void Simplex::print(VertexList * vList)
{
  print(vList, stdout);
}

void Simplex::print(VertexList * vList, FILE * file)
{
  fprintf(file, "Id: %d V: ", id);
  for (int i=0; i<DIM+1;i++) {
    fprintf(file, "%d ", vList->pos(v[i]));
  }
  fprintf(file, "\n");
}

void Simplex::printDistance(Point p)
{
  double dist = distPoint(p, midPoint);

  printf("Dist: %6.4f\n", dist);
}

double Simplex::findVolume(Point pt[], Point newP)
{
  double vol;

#ifdef DIM2D
  double ba0 = pt[1][0] - pt[0][0];
  double ba1 = pt[1][1] - pt[0][1];
  double ca0 = newP[0] - pt[0][0];
  double ca1 = newP[1] - pt[0][1];
    
  vol = -ba0*ca1 + ca0*ba1; // NBNB Inverse def. of area sign in O'Rourke
#else
  double ad0 = pt[0][0] - newP[0];
  double ad1 = pt[0][1] - newP[1];
  double ad2 = pt[0][2] - newP[2];
		 	    
  double bd0 = pt[1][0] - newP[0];
  double bd1 = pt[1][1] - newP[1];
  double bd2 = pt[1][2] - newP[2];
		 	    
  double cd0 = pt[2][0] - newP[0];
  double cd1 = pt[2][1] - newP[1];
  double cd2 = pt[2][2] - newP[2];

  vol =  - ad2 * bd1 * cd0 + ad1 * bd2 * cd0;
  vol +=   ad2 * bd0 * cd1 - ad0 * bd2 * cd1;
  vol += - ad1 * bd0 * cd2 + ad0 * bd1 * cd2;
#endif

  if (equal(vol, 0.0)) { //CHANGE$$
    printf("WARNING(Simplex::findVolume): vol %26.20e\n%s\n",
	   vol, "Tries long double precision\n");
    vol = findVolumePrecise(pt, newP);
  }

  return vol;
}

double Simplex::findVolumePrecise(Point pt[], Point newP)
{
  long double vol;

#ifdef DIM2D
  long double ba0 = pt[1][0] - pt[0][0];
  long double ba1 = pt[1][1] - pt[0][1];
  long double ca0 = newP[0] - pt[0][0];
  long double ca1 = newP[1] - pt[0][1];
  
  vol = -ba0*ca1 + ca0*ba1; // NBNB Inverse def. of area sign in O'Rourke
#else
  long double ad0 = pt[0][0] - newP[0];
  long double ad1 = pt[0][1] - newP[1];
  long double ad2 = pt[0][2] - newP[2];
  
  long double bd0 = pt[1][0] - newP[0];
  long double bd1 = pt[1][1] - newP[1];
  long double bd2 = pt[1][2] - newP[2];
		 	    
  long double cd0 = pt[2][0] - newP[0];
  long double cd1 = pt[2][1] - newP[1];
  long double cd2 = pt[2][2] - newP[2];
  
  vol =  - ad2 * bd1 * cd0 + ad1 * bd2 * cd0;
  vol +=   ad2 * bd0 * cd1 - ad0 * bd2 * cd1;
  vol += - ad1 * bd0 * cd2 + ad0 * bd1 * cd2;
#endif

  if (vol == 0.0)
  {
    printf("WARNING(Simplex::findVolumePrecise): vol == 0\n");
  }

  return (double) vol;
}

#ifdef NOT_IN_USE //CHANGE$$
void Simplex::drawPoint(RandomGenerator & random, Point pt)
{
  // Draw barycentric coordinates t1, t2, t3, t4: \sum ti = 1
  double sum;
  double t[DIM+1];

  do {
    sum = 0.0;
    for (int j=0;j<DIM;j++) {
      t[j] = random.unif01();
      sum += t[j];
    }
  } while (sum>1.0);

  t[DIM] = 1.0 - sum;

  for (int i=0;i<DIM;i++)
    for (int j=0;j<DIM+1;j++)
      pt[i] += t[j] * (*v[j])[i];

  assert(inside(pt));
}
#endif
SimplexList::SimplexList(int nMeanVertices, RandomGenerator * random,
			 double region[])
{
  n = 0;
  nAlloc = 100 * nMeanVertices;
  nAllocDelta = nAlloc;
  nRemoved = 0;
  nRemAlloc = 10000;
  nRemAllocDelta = nRemAlloc;

  id = 0;

  int i;
  int k=0;
  for (i=0;i<DIM;i++)
    for (int j=i+1;j<DIM+1;j++) {
      Simplex::edgeMatrix[i][j] = k;
      Simplex::edgeMatrix[j][i] = k;
      k++;
    }

  list = (Simplex *) malloc(nAlloc*sizeof(Simplex));
  removed = (Simplex **) malloc(nRemAlloc*sizeof(Simplex *));

  Simplex::random = random;  //CHANGE$$
  double maxDiff = 0.0;
  double volume = 1.0;
  for (i=0;i<DIM;i++) {
    double diff = region[2*i+1]-region[2*i];
    volume *= diff;
    if (diff > maxDiff) {
      maxDiff = diff;
    }
  }
  Simplex::pertEps = maxDiff * 1e-12;
  Simplex::equalLimit = volume * 1e-14;
}

SimplexList::~SimplexList()
{
  free(list);
  free(removed);
}

void SimplexList::print()
{
  for(int i=0;i<n;i++)
    if (!list[i].removed)
      list[i].print();
}

void SimplexList::print(VertexList * vList)
{
  print(vList, stdout);
}
void SimplexList::print(VertexList * vList, FILE * file)
{
  for(int i=0;i<n;i++)
    if (!list[i].removed)
      list[i].print(vList, file);
}

void SimplexList::checkIfDelaunay(VertexList * vList)
{
  for (int i=0; i<n;i++) {
    Simplex * s = &list[i];

    if (!s->removed) {
    
      for (int j=0; j<DIM+1;j++) {
	int index;
	Simplex * t = s->getNeigh(s->v[j], index);
	if (t) {
	  double dist = distPoint(s->v[j]->getPt(), t->centerpoint);
	  
	  if (t->radius > dist) {// NOT locally delaunay
	    printf("Error (SimplexList::checkIfDelaunay)\n");
	    s->print(vList);
	    s->print();
	    t->print(vList);
	    t->print();
	    exit(-1);
	  }
	}
      }
    }
  }
}

void SimplexList::checkConsistency(VertexList * vList, EdgeList * eList,
				   int perm[][DIM])
{
  int nV =  vList->getN();
  short * vPos = new short[nV];

  int i;
  for (i=0; i<nV;i++)
    vPos[i] = 0;

  int nE = eList->getNMaxVertices();
  short * ePos = NULL;

  int eAllocated = 0;
  if (nE <=1000) {
    eAllocated = 1;

    nE *= nE;

    ePos = new short[nE];

    for (i=0; i<nE;i++)
      ePos[i] = 0;
  }

  for (i=0; i<n;i++) {
    Simplex * s = &list[i];

    if (!s->removed) {
      // Check 1: orientation
      if (!s->oriented(perm)) {
	printf("Error(Simplex::checkConsistency): Simplex not oriented\n");
	exit(-1);
      }
    
      int j;
      if (eAllocated) {
	for (j=0; j<NEDGES;j++) {
	  int keyIndex = eList->getKeyIndex(s->edge[j]);
	  ePos[keyIndex]++;
	}
      }
      for (j=0; j<DIM+1;j++) {
	vPos[vList->pos(s->v[j])]++;

	int index;
	int tIndex;
	Simplex * t = s->getNeigh(s->v[j], index); // j == index
	if (t) {
	  assert(!t->removed);

	  tIndex = t->getNeigh(s);

	  // Check 2: That s and t do not intersect
	  int volSign[2];
	  volSign[0] = t->volumeSign(perm[tIndex], s->v[index]->getPt());
	  volSign[1] = t->volumeSign(perm[tIndex], t->v[tIndex]->getPt());
	  if (volSign[0] == volSign[1]) {
	    printf("Error(SimplexList::checkConsistency): %s\n",
		   "s and t intersects");
	    printf("s (neigh index = %d): ", index);
	    s->print(vList);
	    printf("t (neigh index = %d): ", tIndex);
	    t->print(vList);
	    exit(-1);
	  }

	  // Check 3: If neighbour contains DIM common vertices...
	  int nCommon = 0;
	  for (int k=0; k<DIM+1;k++) {
	    if (k != tIndex) {
	      for (int l=0; l<DIM+1;l++) {
		if (l != index) {
		  if (s->v[l] == t->v[k]) {
		    nCommon++;
		    break;
		  }
		}
	      }
	    }
	  }
	  if (nCommon != DIM) {
	    printf("Error(SimplexList::checkConsistency): %s%d\n",
		   "# common vertices < ", DIM);
	    printf("s (neigh index = %d): ", index);
	    s->print(vList);
	    printf("t (neigh index = %d): ", tIndex);
	    t->print(vList);
	    exit(-1);
	  }
	}
      }
    }
  }
  vList->checkConsistency(vPos);
  if (eAllocated) {
    eList->checkConsistency(ePos);
    delete [] ePos;
  }

  delete [] vPos;
}
