#ifndef SIMPLEX__
#define SIMPLEX__ 1

#define MAXDIM 100

#define push(x) {if (tms>=ss) assert(st= (Simplex **) realloc(st, ((ss+=ss)+MAXDIM+1)*sizeof(Simplex *))); *(st+tms++) = x;}
#define pop(x)  x = *(st + --tms);
#define initStackMemory() if (!st) st = (Simplex **) malloc((MAXDIM+1)*sizeof(Simplex *));
//#define stackMemory() if (tms>ss) assert(st= (Simplex **) realloc(st, ((ss+=ss)+MAXDIM+1)*sizeof(Simplex *)));

class RandomGenerator;
class Edge;
class EdgeList;
class Vertex;

class Simplex
{
  friend class Delaunay;
  friend class SimplexList;
private:

  inline int equal(double a, double b);
  inline void Axb(double A[][DIM], double b [] , double x []);
  double findVolume(Point pt[], Point newP);
  double findVolumePrecise(Point pt[], Point newP);
  int volumeSign(int * permIn, Point newP, int defaultSign=0);

  long visit;		/* number of last site visiting this simplex */
  short removed;
  short inserted; // Used in insertSimpInSearch/removeSimpFromSearch
  int it;
  int id;
  double centerpoint[DIM];
  double midPoint[DIM];
  double volume;
  double radius;
  Simplex * neigh[DIM+1];
  Vertex * v[DIM+1];

  Edge * edge[NEDGES];

  static double equalLimit;
  static double pertEps; //CHANGE$$
  static RandomGenerator * random; //CHANGE$$

  static int edgeIndex[DIM+1][NEDGE_T];
  static int edgeIndex2[DIM+1][NEDGE_V];
  static int edgeMatrix[DIM+1][DIM+1];
  static int perm[DIM+1][DIM];
  
public:

  void print();
  void print(VertexList * vList);
  void print(VertexList * vList, FILE * file);
  void printDistance(Point p);

  void initialise(Vertex * vertex[], int iter=0);
  void initialise(int iter=0);
  void findSphere();

  void changeOrientation();
  int oriented(int permIn[][DIM], int defaultOriented=0);

  int inside(Point p);
  void setNeigh(Simplex * sOld, Simplex * sNew);

  int convex(Simplex * s, Point p);

  double getVolume();
  double findVolume();

  void drawPoint(RandomGenerator & random, Point pt);

  int isMember(Vertex * vertex);
  int isVisited(int mark);
  void setVisit(int mark);

  int getNeigh(Vertex * v);
  Simplex * getNeigh(Vertex * v, int & top);
  Simplex * getNeigh(int index);
  int getNeigh(Simplex * s);
  void getNeigh(int index, int top[], Simplex * t[]);

  Edge * getEdge(int top, int edgeNr); // Used in flip
  Edge * getEdge2(int top, int edgeNr); // Used in flipRemove
  void setEdge(int j, int k, Edge * e);

  Point getPt(int pos);
  Point getMidPoint();

  Vertex ** getVertex();
  Vertex * getVertex(int index);
  int getRemainingVertex(Vertex * w[]);
  int getRemainingVertex(Vertex * w[], int index);
  void getRemainingVertices(Vertex * vEdge[], Vertex * vNew[]);

  int isDelaunay(Vertex * w);
  int isDelaunay(Vertex ** w, int n);
};


class SimplexList
{
public:
  SimplexList(int nMeanVertices, RandomGenerator * random,
	      double region[]);
  ~SimplexList();

  Simplex * New();
  void Free(Simplex * s);

  void print();
  void print(VertexList * vList);
  void print(VertexList * vList, FILE * file);

  void checkIfDelaunay(VertexList * vList);
  void checkConsistency(VertexList * vList, EdgeList * eList,
			int perm[][DIM]);

  int getNumberOfSimplices();
  int getId();

private:
  Simplex * list;
  Simplex ** removed;
  int simplexSize;
  int n;
  int nAlloc;
  int nAllocDelta;
  int nRemoved;
  int nRemAlloc;
  int nRemAllocDelta;

  int id;
};

inline int SimplexList::getId()
{
  return id;
}

inline int SimplexList::getNumberOfSimplices()
{
  return n - nRemoved;
}


inline double Simplex::getVolume()
{
  return volume;
}

inline void Simplex::getRemainingVertices(Vertex * vEdge[], Vertex * vNew[])
{
  int k=0;
  for (int i=0;i<DIM+1;i++)
    if (v[i] != vEdge[0] && v[i] != vEdge[1])
      vNew[k++] = v[i];

  assert(k==2);
}

inline int Simplex::getRemainingVertex(Vertex * w[], int index)
{
  int i;
  int found;

  for (i=0;i<DIM+1;i++) {
    found = 0;
    for (int j=0;j<DIM+1;j++) {
      if (j != index) {
	if (v[i] == w[j]) {
	  found = 1;
	  break;
	}
      }
    }
    if (!found)
      break;
  }

  assert(!found);

  return i;
}

inline int Simplex::getRemainingVertex(Vertex * w[])
{
  int i;
  int found;

  for (i=0;i<DIM+1;i++) {
    found = 0;
    for (int j=0;j<DIM+1;j++) {
      if (v[i] == w[j]) {
	found = 1;
	break;
      }
    }
    if (!found)
      break;
  }

  assert(!found);

  return i;
}

inline void Simplex::setVisit(int mark)
{
  visit = mark;
}

inline int Simplex::isVisited(int mark)
{
  return (mark == visit);
}

inline int Simplex::getNeigh(Vertex * vertex)
{
  int i;
  for (i=0;i<DIM+1;i++)
    if (vertex == v[i])
      break;

  int index = i;

  return index;
}

inline Simplex * Simplex::getNeigh(int index)
{
  return neigh[index];
}

inline Simplex * Simplex::getNeigh(Vertex * vertex, int & top)
{
  int i;
  for (i=0;i<DIM+1;i++)
    if (vertex == v[i])
      break;

  top = i;

  return neigh[top];
}

inline void Simplex::getNeigh(int index, int top[], Simplex * t[])
{
  int i;
  int n=0;

  for (i=0;i<DIM+1;i++)
    if (i != index) {
      top[n] = i;
      t[n++] = neigh[i];
    }

  assert(n == DIM);

}

inline int Simplex::isMember(Vertex * vertex)
{
  int member=0;
  for (int i=0;i<DIM+1;i++)
    if (vertex == v[i]) {
      member = 1;
      break;
    }

  return member;
}

  
inline int Simplex::getNeigh(Simplex * s)
{
  int i;
  for (i=0;i<DIM+1;i++)
    if (s == neigh[i])
      break;

  assert(i<=DIM);

  int top = i;

  return top;
}

// The edgeNr edge of the edges in the sub simples having top as top node
inline Edge * Simplex::getEdge(int top, int edgeNr)
{
  return edge[edgeIndex[top][edgeNr]];
}

// The edgeNr edge of the edges having top as a vertex
inline Edge * Simplex::getEdge2(int top, int edgeNr)
{
  return edge[edgeIndex2[top][edgeNr]];
}

inline void Simplex::setEdge(int j, int k, Edge * e)
{
  assert(j != k);
  int index = edgeMatrix[j][k];
  edge[index] = e;
}

inline int Simplex::convex(Simplex * s, Point p)
{
  int convexPoly;
  int index=-1;

  for (int i=0; i<DIM+1; i++)
    if (neigh[i] == s) {
      index = i;
      break;
    }

  assert(index>=0);

  // vol(index) > 0 && vol(i)<0 for i != index
  int vol = volumeSign(perm[index], p, 1); //CHANGE$$

  if (vol < 0) { //CHANGE$$
    printf("Error(Simplex::convex): vol %d < 0\n", vol);
    print();
    assert(vol > 0); //CHANGE$$
    convexPoly = 0;
  }
  else {
    
    for (int i=0; i<DIM+1; i++)
    {
      if (i != index) {
	vol = volumeSign(perm[i], p);
	if (vol > 0)
	  break;
      }
    }
    convexPoly = (vol < 0);

  }

  return convexPoly;
}

inline Vertex ** Simplex::getVertex()
{
  return (Vertex **) v;
}

inline Vertex * Simplex::getVertex(int index)
{
  return v[index];
}

inline Point Simplex::getPt(int pos)
{
  return v[pos]->getPt();
}

inline Point Simplex::getMidPoint()
{
  return midPoint;
}

inline void Simplex::setNeigh(Simplex * sOld, Simplex * sNew)
{
  for (int i=0;i<DIM+1;i++)
    if (neigh[i] == sOld)
    {
      neigh[i] = sNew;
      break;
    }
}

inline void Simplex::changeOrientation()
{
  Vertex * tmp = v[0];
  v[0] = v[1];
  v[1] = tmp;
  Simplex * s = neigh[0];
  neigh[0] = neigh[1];
  neigh[1] = s;
}

inline int Simplex::oriented(int permIn[][DIM], int defaultOriented)
{
  int vol = 0;
  int defaultSign = 0;

  if (defaultOriented)
    defaultSign = -1;

  vol = volumeSign(permIn[0], v[0]->getPt(), defaultSign);

  return (vol<0);
}

inline int Simplex::inside(Point p)
{
  int vol = 0;

  for (int i=0; i<DIM+1; i++)
  {
    vol = volumeSign(perm[i], p);
    if (vol >= 0)
      break;
  }

  return (vol<0);
}

inline int Simplex::equal(double a, double b)
{
  int val=0;
  if (a-b>-equalLimit)
    if (a-b<equalLimit)
      val = 1;

  return val;
}

inline int Simplex::volumeSign(int * permIn, Point newP, int defaultSign)
{
  int i;
  int volSign = defaultSign;//CHANGE$$
  double vol=0.0;

  //  Vertex * v[DIM+1] = {v[perm[0]], v[perm[1]], v[perm[2]], v[perm[3]]};

  Point pt[DIM];
  for (i=0;i<DIM;i++)
    pt[i] =v[permIn[i]]->getPt();

  vol = findVolume(pt, newP);
  if (vol == 0) {
    double D[DIM];
    for (i=0;i<DIM;i++)//CHANGE$$
      D[i] = 0.0;

 
    while (vol == 0.0 || (defaultSign != 0 && (volSign != defaultSign))) {//CHANGE$$
      // Pertubate newP
      if (defaultSign != 0 && volSign != 0 && (volSign != defaultSign)) {
	for (i=0;i<DIM;i++) {	
	  newP[i] -= 2.0*D[i];
	  D[i] = -2.0*D[i];
	}
      }
      else {
	printf("WARNING(Simplex::volumeSign): vol == 0.0. Pertubates point.\n");
	
	int I = random->iUnif(0,DIM-1);
	double h = (random->unif01()-0.5)* pertEps;
	D[I] += h;
	newP[I] += h;
      }

      vol = findVolume(pt, newP);
      
      if (vol > 0.0)
	volSign = 1;
      else if (vol < 0.0)
	volSign = -1;
      else
	volSign = 0;
    }
  }

  if (vol > 0.0)
    volSign = 1;
  else if (vol < 0.0)
    volSign = -1;
  else
  {
    volSign = 0;
    printf("WARNING(Simplex::volumeSign): volSign == 0\n");
  }

  return volSign;
}

inline int Simplex::isDelaunay(Vertex * w)

{
  int delaunay = 1;

  double dist = distPoint(w->getPt(), centerpoint);
    
  if (radius > dist) {// NOT locally delaunay
    delaunay = 0;
  }

  return delaunay;
}

inline int Simplex::isDelaunay(Vertex ** w, int n)
{
  int delaunay = 1;

  for (int i=0;i<n;i++) {
    if (!isMember(w[i])) {
      double dist = distPoint(w[i]->getPt(), centerpoint);
    
      if (radius > dist) {// NOT locally delaunay	
	delaunay = 0;
	break;
      }
    }
  }

  return delaunay;
}

inline double Simplex::findVolume()
{
  Point pt[DIM];

  for (int i=0;i<DIM;i++)
    pt[i] = v[i]->getPt();

  Point newP = v[DIM]->getPt();

  double vol = findVolume(pt, newP);
  
  volume = -vol;

  assert(volume > 0.0);

  return volume;
}


inline void Simplex::initialise(Vertex * vertex[], int iter)
{
  for (int i=0;i<DIM+1;i++)
  {
    v[i]  = vertex[i];
    neigh[i] = NULL;
  }
  
  initialise(iter);
}


inline void Simplex::initialise(int iter)
{
  it = iter;
  
  removed = 0;
  visit = 0;
  inserted = 0;

  volume = 0.0;

  for (int j=0; j< DIM; j++)
    midPoint[j] = 0.0;

  double mult = (1.0/((double) DIM+1));
  for (int i=0; i< DIM+1; i++)
    for (int j=0; j< DIM; j++)
      midPoint[j] += mult * (*v[i])[j];  
}



inline void Simplex::findSphere()
{
  double A[DIM][DIM];
  double b[DIM];

  for (int i=0;i<DIM;i++)
  {
    b[i] = normPoint(v[i+1]->getPt()) - normPoint(v[0]->getPt());
    for (int j=0;j<DIM;j++)
    {
      A[i][j] = 2.0*((*v[i+1])[j]-(*v[0])[j]);
    }
  }

  Axb(A, b, centerpoint);

  radius = distPoint(v[0]->getPt(), centerpoint);
  
}

inline void Simplex::Axb(double A[][DIM], double b[], double x[])
{
  int i, j;
  int p[DIM];

  for (i=0;i<DIM;i++)
    p[i] = i;

  for (i=0;i<DIM-1;i++)
  {
    // Find maximum row element of column number i
    double maxx = fabs(A[p[i]][i]);
    int index = i;
    for (j=i+1;j<DIM;j++)
    {
      double h = fabs(A[p[j]][i]);
      if (h > maxx)
      {
	maxx = h;
	index = j;
      }
    }
    if (index > i)
    {
      int tmp = p[index];
      p[index] = p[i];
      p[i] = tmp;
    }

    for (j=i+1;j<DIM;j++)
    {
      double sub = - A[p[j]][i]/A[p[i]][i];
      for (int k=i;k<DIM;k++)
	A[p[j]][k] += sub * A[p[i]][k];
      b[p[j]] += sub * b[p[i]];
    }
  }
    
  // back substitution
  for (i=DIM-1;i>=0;i--)
  {
    double h = 1.0/A[p[i]][i];
    x[i] = b[p[i]] * h;
    for (j=i+1;j<DIM;j++)
      x[i] -= A[p[i]][j] * h * x[j];
  }

}


inline Simplex * SimplexList::New()
{
  Simplex * s;
  if (nRemoved > 0)
    s = removed[--nRemoved];
  else
  {
    if (n>=nAlloc) {
      assert(n<nAlloc); // NBNB May be avoid by having Simplex * list[100];
      // and then make a new list[i] then needed and having a 2D id
      //      nAlloc += nAllocDelta;
      //      list = (Simplex *) realloc(list, nAlloc*sizeof(Simplex));
    }
    s = &list[n++];
  }
  s->id = id;

  id++;

  return s;
}

inline void SimplexList::Free(Simplex * s)
{
  s->removed = 1;
  if (nRemoved>= nRemAlloc)
  {
    nRemAlloc += nRemAllocDelta;
    removed = (Simplex **) realloc(removed, nRemAlloc*sizeof(Simplex *));
  }
  removed[nRemoved++] = s;
}
  
#endif
