/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <assert.h>
//#include <iostream.h>

#include "random.h"
#include "basic.h"
#include "point.h"
#include "vertex.h"
#include "edge.h"
#include "simplex.h"
#include "delaunay.h"
#include "model.h"
#include "input.h"
#include "wells.h"
#include "simulation.h"


#define  G_MODULUS  256*256*256*128 
#define  G_INVMOD  ( (double) 1 / ((double) G_MODULUS )) / ( (double) 2 ) 


Simulation::Simulation(Model * mod, Input * input)
{
  int i;

  if (input->seedFileName == NULL)
    random = new RandomGenerator(*input->seed);
  else
    random = new RandomGenerator(input->seedFileName);

  model = mod;

  init = input->init;
  initRandom = input->initRandom;
  initNPoints = input->initNPoints;

  for (i=0;i<2*DIM;i++)
    regionOrig[i] = input->region[i];

  zAnisotropy = input->zAnisotropy;

  for (i=0;i<DIM;i++) {
    region[2*i] = 0.0;
    region[2*i+1] = (regionOrig[2*i+1] - regionOrig[2*i]);
  }
  region[ZMAX] *= zAnisotropy;

  volRegion = 1.0;
  for (i=0;i<DIM;i++)
    volRegion *= (region[2*i+1] - region[2*i]);

  // Corrects the intensity parameter
  double beta =  mod->Beta()/volRegion;
  mod->setBeta(beta);

  betaDeltaCum = new double[mod->NColour()];
  const double * db = mod->BetaDelta();
  if (db == NULL) {
    double factor = 1.0/mod->NColour();
    for (i=0;i<mod->NColour();i++) {
      betaDeltaCum[i] = (i+1.0)*factor;
    }
  }
  else {
    betaDeltaCum[0] = db[0];
    for (i=1;i<mod->NColour();i++) {
      betaDeltaCum[i] = betaDeltaCum[i-1] + db[i];
    }
  }

  assert(fabs(betaDeltaCum[mod->NColour()-1]-1.0)<0.000000001);
    
  betaDeltaCum[mod->NColour()-1] = 1.0;

  wellData = (input->nWell > 0);

  potVol = 0.0;
  potWell = 0.0;
  potImage = 0.0;

  if (input->volFraction) {
    volFraction = new double[model->NColour()];
    volFractionTolerance = new double[model->NColour()];
    for (i=0;i<model->NColour();i++) {
      volFraction[i] = input->volFraction[i];
      volFractionTolerance[i] = input->volFractionTolerance[i];
    }
  }
  else {
    volFraction = NULL;
  }

  if (wellData) {
    wells = new Wells(input, regionOrig);
  }
  else {
    wells = NULL;
  }

  imageData = (input->imageDim[0] > 0);

  simulateFromPosterior = (imageData || wellData);

  volumeAnnealing = (input->volDim[0]>0);

  path = input->path;

  if (path) {
    fileName = new char[strlen(path)+strlen(input->fileName)+1];
    strcpy(fileName, path);
    strcat(fileName, input->fileName);
  }
  else {
    fileName = new char[strlen(input->fileName)+1];
    strcpy(fileName, input->fileName);
  }
  pointSize = DIM * sizeof(double);

  nIter = input->nIter; 

  prob[ADD] = input->probs[ADD];
  prob[MOVE] = input->probs[MOVE];
  prob[REMOVE] = 1.0 -  prob[ADD] - prob[MOVE];

  burnIn = input->burnIn;

  startFromPrior = input->startFromPrior;

  debugPrint = input->debugPrint;
  debugCheck = input->debugCheck;

  newMPt = new mPoint;
  newMPt->pt = new double[DIM];

  estimateMarginalPosterior = input->estimateMarginalPosterior;

  sigmaMove = input->sigmaMove;

  delaunay = new Delaunay(debugCheck, volumeAnnealing,
			  model->Beta()*volRegion, random, region);

  sumPixelDiff = pixelDiff = 0;
  sumColDiff = 0;

  printTessInterval = input->printTessInterval;

  // ******** Initial simplex that cover the [0,1]^DIM cube
  startPoints = (mPoint *) malloc((DIM+1)*sizeof(mPoint));

  for (i=0;i<DIM+1;i++)
  {
    startPoints[i].pt = (Point) malloc(DIM*pointSize);
    startPoints[i].colour = 0;
  }

  double d[DIM];
  for (i=0;i<DIM;i++)
    d[i] = region[2*i+1]-region[2*i];
  
  if (DIM == 3) {
    startPoints[0].pt[0] = region[0] - 3.5*d[0];
    startPoints[0].pt[1] = region[2] - 1.0*d[1];
    startPoints[0].pt[2] = region[4];
   
    startPoints[1].pt[0] = region[0] + 4.5*d[0];
    startPoints[1].pt[1] = region[2] - 1.0*d[1];
    startPoints[1].pt[2] = region[4];

    startPoints[2].pt[0] = region[0] + 0.5*d[0];
    startPoints[2].pt[1] = region[2] + 3.5*d[1];
    startPoints[2].pt[2] = region[4];

    startPoints[3].pt[0] = region[0] + 0.5*d[0];
    startPoints[3].pt[1] = region[2] + 0.5*d[1];
    startPoints[3].pt[2] = region[4] + 3.0*d[2];
  }
  else if (DIM == 2) {
    startPoints[0].pt[0] = region[0] - 1.0*d[0];
    startPoints[0].pt[1] = region[2] - 0.5*d[1];
   
    startPoints[1].pt[0] = region[0] + 2.0*d[0];
    startPoints[1].pt[1] = region[2] - 0.5*d[1];

    startPoints[2].pt[0] = region[0] + 0.5*d[0];
    startPoints[2].pt[1] = region[2] + 2.0*d[1];
  }
  
 
  // ******** Log 

  char * name;


  logInterval = input->logInterval;
  logBinaer = input->logBinaer;

  if (logInterval == 0) {
    if (input->nIter>1000) {
      logInterval = input->nIter/1000;
    }
    else {
      logInterval = 1;
    }
  }

  if (input->nIter>20) {
    logIntervalIter = input->nIter/20;
  }
  else {
    logIntervalIter = 1;
  }


  if (input->restartFileName) {
    restartFileName = new char[strlen(input->restartFileName)+1];
    strcpy(restartFileName, input->restartFileName);
  }
  else
    restartFileName = NULL;

  name = (char *) malloc((strlen(fileName)+5)*sizeof(char));
  strcpy(name, fileName);
  logFile = fopen(strcat(name, ".log"), "w");  
  free(name);


  simFileName = (char *) malloc((strlen(fileName)+5)*sizeof(char));
  strcpy(simFileName, fileName);
  strcat(simFileName, ".sim");
  if (!restartFileName) {
    simFile = fopen(simFileName, "w");
    if (!simFile) {
      printf("(ERROR): Can not open file %s\n", simFileName);
      exit(-1);
    }
  }

  name = (char *) malloc((strlen(fileName)+5)*sizeof(char));
  strcpy(name, fileName);
  resFile = fopen(strcat(name, ".res"), "w");  
  free(name);

  if (estimateMarginalPosterior) {
    name = (char *) malloc((strlen(fileName)+6)*sizeof(char));
    strcpy(name, fileName);
    margFile = fopen(strcat(name, ".marg"), "w");  
    free(name);
    assert(margFile);
  }
  else {
    margFile = NULL;
  }

  if (!restartFileName) {
    int n = nIter/logInterval;
    fprintf(simFile, "%d %d %d\n", n, nIter, logInterval);
  }

  if (estimateMarginalPosterior) {
    delaunay->initialiseMarginalPosterior(mod->NColour(), input->margGridDim,
					  burnIn);
  }

  if (imageData) {
    delaunay->initialiseImageData(input->imageFileName,
				  input->trueImageFileName,
				  input->imageDim,
				  region);
  }

  Tstart = 1.0;
  double Tstop = 0.0001;
  Trate = pow(Tstop/Tstart, 1.0/nIter);
  if (volumeAnnealing) {
    delaunay->initialiseVolFraction(mod->NColour(), input->volDim,
				    burnIn);
  }

  return;
}

Simulation::~Simulation() {
  delete delaunay;
  
  delete random;

  if (wells)
    delete wells;

  delete [] newMPt->pt;
  delete newMPt;

  for (int i=0;i<DIM+1;i++)
  {
    free(startPoints[i].pt);
  }
  free(startPoints);

  delete [] fileName;

  if (volFraction) {
    delete [] volFraction;
    delete [] volFractionTolerance;
  }

  free(simFileName);
  if (restartFileName)
    delete [] restartFileName;

  // ******** Information to file
  fclose(logFile);
  fclose(simFile);
  fclose(resFile);
  if (estimateMarginalPosterior) {
    fclose(margFile);
  }
}

void Simulation::simulation() {
  int startIt=0;

  delaunay->firstSimplex(startPoints, wells);

  sumAlpha = 0.0;

  if (restartFileName)
    restart(startIt);
  else if (init) {
    delaunay->setIt(-1);

    initMH(); // NBNB Check this
  }
 
  if (wellData)
    potWell = potentialFromWellData();

  if (startIt > burnIn && estimateMarginalPosterior)
    delaunay->initialiseMarginalPosterior();

  if (imageData) {
    delaunay->initialiseImageData();
    potImage = potentialFromImageData();
  }

  if (volumeAnnealing) {
    delaunay->initialiseVolFraction();
    potVol = potentialFromVolFraction(0);
  }

  // ********** SIMULATION LOOP ************
  for (int it=startIt; it<nIter; it++) {

    printIteration(it);

    noMove = 0;
    errorInDelaunay = 0;

    delaunay->setIt(it);

    if (volFraction)
      potVol = potentialFromVolFraction(it);

    //choose action: ADD, REMOVE, MOVE
    drawAction();
    if (action == ADD) {
      addProposal();
    }
    else if (action == REMOVE) {
      removeProposal();
    }
    else { // MOVE
      moveProposal();
      action = MOVE;
    }

    if (errorInDelaunay) {
      delaunay->restore((int) model->Beta()*volRegion, random, wells, region);
      noMove = 1;
    }

    if (!noMove) {
      double pot = acceptPotential(it);

      accept = acceptance(pot);

      if (!accept)
	delaunay->rejectLastUpdate(wells);
      else {
	int N0=0;
	int N1=0;
	double L0=0.0;
	if (wells) {
	  wells->getStatistics(N0, N1, L0);
	}
	delaunay->acceptLastUpdate();
	if (imageData)
	  potImage = potentialFromImageData();
	if (wellData)
	  potWell = potentialFromWellData();
	if (debugPrint) {
	  printf("N0 %3d N1 %3d L0 %8.4f potImage %8.4f potWell %8.4f potVol %8.4f\n",
		 N0, N1, L0, potImage, potWell, potVol);
	}
	updateLogInfo(it);
      }
    }

    if (estimateMarginalPosterior && (it == burnIn)) {
      delaunay->initialiseMarginalPosterior();
    }
    
    if (((it+1) % printTessInterval) == 0 && (it+1) != nIter)
      printTessellation(it+1);
    
    printSimInfo(it);
  }

  printf("Final check of the Delaunay tessellation\n");
  delaunay->checkConsistency(wells);

  delaunay->updateMarginalPosteriorFinal();

  if (estimateMarginalPosterior)
    delaunay->printMarginalPosterior(margFile, nIter);
  
  delaunay->printLogInfo(logFile);
  if (wells)
    wells->printLogInfo(logFile);

  printTessellation(nIter);

  printRes();
}

void Simulation::printRes()
{
  for (int i=0;i<2*DIM;i++)
    fprintf(resFile, "%12.2f", region[i]);
  fprintf(resFile, "\n");

  for (int i=0;i<2*DIM;i++)
    fprintf(resFile, "%12.2f", regionOrig[i]);
  fprintf(resFile, "\n");
  
  fprintf(resFile, "%12.2f\n", zAnisotropy);

  delaunay->printRes(resFile);
}

void Simulation::initMH() {
  int i;

  for (i=0; i<initNPoints; i++) {
    //Draw a random number in R^{DIM}
    drawPoint();

    if (!initRandom)
      newMPt->colour = 0;

    delaunay->insert(*newMPt, wells);
  }

  sumColDiff = delaunay->initColDiff();

  return;
}

void Simulation::addProposal() {
  drawPoint(); // New point stored in newMPt

  delaunay->insert(*newMPt, wells);
  //  delaunay->insertData(wells);
}


void Simulation::removeProposal()
{
  Vertex * v = delaunay->selectVertex(*random);

  errorInDelaunay = delaunay->remove(v, wells);
  //  delaunay->removeData(wells);
}

void Simulation::moveProposal()
{
  Vertex * v = delaunay->selectVertex(*random);

  movePoint(v); // New point stored in newMPt

  if (inside(newMPt->pt)) {
    delaunay->move(v, *newMPt, wells);
  }
  else {
    noMove = 1;
  }
}

void Simulation::updateLogInfo(int it) {
  sumPixelDiff += pixelDiff;
  sumColDiff += colDiff;

  if (estimateMarginalPosterior && it >= burnIn) {
    delaunay->updateMarginalPosterior(action);
  }

  return;
}


int Simulation::acceptance(double pot) {
  double alpha;

  if (noMove) {
    alpha = 0.0;
    accept = 0;
  }
  else {
    alpha = exp(-pot); // NBNB Sjekk for store verdier av |pot|

    if (alpha > 1.0)
      alpha = 1.0;

    double u = random->unif01();
    accept = (u<alpha);
  }
  sumAlpha += alpha;
   
  return accept;
}
  


void Simulation::drawAction() {
  int nPoints = delaunay->getNPoints();
  double u;

  if (nPoints <= DIM+1) {
    action = ADD;
  }
  else {
    u = random->unif01();
    if (u<prob[ADD])
      action = ADD;
    else if (u<prob[MOVE]+prob[ADD])
      action = MOVE;
    else
      action = REMOVE;
  }

#ifdef OUT
  if (nPoints < 400)
    action = ADD; //NBNBNB Temporary...
  else if (nPoints>=500)
    action = REMOVE; //NBNBNB Temporary...
#endif

  printAction();

  return;
}

// draws point uniformely over region S
void Simulation::drawPoint()
{
  int i;
  for (i=0; i<DIM; i++) {
    newMPt->pt[i] = random->unif01();
    newMPt->pt[i] += (random->unif01()-0.5) * G_INVMOD;
    newMPt->pt[i] = region[2*i] + newMPt->pt[i]* (region[2*i+1]-region[2*i]);
  }
  double u = random->unif01();
  for (i=0; i<model->NColour(); i++) {
    if (u <= betaDeltaCum[i]) {
      newMPt->colour = i;
      break;
    }
  }

}

void Simulation::movePoint(Vertex * v)
{  
  int i;
#ifdef OUT  //NBNB May be used in future implementation???
  // Choose simplex from v->simp[..] according to volume size
  int n;

  double sumVol;
  double * vol;
  v->getVolume(vol, n, sumVol);

  double u = sumVol * random->unif01();
  double cumVol = vol[0];
  for (i=0;i<n-1;i++) {
    if (cumVol > u)
      break;
    cumVol += vol[i+1];
  }

  int index = i;

  Simplex * s = v->getSimp(index);

  s->drawPoint(*random, newMPt->pt);
  
#endif
  Point pt = v->getPt();

  movePot = (DIM/2.0)*log(2.0 * G_PI) + DIM * log(sigmaMove);

  for (int i=0;i<DIM;i++) {
    newMPt->pt[i] = random->normal(pt[i], sigmaMove);

    // Compute the move potential NBNB Necessary??
    double d = (newMPt->pt[i] - pt[i])/sigmaMove;
    movePot += d*d/2.0;
  }
  
  double u = random->unif01();
  for (i=0; i<model->NColour(); i++) {
    if (u <= betaDeltaCum[i]) {
      newMPt->colour = i;
      break;
    }
  }  
  
  //  newMPt->colour = random->iUnif(0,model->NColour()-1);
}


double Simulation::acceptPotential(int it)
{
  double pot = potentialFromPrior();
  double potNew;

  if (volFraction) {
    potNew = potentialFromVolFraction(it);
    pot += (potNew - potVol);
  }

  if (simulateFromPosterior && it >= startFromPrior) {
    pot += potentialFromLikelihood();
  }

  return pot;
}


double Simulation::potentialFromPrior()
{
  double pot = updateFirstOrder();
  int nPoints = delaunay->getNPoints();

  if (action != MOVE) { // Compute the potential from nPoints
    int n = nPoints;

    if (action == REMOVE)
      n++;

    double r = model->Beta() * prob[REMOVE]/prob[ADD];
    r /= (double) n;

    r *= volRegion;

    pot -= log(r);

    if (action == REMOVE) {
      pot *= -1;
    }
  }

  return pot;
}


double Simulation::potentialFromLikelihood()
{
  double pot = 0.0;
  double potNew;

  if (imageData) {
    potNew = potentialFromImageData();
    pot += (potNew - potImage);
  }

  if (wellData) {
    potNew = potentialFromWellData();

    pot += (potNew - potWell);
  }

  return pot;
}


double Simulation::potentialFromVolFraction(int it)
{
  double pot = 0.0;
  double sigma = 1.0;
  double T = sigma * pow(Trate, it);


  const double * volReal = delaunay->volFraction();

  double s = sigma * T;
  double s2 = s*s;
  for (int i=0;i<model->NColour();i++) {
    double d = volReal[i] - volFraction[i];

    if (fabs(d) < volFractionTolerance[i])
      d = 0.0;
    else
      d = d - volFractionTolerance[i];

    pot += d*d/s2;
  }

  return pot;
}

double Simulation::potentialFromImageData()
{
  double pot = delaunay->SumColDiff()/(2.0 * model->ImageSigma());

  return pot;
}


double Simulation::potentialFromWellData()
{
  int N0;
  int N1;
  double L0;
  const double * mu = model->Mu();
  const double * logMu = model->LogMu();

  if (wells)
    wells->getStatistics(N0, N1, L0);

  double pot = -logMu[0]*N0 - logMu[1]*N1 + (mu[0]-mu[1])*L0;

  return pot;
}


double Simulation::updateFirstOrder()
{
  colDiff = delaunay->sumFirstOrderColDiff(action);

  if (action == REMOVE)
    colDiff *= -1;

  double pot = model->Theta() * ((double) colDiff);

  return pot;
}


void Simulation::printSimInfo(int it)
{
  if ((it+1) % logInterval == 0) {
    int nPoints = delaunay->getNPoints();

    int N0=0;
    int N1=0;
    double L0=0.0;
    if (wells) {
      wells->getStatistics(N0, N1, L0);
      double wellLength = wells->getTotalWellLength();
      L0 /= wellLength;
    }      
    fprintf(simFile, "%5d %5d %5d %5d %5d %8.4f %8.4f %8.4f %8.4f %6.4f\n",
	    it, nPoints, sumColDiff,
	    N0, N1, L0, potImage, potWell, potVol, sumAlpha/logInterval);

    fflush(simFile);
    sumAlpha = 0.0;
  }
}

void Simulation::printTessellation(int it)
{
  char * name = (char *) malloc((strlen(fileName)+6)*sizeof(char));
  strcpy(name, fileName);
  FILE * tessFile = fopen(strcat(name, ".tess"), "w");
  free(name);

  fprintf(tessFile, "%u\n", random->getSeed());

  delaunay->printTessellation(tessFile, it);

  fclose(tessFile);
}

void Simulation::restart(int & startIt)
{
  int i;
  int j;
  int nPoints;
  FILE * restartFile = fopen(restartFileName, "r");

  int seed;
  fscanf(restartFile, "%d %d %d", &seed, &startIt, &nPoints);

  unsigned int uiseed = (unsigned int) seed;

  random->setSeed(uiseed);

  printf("Restart delaunay triangulation, start iteration %d\n", startIt);

  mPoint * mPt = new mPoint[nPoints];
  for (i=0;i<nPoints;i++) {
    mPt[i].pt = new double[DIM];
    for (j=0;j<DIM;j++) {
      fscanf(restartFile, "%lf", &(mPt[i].pt[j]));
    }
    fscanf(restartFile, "%hd", &(mPt[i].colour));
  }

  delaunay->setIt(startIt-1);

  delaunay->restart(restartFile, mPt, nPoints, wells, estimateMarginalPosterior);

  fclose(restartFile);

  sumColDiff = delaunay->initColDiff();

  for (i=0;i<nPoints;i++) {
    delete [] mPt[i].pt;
  }

  delete [] mPt;
  

  // Read sim file
  simFile = fopen(simFileName, "r");
  

  const int nColumns = 10; // NBNB Hard coded value

  int n;
  int nIterSim;
  int logIntervalSim;
  fscanf(simFile, "%d %d %d\n", &n, &nIterSim, &logIntervalSim);
  
  int nRead = (int) ((startIt+1.0)/((double) logIntervalSim));

  // Loop 1: Determine the number of records
  double ** record;
  if (nRead > 0) {
    record = new double * [nRead];
    for (i=0;i<nRead;i++) {
      record[i] = new double[nColumns];
      for (j=0;j<nColumns;j++)
	fscanf(simFile, "%lf", &record[i][j]);    
    }
  }

  fclose(simFile);

  simFile = fopen(simFileName, "w");
  fprintf(simFile, "%d %d %d\n", n, nIter, logInterval);
  
  // NBNB Hard coded format, must be the same as in Simulation::printSimInfo
  for (i=0;i<nRead;i++) {
    fprintf(simFile, "%5d %5d %5d %5d %5d %8.4f %8.4f %8.4f %8.4f %6.4f\n",
	    (int) record[i][0], (int) record[i][1], (int) record[i][2],
	    (int) record[i][3], (int) record[i][4],
	    record[i][5], record[i][6], record[i][7],
	    record[i][8], record[i][9]);
  }
  fflush(simFile);

  if (nRead > 0) {
    for (i=0;i<nRead;i++) {
      delete [] record[i];
    }
    delete [] record;
  }
}
