/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <errno.h>
#include <assert.h>


#include "random.h"
#include "basic.h"
#include "point.h"
#include "vertex.h"
#include "wells.h"
#include "simplex.h"

int ** Vertex::sumMarg;
int * Vertex::lastMargUpdate;
int * Vertex::lastColour;

void Vertex::printAll() {
  printf("(%6.4f %6.4f %6.4f) col %d clDiff %d clDNew %d it %d \n",
	 pt[0], pt[1], pt[2], colour, colDiff, colDiffNew, it);
  return;
}

void Vertex::print()
{
  print(stdout);
}

void Vertex::print(FILE * file)
{
  fprintf(file, "(%6.4f %6.4f %6.4f) C %d\n", pt[0], pt[1], pt[2], colour);
  return;
}

void Vertex::printTessellation(FILE * file)
{
  int i;
  for (i=0;i<DIM;i++)
    fprintf(file, "%6.4f ", pt[i]);

  fprintf(file, "%d\n", colour);
}

void Vertex::printFull()
{
  printFull(stdout);
}

void Vertex::printFull(FILE * file)
{
  fprintf(file, "(%6.4f %6.4f %6.4f) C %d Cd %d Cd new %d\n",
	  pt[0], pt[1], pt[2], colour, colDiff, colDiffNew);
  return;
}

void Vertex::printRes(VertexList * vList, FILE * file)
{
  int i;
  for (i=0;i<DIM;i++)
    fprintf(file, "%6.4f ", pt[i]);
  fprintf(file, "%d ", colour);
  fprintf(file, "%d ", nNeigh);
  for (i=0;i<nNeigh;i++)
    fprintf(file, "%d ", vList->pos(neigh[i]));
  fprintf(file, "\n");
  return;
}

void Vertex::printMarkedPoint(FILE * file)
{
  int i;
  for (i=0;i<DIM;i++)
    fprintf(file, "%6.4f ", pt[i]);
  fprintf(file, "%d\n", colour);
}

void Vertex::printNeigh()
{
  int i;

  for (i=0;i<nNeigh;i++)
    neigh[i]->print();

  return;
}

void Vertex::printNeigh(VertexList * vList)
{
  printNeigh(vList, stdout, 0);
  printf("Id's: ");
  for (int i=0;i<nNeigh;i++)
    printf("%d ", neigh[i]->getId());
  printf("\n");
}

void Vertex::printNeighC(VertexList * vList)
{
  printNeigh(vList, stdout, 1);
}


void Vertex::printNeigh(VertexList * vList, FILE * file, int printColour)
{
  int i;

  fprintf(file, "Vertex %d ", vList->pos(this));
  if (printColour)
    fprintf(file, "(%d) ", colour);
  fprintf(file, "nNeigh %d, ", nNeigh);
  for (i=0;i<nNeigh;i++) {
    fprintf(file, "%d ", vList->pos(neigh[i]));
    if (printColour)
      fprintf(file, "(%d) ", neigh[i]->colour);
  }

  fprintf(file, "\n");

  return;
}

void Vertex::printSimp() {
  int i;

  for (i=0;i<nSimp;i++)
    simp[i]->print();

  return;
}

void Vertex::printSimp(VertexList * vList)
{
  printSimp(vList, stdout);
}

void Vertex::printSimp(VertexList * vList, FILE * file)
{
  int i;

  for (i=0;i<nSimp;i++)
    simp[i]->print(vList, file);

  return;
}

void Vertex::printSegments()
{
  for (int i=0;i<nSegment;i++) {
    printf("%d %d\n", segment[i][0], segment[i][1]);
  }
}

VertexList::VertexList(int nMeanVertices)
{
  n = 0;
  nAlloc = 20 * nMeanVertices;
  nAllocDelta = nAlloc;
  nRemoved = 0;
  nRemAlloc = 10000;
  nRemAllocDelta = nRemAlloc;

  id = 0;

  imageInit = 0;
  volInit = 0;

  nMarg = 0;
  margGridPt = NULL;
  sumMarg = NULL;
  lastMargUpdate = NULL;
  lastColour = NULL;
  initialisedMarg = 0;

  imageTrueColour = NULL;
  imageGridPt = NULL;
  
  volGridPt = NULL;

  list = (Vertex *) malloc(nAlloc*sizeof(Vertex));
  removed = (Vertex **) malloc(nRemAlloc*sizeof(Vertex *));
}

VertexList::~VertexList()
{
  int i;
  for (i=0;i<n;i++) {
    Vertex * v = &list[i];
    v->nullify();
  }

  free(list);
  free(removed);

  if (lastMargUpdate) {
    delete [] lastMargUpdate;
    delete [] lastColour;
    for (i=0;i<nMarg;i++) {
      delete [] sumMarg[i];
      delete [] margGridPt[i];
    }
    delete [] sumMarg;
    delete [] margGridPt;
  }

  if (imageTrueColour) {
    delete [] imageTrueColour;
    delete [] imageColour;

    for (i=0;i<nImage;i++) {
      delete [] imageGridPt[i];
    }
    delete [] imageGridPt;
  }

  if (volGridPt) {
    delete [] sumVol;
    delete [] volFraction;

    for (i=0;i<nVol;i++) {
      delete [] volGridPt[i];
    }
    delete [] volGridPt;
  }
}

void VertexList::print()
{
  print(stdout);
}

void VertexList::print(FILE * file)
{
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      fprintf(file, "%d ", pos(v));
      v->print(file);
    }
  }
}

void VertexList::printSegments()
{
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      printf("Vertex %d:\n", pos(v));
      v->printSegments();
    }
  }
}



void VertexList::printFull()
{
  printFull(stdout);
}

void VertexList::printFull(FILE * file)
{
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      fprintf(file, "%d ", pos(v));
      v->printFull(file);
    }
  }
}

void VertexList::printRes(FILE * file) // Write to the result file
{
  fprintf(file, "%d %d\n", n, n-nRemoved);
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      fprintf(file, "%d ", pos(v));
      v->printRes(this, file);
    }
  }
}

void VertexList::printMarginalPosterior(FILE * file, int nIter)
{
  int i;
  int j;

  for (i=0;i<DIM;i++)
    fprintf(file, "%d ", margGridDim[i]);
  fprintf(file, "%d ", nColour);
  fprintf(file, "\n");

  int p = (nIter - burnIn-1);
  double factor = 1.0/((double) p);

  for (i=0;i<nMarg;i++) {
    int sum=0;
    for (j=0;j<nColour;j++) {
      sum += sumMarg[i][j];
    }
    if (sum != p) {
      printf("Error((VertexList::printMarginalPosterior): sum %d != (nIter-burnIn-1) %d",
	     sum, p);
      assert(0);
    }
  }
  for (j=0;j<nColour;j++) {
    for (i=0;i<nMarg;i++)
      fprintf(file, "%8.6f ", factor*sumMarg[i][j]);
    fprintf(file, "\n");
  } 
}


void VertexList::printNeigh()
{
  printNeigh(stdout);
}

void VertexList::printNeigh(FILE * file)
{
  printNeigh(file, 1);
}

void VertexList::printNeighC()
{
  printNeigh(stdout, 1);
}

void VertexList::printNeigh(FILE * file, int printColour)
{
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      fprintf(file, "v %d: ", pos(v));
      v->printNeigh(this, file, printColour);
    }
  }
}

void VertexList::checkConsistency(short * nSimpInVert)
{
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      assert(v->nSimp == nSimpInVert[pos(v)]);
    }
  }
}


void VertexList::checkConsistency()
{
  int * occupied = new int[n];

  for (int i=DIM+1; i<n;i++) {
    Vertex * v = &list[i];

    if (!v->removed) {
      if ((DIM == 2 && v->nNeigh != v->nSimp) ||
	  (DIM == 3 && ((2*v->nNeigh-4) != v->nSimp))) {
	printf("Error(VertexList::checkConsistency) ");
	printf(" vertex %d:  ", pos(v));
	printf("# neigh %d, #simp %d\n", v->nNeigh, v->nSimp);
	v->printNeigh(this);
	v->printSimp(this);
	assert(0);
      }
      
      int colDiff = 0;
      for (int j=0;j<v->nNeigh;j++) {
	if (!v->neigh[j]->onBoundary())
	  colDiff += (v->colour != v->neigh[j]->colour);
      }
      if (colDiff != v->colDiffNew) {
	printf("Error(VertexList::checkConsistency) ");
	printf(" vertex %d:  with neighbours\n", pos(v));
	v->printNeigh(this);
	assert(0);
      }

      int j;
      for (j=0; j<n;j++) {
	occupied[j] = 0;
      }
      for (j=0;j<v->nNeigh;j++) {
	int index = pos(v->neigh[j]);
	if (occupied[index]) {
	  printf("Error(VertexList::checkConsistency) ");
	  printf(" vertex %d:  has duplicated neighbour %d\n", pos(v), index);
	  printf(" Neighbour list:\n");
	  v->printNeigh(this);
	  assert(0);
	}
	else
	  occupied[index] = 1;
      }

      for (j=0;j<v->nNeigh;j++) {
	if (!v->neigh[j]->isNeigh(v)) {
	  printf("Error(VertexList::checkConsistency) ");
	  printf("vertex %d not neighbour of %d\n", pos(v), pos(v->neigh[j]));
	  printf(" Neighbour list 1:\n");
	  v->printNeigh(this);
	  printf(" Neighbour list 2:\n");
	  v->neigh[j]->printNeigh(this);
	  assert(0);
	}
      }
      
    }
  }


  delete [] occupied;
}

void VertexList::checkVolumeFraction()
{
  if (volInit) {
    int sumVolData = 0;
    for (int i=0; i<n;i++) {
      Vertex * v = &list[i];

      if (!v->removed) {
	if (v->nVolData < 0) {
	  printf("Error(VertexList::checkConsistency) ");
	  printf(" vertex %d:  has negative # volume data\n", pos(v));
	  assert(0);
	}

	sumVolData += v->nVolData;
      }
    }

    if (sumVolData != nVol) {
      printf("Error(VertexList::checkConsistency) ");
      printf("Sum of volume data (%d) != total number of volume data (%d)\n",
	     sumVolData, nVol);
      assert(0);
    }
  }
}

void VertexList::checkImage()
{
  if (imageInit) {
    int sumImageData = 0;
    for (int i=0; i<n;i++) {
      Vertex * v = &list[i];

      if (!v->removed) {
	if (v->nImageData < 0) {
	  printf("Error(VertexList::checkConsistency) ");
	  printf(" vertex %d:  has negative # volume data\n", pos(v));
	  assert(0);
	}

	sumImageData += v->nImageData;
      }
    }

    if (sumImageData != nImage) {
      printf("Error(VertexList::checkConsistency) ");
      printf("Sum of image data (%d) != total number of image data (%d)\n",
	     sumImageData, nImage);
      assert(0);
    }
  }
}

void VertexList::checkMarginalPosterior()
{
  if (initialisedMarg) {
    int sum = 0;
    for (int i=DIM+1; i<n;i++) {
      Vertex * v = &list[i];
      
      if (!v->removed) {
	sum += v->nMargData;
      }
    }

    if (sum != nMarg) {
      printf("ERROR(VertexList::checkConsistency): sum %d != nMarg %d\n",
	     sum, nMarg);
      printMargData();
      assert(0);
    }
  }

}


void VertexList::printMargData()
{
  for (int i=0; i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      printf("Vertex %d nMargData %d\n", i, v->nMargData);
    }
  }
}

void VertexList::printTessellation(FILE * file)
{
  int i;
  int j;

  for (i=DIM+1; i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed)
      v->printTessellation(file);
  }

  for (i=0;i<nMarg;i++) {
    fprintf(file, "%d ", lastMargUpdate[i]);
    fprintf(file, "%d ", lastColour[i]);
    for (j=0;j<nColour;j++)
      fprintf(file, "%d ", sumMarg[i][j]);
    fprintf(file, "\n");
  }

}


void VertexList::restartMarg(FILE * file) //CHANGE$$
{
  for (int i=0;i<nMarg;i++) {
    fscanf(file, "%d", &lastMargUpdate[i]);
    fscanf(file, "%d", &lastColour[i]);
    for (int j=0;j<nColour;j++)
      fscanf(file, "%d", &sumMarg[i][j]);
  }
}



void VertexList::checkAcceptedConsistency()
{
  for (int i=DIM+1; i<n;i++) {
    Vertex * v = &list[i];

    if (!v->removed) {
      if (v->colDiff != v->colDiffNew) {
	printf("Error(VertexList::checkAcceptedConsistency) ");
	printf(" vertex %d:  with neighbours\n", pos(v));
	v->printNeigh(this);
	assert(0);
      }
    }
  }
}


Vertex::Vertex(Point vert, short col, int iter) {
  initialise(vert, col, iter);
}

void Vertex::initialise(Point vert, short col, int iter) {
  nSimp =0;
  nDeltaAlloc = 100;
  nSimpAlloc = nDeltaAlloc;
  nNeighAlloc = nDeltaAlloc;
  simp = (Simplex **) malloc(nSimpAlloc*sizeof(Simplex *));
  nNeigh = 0;
  neigh = (Vertex **) malloc(nNeighAlloc*sizeof(Vertex *));
  removed = 0;

  pt = (Point) malloc(DIM*sizeof(double));
  int i;
  for (i=0;i<DIM;i++)
    pt[i] = vert[i];
  this->colour = col;
  this->it = iter;
  visit = -1;
  colDiff = 0;
  colDiffNew = 0;

  //  lastMargUpdate = max(it, burnIn);

  nSegment = 0;
  nSegmentAlloc = nDeltaAlloc;
  segment = (int **) malloc(nSegmentAlloc * sizeof(int *));
  for (i=0;i<100;i++)
    segRemoved[i] = 0;

  nMargData = 0;
  nMargAlloc = 0;
  margData = NULL;

  nImageData = 0;
  nImageAlloc = 0;
  imageData = NULL;

  nVolData = 0;
  nVolAlloc = 0;
  volData = NULL;
}


Vertex::~Vertex() {
  nullify();
}

void Vertex::nullify() {
  nSimp = 0;
  nNeigh = 0;
  free(simp);
  free(neigh);
  free(pt);

  for (int i=0;i<nSegment;i++)
    delete [] segment[i];
  free(segment);
  nSegment = 0;
  if (margData)
    free(margData);
  if (volData)
    free(volData);
  if (imageData)
    free(imageData);
  nVolData = 0;
}

int Vertex::sumFirstOrderColDiff(int action, int mark)
{
  int totColDiff = 0;

  if (action == ADD)
    totColDiff = colDiffNew;
  else
    totColDiff = -colDiff;

  // Update neighbours colDiff

  for (int i=0;i<nNeigh;i++) {
    Vertex * w = neigh[i];
    if (!w->onBoundary() && (mark == 0 || w->visit != mark)) {
      totColDiff += (w->colDiffNew - w->colDiff);

      w->visit = mark;
    }
  }


  return totColDiff;
}

void Vertex::getVolume(double *& vol, int & n, double & sumVol)
{
  vol = new double[nSimp];
  sumVol = 0.0;
  for (int i=0;i<nSimp;i++) {
    Simplex * s = simp[i];
    if (s->getVolume() == 0.0)
      vol[i] = s->findVolume();
    else
      vol[i] = s->getVolume();
    sumVol += vol[i];
  }
  n = nSimp;
}

void Vertex::findNeigh(Vertex **& v, int & nV, int & mark)
{
  Simplex * s= simp[0];

  nV = 0;

  int top;
  int index = -1;
  for (int j=0;j<DIM+1;j++) {
    Vertex * w = s->getVertex(j);
    if (w != this) {
      index = j;
      v[nV++] = w;
    }
  }

  assert(index >=0);

  mark++;
  s->setVisit(mark);

  s = s->getNeigh(s->getVertex(index), top);

  if (!s)
    return;

  // Keep a stack of incident simplices
  static Simplex **st;
  static long ss = MAXDIM;
  long tms = 0;

  initStackMemory();
  push(s);

  int n = 1;
  while (tms) {
    pop(s);
    
    if (!s->isVisited(mark)) {
    
      int indexV = -1;
      int j;
      for (j=0;j<DIM+1;j++) {
	Simplex * t = s->getNeigh(s->getVertex(j), top);
	if (t && t->isVisited(mark)) { // j == top
	  indexV = j;
	  break;
	}
      }

      assert(indexV >= 0);

      Vertex * vNew = s->getVertex(indexV);
      int found = 0;
      for (int k=0;k<nV;k++) {
	if (vNew == v[k]) {
	  found = 1;
	  break;
	}
      }

      if (!found)
	v[nV++] = vNew;

      s->setVisit(mark);

      for (j=0;j<DIM+1;j++) {
	if (s->getVertex(j) != this && !s->getNeigh(j)->isVisited(mark))
	  push(s->getNeigh(j));
      }
      n++;
    }
  }

  assert(n == nSimp);
  assert((nV*2-4) == nSimp);
  assert(nV <= 100);
}

void Vertex::insertNeigh(int & mark) //Called from firstSimplex
{
  findNeigh(neigh, nNeigh, mark);
}

void Vertex::insertNeigh(Vertex ** v, int & mark) //Called from Connect
{
  for (int i=0;i<DIM+1;i++) {
    v[i]->insertNeigh(this);
    insertNeigh(v[i]);
  }
}

void Vertex::insertNeigh(Vertex * v) //Called from flipTriangle and insertNeigh
{
  if (nNeigh == nNeighAlloc) {
    nNeighAlloc += nDeltaAlloc;
    neigh = (Vertex **) realloc(neigh, nNeighAlloc*sizeof(Vertex *));
  }
  neigh[nNeigh++] = v;
  if (!v->onBoundary())
    colDiffNew += (v->colour != colour);
}

void Vertex::removeNeigh() //Called from disConnect
{
  assert(nNeigh == (DIM+1));

  for (int i=0;i<nNeigh;i++) {
    neigh[i]->removeNeigh(this);
  }

  colDiffNew = 0;
  nNeigh = 0;
}

void Vertex::removeNeigh(Vertex * v) //Called from flipEdge and removeNeigh
{
  int i;
  int index = -1;;
  for (i=0;i<nNeigh;i++) {
    if (neigh[i] == v) {
      index = i;
      break;
    }
  }

  assert(index>=0);

  if (!v->onBoundary())
    colDiffNew -= (v->colour != colour);

  for (i=index+1;i<nNeigh;i++)
    neigh[i-1] = neigh[i];

  nNeigh--;
}

int VertexList::initColDiff() // Called from initMH
{
  int sumColDiff = 0;

  for (int i=DIM+1;i<n;i++) {
    Vertex * v = &list[i];

    int col = v->colour;
    int sum = 0;
    for (int j=0;j<v->nNeigh;j++) {
      if (!v->neigh[j]->onBoundary())
	sum += (col != v->neigh[j]->colour);
    }

    v->colDiff = (short) sum;

    sumColDiff += sum;
  }

  assert((sumColDiff % 2) == 0);

  sumColDiff /= 2;

  return sumColDiff;
}


void VertexList::findNearestVertex(Point pt, Vertex *& v)
{
  int i;
  double distMin=DBL_MAX;
  Vertex * vMin = NULL;

  if (!v) {
    for (i=0;i<n;i++) {
      Vertex * w = &list[i];
      if (!w->removed) {
	double dist = distPoint(w->pt, pt);
	
	if (dist<distMin) {
	  vMin = w;
	  distMin = dist;
	}
      }
    }
  }
  else {
    for (i=0;i<v->nNeigh;i++) {
      Vertex * w = v->neigh[i];
      double dist = distPoint(w->pt, pt);
	
      if (dist<distMin) {
	vMin = w;
	distMin = dist;
      }
    }
  }

  assert(vMin);

  v = vMin;
}

void Vertex::removeSegment(int seg[])
{
  int i;
  int index =-1;
  for (i=0;i<nSegment;i++) {
    if (!segRemoved[i]) {
      if (segment[i][0] == seg[0] && segment[i][1] == seg[1]) {
	index = i;
	break;
      }
    }
  }

  assert(index>=0);

  segRemoved[index] = 1;
}

void Vertex::clearSegment()
{
  int index = 0;

  for (int i=0;i<nSegment;i++)
  {
    if (segRemoved[i]) {
      delete [] segment[i];
      segRemoved[i] = 0;
    }
    else {
      segment[index++] = segment[i];
    }
  }
  nSegment = index;
}

void Vertex::insertSegment(int seg[])
{
  assert(nSegment<100); //NBNB
  segment[nSegment] = new int[2];
  segment[nSegment][0] = seg[0];
  segment[nSegment][1] = seg[1];
  nSegment++;
}
  
void Vertex::initialiseSegments(Wells * wells)
{
  wells->getAllSegments(segment, nSegment);
  wells->initialiseSegments(this);
}


int VertexList::totalNumberOfSegments()
{
  int sumSegment = 0;
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      sumSegment += v->nSegment;
    }
  }

  return sumSegment;
}

int Vertex::isNearest(Point ptIn)
{
  int nearest = 1;
  double dist = distPoint(pt, ptIn);
  for (int i=0;i<nNeigh;i++) {
    double d = distPoint(neigh[i]->getPt(), ptIn);
    if (d < dist) {
      printf("Message(Vertex::isNearest): Neighbour (no %d, id %d)\n",
	     i, neigh[i]->getId());
      neigh[i]->print();
      printf("nearer point:\n");
      for (int j=0;j<DIM;j++)
	printf("%6.2f ", ptIn[j]);
      printf("\n");
      printf("than its parent vertex (id %d):\n", id);
      print();
      printf("Dist(pt, parent-vertex): %16.12f\n", dist);
      printf("Dist(pt, neigh-vertex): %16.12f\n", d);

      nearest = 0;
      break;
    }
  }

  return nearest;
}

void VertexList::checkCleared() {
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      for (int j=0;j<v->nSegment;j++)
	if (v->segRemoved[j]) {
	  printf("Error(VertexList::checkCleared): Vertex (pos=%d, id=%d)\n",
		 i, v->id);
	  printf("Segment %d is not cleared\n", j);
	}
    }
  }
}

void VertexList::initialiseImageData(char * dataFileName, char * trueFileName,
				     int * dim, double region[])
{
  int i;
  int j;
  int k;
  int l;

  imageInit = 1;

  for (i=0;i<DIM;i++)
    imageGridDim[i] = dim[i];

  nImage = 1;
  for (i=0;i<DIM;i++)
    nImage *= imageGridDim[i];

  imageColour = new double[nImage];
  imageTrueColour = new int[nImage];

  double * tmp = new double[nImage];
  int * tmp2 = new int[nImage];

  sumPixelDiff = 0;
  sumColDiff = 0.0;

  FILE * file = fopen(dataFileName, "r");
  for (i=0;i<nImage;i++) {
    fscanf(file, "%lf", &tmp[i]);
  }
  fclose(file);

  file = fopen(trueFileName, "r");
  for (i=0;i<nImage;i++) {
    fscanf(file, "%d", &tmp2[i]);
  }
  fclose(file);

  int index=0;
  int index2=0;
#ifndef DIM2D
  for (k=0;k<imageGridDim[2];k++) {
#endif
    for (j=0;j<imageGridDim[1];j++) {
      for (i=0;i<imageGridDim[0];i++) {
#ifdef DIM2D
	index2 = i*imageGridDim[1] + j;
#else
	index2 = i*imageGridDim[1]*imageGridDim[2] +
	  j*imageGridDim[2]+k;
#endif
	imageColour[index] = tmp[index2];
	imageTrueColour[index++] = tmp2[index2];
      }
    }
#ifndef DIM2D
  }
#endif

  delete [] tmp;
  delete [] tmp2;


  double d[DIM];
  for (i=0;i<DIM;i++)
    d[i] = (region[2*i+1]-region[2*i])/((double) imageGridDim[i]);

  imageGridPt = new double *[nImage];
  
  double pt[DIM];
  index = 0;

  if (DIM == 2) {
    pt[1] = region[YMIN] - 0.5*d[1];
    for (j=0;j<imageGridDim[1];j++) {
      pt[1] += d[1];
      pt[0] = region[XMIN] - 0.5*d[0];
      for (i=0;i<imageGridDim[0];i++) {
	pt[0] += d[0];
	imageGridPt[index] = new double[DIM];
	for (l=0;l<DIM;l++)
	  imageGridPt[index][l] = pt[l];
	index++;
      }
    }
  }
  else {
    pt[2] = region[ZMIN] - 0.5*d[2];
    for (k=0;k<imageGridDim[2];k++) {
      pt[2] += d[2];
      pt[1] = region[YMIN] - 0.5*d[1];
      for (j=0;j<imageGridDim[1];j++) {
	pt[1] += d[1];
	pt[0] = region[XMIN] - 0.5*d[0];
	for (i=0;i<imageGridDim[0];i++) {
	  pt[0] += d[0];
	  imageGridPt[index] = new double[DIM];
	  for (l=0;l<DIM;l++)
	    imageGridPt[index][l] = pt[l];
	  index++;
	}
      }
    }
  }
}

void VertexList::initialiseMarginalPosterior(int nCol, int dim[], int bIn,
					     double region[])
{
  int i;
  for (i=0;i<DIM;i++)
    margGridDim[i] = dim[i];

  nMarg = 1;
  for (i=0;i<DIM;i++)
    nMarg *= margGridDim[i];

  sumMarg = new int * [nMarg];

  nColour = nCol;
  for (i=0;i<nMarg;i++) {
    sumMarg[i] = new int[nColour];
    for (int j=0;j<nColour;j++)
      sumMarg[i][j] = 0;
  }

  burnIn = bIn;
  lastMargUpdate = new int [nMarg];
  for (i=0;i<nMarg;i++) {
    lastMargUpdate[i] = burnIn;
  }

  lastColour = new int [nMarg];

  Vertex::sumMarg = sumMarg;
  Vertex::lastMargUpdate = lastMargUpdate;
  Vertex::lastColour = lastColour;
  
  double d[DIM];
  for (i=0;i<DIM;i++)
    d[i] = (region[2*i+1]-region[2*i])/((double) margGridDim[i]);

  margGridPt = new double *[nMarg];
  
  double pt[DIM];
  int index = 0;

#ifndef DIM2D
  pt[2] = region[ZMIN] - 0.5*d[2];
  for (int k=0;k<margGridDim[2];k++) {
    pt[2] += d[2];
#endif
    pt[1] = region[YMIN] - 0.5*d[1];
    for (int j=0;j<margGridDim[1];j++) {
      pt[1] += d[1];
      pt[0] = region[XMIN] - 0.5*d[0];
      for (i=0;i<margGridDim[0];i++) {
	pt[0] += d[0];
	margGridPt[index] = new double[DIM];
	for (int l=0;l<DIM;l++)
	  margGridPt[index][l] = pt[l];
	index++;
      }
    }
#ifndef DIM2D
  }
#endif
}
void VertexList::initialiseVolFraction(int nCol, int dim[], int bIn,
				       double region[])
{
  int i;
  for (i=0;i<DIM;i++)
    volGridDim[i] = dim[i];

  nVol = 1;
  for (i=0;i<DIM;i++)
    nVol *= volGridDim[i];

  sumVol = new double[nCol];
  volFraction = new double[nCol];

  nColour = nCol;
  for (i=0;i<nCol;i++) {
    sumVol[i] = 0;
  }

  burnIn = bIn;

  double d[DIM];
  for (i=0;i<DIM;i++)
    d[i] = (region[2*i+1]-region[2*i])/((double) volGridDim[i]);

  volGridPt = new double *[nVol];
  
  double pt[DIM];
  int index = 0;

#ifndef DIM2D
  pt[2] = region[ZMIN] - 0.5*d[2];
  for (int k=0;k<volGridDim[2];k++) {
    pt[2] += d[2];
#endif
    pt[1] = region[YMIN] - 0.5*d[1];
    for (int j=0;j<volGridDim[1];j++) {
      pt[1] += d[1];
      pt[0] = region[XMIN] - 0.5*d[0];
      for (i=0;i<volGridDim[0];i++) {
	pt[0] += d[0];
	volGridPt[index] = new double[DIM];
	for (int l=0;l<DIM;l++)
	  volGridPt[index][l] = pt[l];
	index++;
      }
    }
#ifndef DIM2D
  }
#endif
}

void VertexList::initialiseImageData()
{
  Vertex * vMin = NULL;

  int index=0;
#ifndef DIM2D
  for (int k=0;k<imageGridDim[2];k++) {
#endif
    for (int j=0;j<imageGridDim[1];j++) {
      vMin = NULL;
      for (int i=0;i<imageGridDim[0];i++) {
	if (!vMin) {
	  vMin = findNearestVertex(imageGridPt[index]);
	}
	else {
	  vMin = vMin->findNearestVertex(imageGridPt[index]);
	}
	vMin->insertImage(index);
	double colDiff = imageColour[index] - vMin->colour;
	sumColDiff += colDiff*colDiff;
	sumPixelDiff += (vMin->colour != imageTrueColour[index]);
	index++;
      }
    }
#ifndef DIM2D
  }
#endif
}

void VertexList::initialiseMarginalPosterior()
{
  Vertex * vMin = NULL;

  initialisedMarg = 1;

  int index=0;
#ifndef DIM2D
  for (int k=0;k<margGridDim[2];k++) {
#endif
    for (int j=0;j<margGridDim[1];j++) {
      vMin = NULL;
      for (int i=0;i<margGridDim[0];i++) {
	if (!vMin) {
	  vMin = findNearestVertex(margGridPt[index]);
	}
	else {
	  vMin = vMin->findNearestVertex(margGridPt[index]);
	}
	vMin->insertMarg(index);
	lastColour[index] = vMin->colour;
	index++;
      }
    }
#ifndef DIM2D
  }
#endif
}

void VertexList::initialiseVolFraction()
{
  Vertex * vMin = NULL;

  volInit = 1;

  int index=0;
#ifndef DIM2D
  for (int k=0;k<volGridDim[2];k++) {
#endif
    for (int j=0;j<volGridDim[1];j++) {
      vMin = NULL;
      for (int i=0;i<volGridDim[0];i++) {
	if (!vMin) {
	  vMin = findNearestVertex(volGridPt[index]);
	}
	else {
	  vMin = vMin->findNearestVertex(volGridPt[index]);
	}
	vMin->insertVol(index);
	sumVol[vMin->colour]++;
	index++;
      }
    }
#ifndef DIM2D
  }
#endif
}

Vertex * VertexList::findNearestVertex(double * gridPoint)
{
  Vertex * vMin = NULL;
  double distMin = DBL_MAX;
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      double dist = distPoint(gridPoint, v->pt);
      if (dist < distMin) {
	distMin = dist;
	vMin = v;
      }
    }
  }
  assert(vMin);

  return vMin;
}

Vertex * Vertex::findNearestVertex(double * gridPoint)
{
  Vertex * vMin = this;
  double distMin = distPoint(gridPoint, pt);
  for (int i=0;i<nNeigh;i++) {
    Vertex * v = neigh[i];
    double dist = distPoint(gridPoint, v->pt);
    if (dist < distMin) {
      distMin = dist;
      vMin = v;
    }
  }

  assert(vMin);

  return vMin;
}


void VertexList::printLogInfo(FILE * logFile, int nIter)
{
  if (!initialisedMarg)
    return;

  int i;
  int p = (nIter - burnIn-1);

  double factor = 1.0/((double) p*nMarg);


  fprintf(logFile, "Facies proportion in realisation:\n");
  double * faciesProp = new double[nColour];
  for (i=0;i<nColour;i++) {
    faciesProp[i] = 0.0;
    for (int j=0;j<nMarg;j++)
      faciesProp[i] += sumMarg[j][i];
    faciesProp[i] *= factor;
  }

  fprintf(logFile, "Facies proportion\n");
  for (i=0;i<nColour;i++) {
    fprintf(logFile, "%d \t%6.4f\n", i, faciesProp[i]);    
  }

  delete [] faciesProp;
}


void VertexList::getMarkedPoints(mPoint * mPt, Vertex * last)
{
  int nPoints = 0;
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed && v != last)
      v->getMPt(&mPt[nPoints++]);
  }
}
