#ifndef VERTEX_
#define VERTEX_ 1

#include <float.h>

class Simplex;
class Wells;

class Vertex {
  friend class VertexList;
  friend class Delaunay;
public:
  Vertex();
  Vertex(Point vert, short col, int it);
  ~Vertex();

  void initialise(Point vert, short col, int it);

  int getColour();
  Point getPt();
  void getMPt(mPoint * p);
  int getId();

  double & operator[](int pos); // pt[pos];

  int sumFirstOrderColDiff(int action, int mark=0);
  void checkColDiff();
  void updateColDiff(int allNeigh = 1);

  void nullify();
  void copy(Vertex * v);

  int isRemoved();
  int isNearest(Point pt);
  int onBoundary();

  void insertNeigh(int & mark);
  void insertNeigh(Vertex ** v, int & mark);
  void insertNeigh(Vertex * v);
  void removeNeigh();
  void removeNeigh(Vertex * v);
  
  void insertMarg(int index);
  void removeMarg(int pos);

  void insertVol(int index);
  void removeVol(int pos);

  void updateProp(int iter);
  void restartMarg(FILE * file);
  Vertex * findNearestVertex(double * gridPoint);

  void print();
  void print(FILE * file);
  void printTessellation(FILE * file);
  void printFull();
  void printFull(FILE * file);
  void printRes(VertexList * vList, FILE * file);
  void printAll();
  void printMarkedPoint(FILE * file);
  void printNeigh(); 
  void printNeigh(VertexList * vList);
  void printNeighC(VertexList * vList);
  void printSimp(VertexList * vList);
  void printNeigh(VertexList * vList, FILE * file, int printColour);
  void printSimp(VertexList * vList, FILE * file);
  void printSimp();
  void printSegments();

  void insert(Simplex * s);
  void remove(Simplex * s);

  void move(Point newP, int newCol);
  
  int isNeigh(Vertex * v);
  void findNeigh(Vertex **& v, int & nV, int & mark);

  int getnSimp();
  void getSimp(Simplex ** & s, int & n);
  Simplex * getSimp(int index);
  void getVolume(double *& vol, int & n, double & sumVol);


  void removeSegment(int seg[]);
  void insertSegment(int seg[]);
  void clearSegment();
  void initialiseSegments(Wells * wells);

  void insertImage(int index);
  void removeImage(int pos);

private:
  Point pt;
  short colour;

  int id;

  int nMargData;
  int * margData; // Index
  int * margRemoved;

  static int ** sumMarg;
  static int * lastMargUpdate;
  static int * lastColour;

  int nImageData;
  int * imageData;
#ifdef KAN_EFFEKTIVISERE_KODEN
  int * imageRemoved;
  int nImageRemoved;
#endif
  int nVolData;
  int * volData;

  int nDeltaAlloc;
  int nSimpAlloc;
  int nNeighAlloc;
  int nSegmentAlloc;
  int nMargAlloc;
  int nImageAlloc;
  int nVolAlloc;

  int nSimp;
  Simplex ** simp;

  int nNeigh;
  Vertex ** neigh;

  short colDiff;
  short colDiffNew;

  int it;
  int removed;
  int visit; // Updates colDiff in Simulation::moveProposal

  int nSegment;
  int ** segment; // (well,segment)
  int segRemoved[100]; // NBNB
};

class VertexList
{
public:
  VertexList(int nMeanVertices);
  ~VertexList();

  Vertex * insert(mPoint & x, int it);
  void remove(Vertex * v);

  void insert(Simplex * s);

  Vertex & operator[](int pos);
  int pos(Vertex * v) const;

  int getN();
  int getNumberOfVertices();

  int totalNumberOfSegments();

  Vertex * getCopy(Vertex * v);

  void getMarkedPoints(mPoint * mPt, Vertex * last);

  void printMarkedPoints(FILE * file);

  int isRemoved(int pos);

  void restartMarg(FILE * file); //CHANGE$$  
  void insertMarg(Vertex * last);
  void removeMarg(Vertex * last);
  void updateMarginalPosterior(Vertex * last, int it);
  void updateMarginalPosteriorFinal(int it);
  void initialiseMarginalPosterior(int nColour, int dim[],
				   int bIn, double region[]);
  void initialiseMarginalPosterior();
  void initialiseImageData(char * dataFileName, char * trueFileName,
			   int * dim, double region[]);
  void initialiseImageData();
  void initialiseVolFraction(int nCol, int dim[], int bIn,
			     double region[]);
  void initialiseVolFraction();
  void insertVol(Vertex * last);
  void removeVol(Vertex * last, Vertex * lastCpy);
  const double * volumeFraction() const;
  Vertex * findNearestVertex(double * gridPoint);

  void insertImage(Vertex * last);
  void removeImage(Vertex * last, Vertex * lastCpy);

  void print();
  void printFull();
  void printNeigh();
  void printNeighC();
  void print(FILE * file);
  void printFull(FILE * file);
  void printNeigh(FILE * file);
  void printNeigh(FILE * file, int printColour);
  void printMargData();

  void printRes(FILE * file); // Write to the result file
  void printTessellation(FILE * file);
  void printMarginalPosterior(FILE * file, int nIter);
  void printLogInfo(FILE * logFile, int nIter);

  void printSegments();

  int initColDiff(); // Called from initMH via Delaunay::initColDiff

  void checkConsistency();
  void checkMarginalPosterior();
  void checkVolumeFraction();
  void checkImage();
  void checkAcceptedConsistency();
  void checkConsistency(short * nSimpInVert);
  void checkCleared();

  void findNearestVertex(Point pt, Vertex *& v);

  double SumColDiff() const {return sumColDiff;};
  double SumPixelDiff() const {return sumPixelDiff;};

private:
  Vertex * list;
  Vertex ** removed;
  int size;
  int n;
  int nAlloc;
  int nAllocDelta;
  int nRemoved;
  int nRemAlloc;
  int nRemAllocDelta;

  int initialisedMarg;
  int burnIn;
  int nMarg;
  int margGridDim[DIM];
  int nColour;
  double ** margGridPt;
  int ** sumMarg;
  int * lastMargUpdate;
  int * lastColour;

  int imageInit;
  int nImage;
  int imageGridDim[DIM];
  double * imageColour;
  int * imageTrueColour;
  double ** imageGridPt;
  int sumPixelDiff; // Difference between imageTrueColour's and actual colours
  double sumColDiff; // Squared difference between colours and image data

  int volInit;
  int nVol;
  int volGridDim[DIM];
  double * volFraction;
  double ** volGridPt;
  double * sumVol;

  int id;
};

inline void Vertex::move(Point newP, int newCol)
{
  for (int i=0;i<DIM;i++)
    pt[i] = newP[i];

  colour = (short) newCol;
}


inline double & Vertex::operator[](int pos)
{
  return pt[pos];
}

inline void Vertex::insert(Simplex * s)
{
  if (nSimp == nSimpAlloc) {
    nSimpAlloc += nDeltaAlloc;
    simp = (Simplex **) realloc(simp, nSimpAlloc*sizeof(Simplex *));
  }
  simp[nSimp++] = s;
}

inline void Vertex::remove(Simplex * s) // NBNB Do take much CPU time
{
  int i;
  for (i=0;i<nSimp;i++) {
    if (simp[i] == s)
      break;
  }

  if (i >= nSimp)
    assert(i < nSimp);

  for (int j=i+1; j<nSimp;j++)
    simp[j-1] = simp[j];

  nSimp--;
}
 
inline void Vertex::checkColDiff()
{
  for (int i=0;i<nNeigh;i++) {
    if (!neigh[i]->onBoundary() && (neigh[i]->colDiff != neigh[i]->colDiffNew)) {
      printf("Error(Vertex::checkColDiff): Neigh %d id %d colDiff %d != colDiffNew %d\n",
	     i, neigh[i]->getId(), neigh[i]->colDiff, neigh[i]->colDiffNew);
      exit(-1);
    }
  }
  
}

inline void Vertex::updateColDiff(int allNeigh)
{
  colDiff = colDiffNew;
    
  if (allNeigh) {
    for (int i=0;i<nNeigh;i++)
      if (!neigh[i]->onBoundary())
	neigh[i]->updateColDiff(0);
  }
}


inline void Vertex::getSimp(Simplex **& s, int & n)
{
  s = new Simplex * [nSimp];
  for (int i=0;i<nSimp;i++)
    s[i] = simp[i];

  n = nSimp;
}


inline Simplex * Vertex::getSimp(int index)
{
  Simplex * s = simp[index];
  return s;
}

inline int Vertex::getnSimp()
{
  return nSimp;
}

#ifdef OUT
inline void VertexList::imageColourDiffRemove(Vertex * v)
{
  int i;
  int j;
  double d2;
  double distMin = DBL_MAX;
  Vertex * w;
  
  int nNeigh = v->nNeigh;
  int * I = v->imageData;
  int nImageData = v->nImageData;
  Vertex ** neigh = v->neigh;

  for (i=0;i<nImageData;i++) {
    for (j=0;j<nNeigh;j++) {
      d2 = distPoint(imageGridPt[I[i]], neigh[j]->pt); 
      if (d2 < distMin) {
	w = neigh[j];
	distMin = d2;	
      }
    }
    imageColourDiffUpdate(w, v, i, v->imageData[i]);
  }
}

inline void VertexList::imageColourDiffAdd(Vertex * v)
{
  int i;
  int j;
  double d1;
  double d2;

  int nNeigh = v->nNeigh;
  
  for (i=0;i<nNeigh;i++) {
    Vertex * w = v->neigh[i];
    int n = w->nImageData;
    int * I = w->imageData;
    for (j=0;j<n;j++) {
      d1 = distPoint(imageGridPt[I[j]], v->pt); 
      d2 = distPoint(imageGridPt[I[j]], w->pt); 
      if (d1 < d2) {
	imageColourDiffUpdate(v, w, j, I[j]);
      }
    }
  }
}
  
inline void VertexList::imageColourDiffUpdate(Vertex * to, Vertex * from,
					      int pos, int index)
{
  double colDiff = 0.0;
  double s;

  s = (imageColour[index] - from->colour);
  colDiff -= 0.5*s*s;

  s = (imageColour[index] - to->colour);
  colDiff += 0.5*s*s;

  sumPixelDiff += (to->colour != imageTrueColour[index]) -
    (from->colour != imageTrueColour[index]);

  sumColDiff += colDiff;

  to->insertImage(index);
  from->removeImage(pos);
}
#endif

inline Vertex * VertexList::insert(mPoint & x, int it)
{
  Vertex * v;

  if (nRemoved > 0) {
    v = removed[--nRemoved];
    v->nullify(); // NBNB Not very efficent
  }
  else
  {
    if (n>=nAlloc)
    {
      assert(n<nAlloc);// May be avoid by Vertex * list[100]...
      //      nAlloc += nAllocDelta;
      //      list = (Vertex *) realloc(list, nAlloc*sizeof(Vertex));
    }
    
    v = &list[n++];
  }

  v->initialise(x.pt, x.colour, it);

  v->id = id;

  id++;

  return v;
}

inline void VertexList::remove(Vertex * v)
{
  if (nRemoved>= nRemAlloc)
  {
    nRemAlloc += nRemAllocDelta;
    removed = (Vertex **) realloc(removed, nRemAlloc*sizeof(Vertex *));
  }
  removed[nRemoved++] = v;
  v->removed = 1;
}


inline int VertexList::isRemoved(int pos)
{
  return (list[pos].isRemoved());
}

inline int Vertex::onBoundary()
{
  return (id <= DIM);
}


inline int VertexList::getN()
{
  return n;
}


inline int VertexList::getNumberOfVertices()
{
  return n - nRemoved;
}


inline int VertexList::pos(Vertex * v) const
{
  int ind = (int) (v - list); //Sjekk...

  return ind;
}

inline Vertex & VertexList::operator[](int pos)
{
  return list[pos];
}

inline int Vertex::isRemoved()
{
  return removed;
}


inline void Vertex::copy(Vertex * v)
{
  int i;

  v->nSimp = nSimp;
  v->nDeltaAlloc = nDeltaAlloc;
  if (!v->simp) {
    v->nSimpAlloc = nSimpAlloc;
    v->simp = (Simplex **) malloc(nSimpAlloc*sizeof(Simplex *));
  }
  else if (v->nSimpAlloc < nSimpAlloc) {
    v->nSimpAlloc = nSimpAlloc;
    v->simp = (Simplex **) realloc(v->simp, nSimpAlloc*sizeof(Simplex *));
  }
  for (i=0;i<v->nSimp;i++)
    v->simp[i] = simp[i];
  if (!v->neigh) {
    v->nNeighAlloc = nNeighAlloc;
    v->neigh = (Vertex **) malloc(nNeighAlloc*sizeof(Vertex *));
  }
  else if (v->nNeighAlloc < nNeighAlloc) {
    v->nNeighAlloc = nNeighAlloc;
    v->neigh = (Vertex **) realloc(v->neigh, nNeighAlloc*sizeof(Vertex *));
  }
  v->nNeigh = nNeigh;
  for (i=0;i<v->nNeigh;i++)
    v->neigh[i] = neigh[i];
  if (!v->pt)
    v->pt =(Point) malloc(DIM*sizeof(double));
  for (i=0;i<DIM;i++)
    v->pt[i] = pt[i];
  v->colour = colour;
  v->it = it;
  v->colDiff = colDiff;
  v->colDiffNew = colDiffNew;

  v->nSegment = nSegment;
  for (i=0;i<nSegment;i++)
    v->segment[i] = segment[i];

  if (nMargData > 0) {
    if (!v->margData) {
      v->nMargAlloc = nMargAlloc;
      v->margData = (int *) malloc(nMargAlloc*sizeof(int));
    }
    else if (v->nMargAlloc < nMargAlloc) {
      v->nMargAlloc = nMargAlloc;
      v->margData = (int *) realloc(v->margData, nMargAlloc*sizeof(int));
    }
    v->nMargData = nMargData;
    for (i=0;i<nMargData;i++)
      v->margData[i] = margData[i];
  }

  if (nImageData > 0) {
    if (!v->imageData) {
      v->nImageAlloc = nImageAlloc;
      v->imageData = (int *) malloc(nImageAlloc*sizeof(int));
    }
    else if (v->nImageAlloc < nImageAlloc) {
      v->nImageAlloc = nImageAlloc;
      v->imageData = (int *) realloc(v->imageData, nImageAlloc*sizeof(int));
    }
    v->nImageData = nImageData;
    for (i=0;i<nImageData;i++)
      v->imageData[i] = imageData[i];
  }

  if (nVolData > 0) {
    if (!v->volData) {
      v->nVolAlloc = nVolAlloc;
      v->volData = (int *) malloc(nVolAlloc*sizeof(int));
    }
    else if (v->nVolAlloc < nVolAlloc) {
      v->nVolAlloc = nVolAlloc;
      v->volData = (int *) realloc(v->volData, nVolAlloc*sizeof(int));
    }
    v->nVolData = nVolData;
    for (i=0;i<nVolData;i++)
      v->volData[i] = volData[i];
  }
  return;
}

inline int Vertex::getId()
{
  return id;
}

inline int Vertex::getColour()
{
  return colour;
}

inline Point Vertex::getPt()
{
  return pt;
}

inline void Vertex::getMPt(mPoint * p)
{
  for (int i=0;i<DIM;i++)
    p->pt[i] = pt[i];

  p->colour = colour;
}

inline int Vertex::isNeigh(Vertex * v)
{
  int found = 0;
  for (int i=0;i<nNeigh;i++) {
    if (v == neigh[i]) {
      found = 1;
      break;
    }
  }
  
  return found;
}

inline void VertexList::updateMarginalPosterior(Vertex * last, int it) {
  last->updateProp(it);
  //  for (int i=0;i<last->nNeigh;i++) { // NBNB Loop necessary??
  //    Vertex * v = last->neigh[i];
  //
  //    v->updateProp(it);
  //  }

  return;
}

inline void Vertex::updateProp(int iter)
{
  for (int i=0;i<nMargData;i++) {
    int I = margData[i];
    //    sumMarg[I][colour] += (iter -lastMargUpdate[I]);
    sumMarg[I][lastColour[I]] += (iter -lastMargUpdate[I]);
    lastMargUpdate[I] = iter;
    lastColour[I] = colour;
  }

}


inline void VertexList::updateMarginalPosteriorFinal(int it) {
  for (int i=0;i<n;i++) {
    Vertex * v = &list[i];
    if (!v->removed) {
      v->updateProp(it);
    }
  }

  return;
}


inline void Vertex::insertMarg(int index)
{
  if (nMargAlloc == 0) {
    nMargAlloc = nDeltaAlloc;
    margData = (int *) malloc(nMargAlloc*sizeof(int));
  }
  else if (nMargData == nMargAlloc) {
    nMargAlloc += nDeltaAlloc;
    margData = (int *) realloc(margData, nMargAlloc*sizeof(int));
  }

  margData[nMargData++] = index;
}

inline void Vertex::removeMarg(int pos)
{
  for (int i=pos;i<nMargData-1;i++) {
    margData[i] = margData[i+1];
  }
  nMargData--;
}


inline void VertexList::insertMarg(Vertex * last) {
  Point pt = last->pt;

  for (int i=0;i<last->nNeigh;i++) {
    Vertex * v = last->neigh[i];

    for (int j=v->nMargData-1;j>=0;j--) {
      int index = v->margData[j];
      double dist1 = distPoint(margGridPt[index], pt);
      double dist2 = distPoint(margGridPt[index], v->pt);

      if (dist1 < dist2) {
	last->insertMarg(index);
	v->removeMarg(j);
      }
    }
  }

}

inline void VertexList::removeMarg(Vertex * last) {
  for (int i=last->nMargData-1;i>=0;i--) {
    Vertex * vNearest = NULL;
    double distMin = DBL_MAX;
    int index = last->margData[i];
    for (int j=0;j<last->nNeigh;j++) {
      double d2 = distPoint(margGridPt[index], last->neigh[j]->pt); 
      if (d2 < distMin) {
	vNearest = last->neigh[j];
	distMin = d2;
      }
    }
    vNearest->insertMarg(index);
    last->removeMarg(i);
  }

}

inline void Vertex::insertVol(int index)
{
  if (nVolAlloc == 0) {
    nVolAlloc = nDeltaAlloc;
    volData = (int *) malloc(nVolAlloc*sizeof(int));
  }
  else if (nVolData == nVolAlloc) {
    nVolAlloc += nDeltaAlloc;
    volData = (int *) realloc(volData, nVolAlloc*sizeof(int));
  }

  volData[nVolData++] = index;
}

inline void Vertex::removeVol(int pos)
{
  assert(pos<nVolData);
  for (int i=pos;i<nVolData-1;i++) {
    volData[i] = volData[i+1];
  }
  nVolData--;
}


inline void VertexList::insertVol(Vertex * last) {
  Point pt = last->pt;

  for (int i=0;i<last->nNeigh;i++) {
    Vertex * v = last->neigh[i];

    int nV = v->nVolData;
    for (int j=nV-1;j>=0;j--) {
      int index = v->volData[j];
      double dist1 = distPoint(volGridPt[index], pt);
      double dist2 = distPoint(volGridPt[index], v->pt);
      
      if (dist1 < dist2) {
	last->insertVol(index);
	v->removeVol(j);
	sumVol[last->colour]++;
	sumVol[v->colour]--;
      }
    }
  }
}

inline void VertexList::removeVol(Vertex * last, Vertex * lastCpy) {
  int nV = last->nVolData;
  for (int i=nV-1;i>=0;i--) {
    Vertex * vNearest = NULL;
    double distMin = DBL_MAX;
    int index = last->volData[i];
    for (int j=0;j<lastCpy->nNeigh;j++) {
      Vertex * v = lastCpy->neigh[j];
      //      if (!v->onBoundary()) {
      double d2 = distPoint(volGridPt[index], v->pt); 
      if (d2 < distMin) {
	vNearest = v;
	distMin = d2;
      }
      //      }
    }

    assert(vNearest);

    vNearest->insertVol(index);
    last->removeVol(i);
    sumVol[vNearest->colour]++;
    sumVol[last->colour]--;
  }

}

inline const double * VertexList::volumeFraction() const
{
  for (int i=0;i<nColour;i++)
    volFraction[i] = sumVol[i]/nVol;

  return volFraction;
}

inline void Vertex::insertImage(int index)
{
  if (nImageAlloc == 0) {
    nImageAlloc = nDeltaAlloc;
    imageData = (int *) malloc(nImageAlloc*sizeof(int));
  }
  else if (nImageData == nImageAlloc) {
    nImageAlloc += nDeltaAlloc;
    imageData = (int *) realloc(imageData, nImageAlloc*sizeof(int));
  }

  imageData[nImageData++] = index;
}

inline void Vertex::removeImage(int pos)
{
#ifdef KAN_EFFEKTIVISERE_KODEN
  if (nRemoved>= nRemAlloc)
  {
    nRemImageAlloc += nRemImageAllocDelta;
    removed = (int *) realloc(removed, nRemImageAlloc*sizeof(int));
  }
  imageRemoved[nImageRemoved++] = pos;
#endif
  for (int i=pos;i<nImageData-1;i++) {
    imageData[i] = imageData[i+1];
  }
  nImageData--;
}


inline void VertexList::insertImage(Vertex * last) {
  double s;
  double colDiff;
  Point pt = last->pt;

  for (int i=0;i<last->nNeigh;i++) {
    Vertex * v = last->neigh[i];

    int nI = v->nImageData;

    for (int j=nI-1;j>=0;j--) {
      int index = v->imageData[j];
      double dist1 = distPoint(imageGridPt[index], pt);
      double dist2 = distPoint(imageGridPt[index], v->pt);

      if (dist1 < dist2) {
	last->insertImage(v->imageData[j]);
	v->removeImage(j);

	s = (imageColour[index] - v->colour);
	colDiff = - s*s;
	s = (imageColour[index] - last->colour);
	colDiff += s*s;

	sumColDiff += colDiff;
	sumPixelDiff += (last->colour != imageTrueColour[index]) -
	  (v->colour != imageTrueColour[index]);
      }
    }
  }

}

inline void VertexList::removeImage(Vertex * last, Vertex * lastCpy) {
  double s;
  double colDiff;

  int nI = last->nImageData;

  for (int i=nI-1;i>=0;i--) {
    int index = last->imageData[i];
    Vertex * vNearest = NULL;
    double distMin = DBL_MAX;
    for (int j=0;j<lastCpy->nNeigh;j++) {
      Vertex * v = lastCpy->neigh[j];
      double d2 = distPoint(imageGridPt[index], v->pt); 
      if (d2 < distMin) {
	vNearest = v;
	distMin = d2;
      }
    }

    assert(vNearest);

    vNearest->insertImage(index);
    last->removeImage(i);

    s = (imageColour[index] - last->colour);
    colDiff = -s*s;
    s = (imageColour[index] - vNearest->colour);
    colDiff += s*s;

    sumColDiff += colDiff;
    sumPixelDiff += (vNearest->colour != imageTrueColour[index]) -
      (last->colour != imageTrueColour[index]);
  }

}

#endif
