/*
vorSim3D: C++-routines for simulation from a coloured Voronoi
tessellation model.

Version 1.0

Copyright (C)  2001 ivind Skare

  This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU
  General Public License as published by the Free Software Foundation; either ve
rsion 2 of the
  License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT AN
Y WARRANTY; without
  even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPO
SE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along with t
his program; if not,
  write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Bost
on, MA 02111-1307,
  USA.


  The author's contact information:

 ivind Skare                Office tel.: (+47) 22 85 26 56
 Norwegian Computing Center         Fax:         (+47) 22 69 76 60 
 P.O.Box 114 Blindern               E-mail:      Oivind.Skare@nr.no 
 NO-0314 Oslo, Norway
*/
/*Include Files:*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <errno.h>
#include <assert.h>


#include "basic.h"
#include "input.h"
#include "point.h"
#include "vertex.h"
#include "wells.h"

#define sign(A) (A < 0 ? (-1) : (1))

Wells::Wells(Input * input, double regionOrig[])
{
  nWell = input->nWell;

  if (nWell>0) {
    path = new char [strlen(input->path)+strlen(input->wellPath)+2];
    if (strlen(input->path) > 0) {
      strcpy(path, input->path);
      strcat(path, "/");
      strcat(path, input->wellPath);
    }
    else {
      strcpy(path, input->wellPath);
    }

    nAlloc = 100;

    N0 = N1 = 0;
    L0 = 0;

    nFacies = input->nColour;

    zRegion = regionOrig[5] - regionOrig[4];
    zAnisotropy = input->zAnisotropy;
    zRegion *= zAnisotropy;

    updateFlag = 0;

    printf("Reads (and transforms) wells");
    fflush(stdout);
    readWells(input->wellNames, regionOrig);
    printf(".... wells read\n");

    nSegment = new int *[nWell];
    int i;
    int j;
    int nSeg;
    for (i=0;i<nWell;i++) {
      nSeg = nSegmentInWell[i];
      nSegment[i] = new int[nSeg];
      for (j=0;j<nSeg;j++)
	nSegment[i][j] = 1;
    }
    
    vertInSegment = new Vertex ***[nWell];
    for (i=0;i<nWell;i++) {
      nSeg = nSegmentInWell[i];
      vertInSegment[i] = new Vertex **[nSeg];
      for (j=0;j<nSeg;j++)
	vertInSegment[i][j] = new Vertex *[nAlloc];
    }
  }
}

Wells::~Wells()
{
  for (int i=0;i<nWell;i++) {
    int j;
    int n = nSegmentInWell[i];
    for (j=0;j<n;j++) {
      delete [] vertInSegment[i][j];
      
      for (int k=0;k<nSegment[i][j];k++) {
	delete [] wellSegment[i][j][k];
      }
      delete [] wellSegment[i][j];
    }
    delete [] vertInSegment[i];
    delete [] wellSegment[i];
    delete [] wellNames[i];
    delete [] facies[i];
    delete [] wellLength[i];
    delete [] nSegment[i];
  }
  delete[] vertInSegment;
  delete [] wellSegment;
  delete [] nSegmentInWell;
  delete [] wellNames;
  delete [] facies;
  delete [] wellLength;
  delete [] nSegment;

  delete [] path;
}

void Wells::readWells(char ** names, double regionOrig[])
{
  wellNames = new char *[nWell];
  nSegmentInWell = new int[nWell];
  wellSegment = new Point **[nWell];
  facies = new int * [nWell];

  for (int i=0;i<nWell;i++) {
    wellNames[i] = new char[strlen(names[i])+1];
    strcpy(wellNames[i], names[i]);

    readWell(i);
  }

  transformWells(regionOrig);
}

void Wells::readWell(int index)
{
  int nFaciesRecords = 4;

  int n = strlen(wellNames[index]) + strlen(path) + 6;
  char * name = new char[n];

  strcpy(name, path);
  name = strcat(name,wellNames[index]);
  name = strcat(name, ".well");
  FILE * ff = fopen(name, "r");
  if (!ff) {
    printf("ERROR(Wells:readWell): When opening file %s\n", name);
    assert(ff);
  }

  delete [] name;
  name = NULL;

  findNumberOfSegments(ff, nFaciesRecords, nSegmentInWell[index]);

  rewind(ff);


  wellSegment[index] = new Point * [nSegmentInWell[index]];
  int i;
  for (i=0;i<nSegmentInWell[index];i++) {
    wellSegment[index][i] = new Point[nAlloc];
    wellSegment[index][i][0] = new double[DIM];
  }

  facies[index] = new int [nSegmentInWell[index]];

  char * tmp = new char [1000];
  double * faciesRecord = new double[nFaciesRecords];
  for (i=0;i<nFaciesRecords;i++) // Heading of file
    fscanf(ff, "%s",tmp);

  delete [] tmp;

  int segment;
  for (segment=0;segment<nSegmentInWell[index];segment++) {
    for (i=0;i<nFaciesRecords;i++)
      fscanf(ff, "%lf", &faciesRecord[i]);

    wellSegment[index][segment][0][0] = faciesRecord[0];
    wellSegment[index][segment][0][1] = faciesRecord[1];
    wellSegment[index][segment][0][2] = faciesRecord[2];
    int nr = (int) faciesRecord[3];
    facies[index][segment] = nr;
  }

  delete [] faciesRecord;

  fclose(ff);
}


void Wells::transformWells(double regionOrig[])
{
  for (int index=0;index<nWell;index++) {
    for (int segment=0;segment<nSegmentInWell[index];segment++) {
      double z = wellSegment[index][segment][0][2];
      z = (z - regionOrig[4])*zAnisotropy;
      wellSegment[index][segment][0][2] = z;
    }
  }
}

///////////////////////////////////////////////////////////
// Finds point p along well segment {segment[0],segment[1]}
// such that |v1-p| = |v2-p|
// p is given by the interpolation by the value t between start and end point
// of the segment
// increasing: Interval [t,1] is nearest v1
// outside: Midpoint (t) is outside interval [0,1]
// nearest: (if outside) the midpoint t=0.5 is nearest v1
///////////////////////////////////////////////////////////
void Wells::findMidPoint(int segment[], Vertex * v1, Vertex * v2,
			 double & t, int & increasing, int & outside,
			 int & nearest)
{
  int i;
  Point p1 = v1->getPt();
  Point p2 = v2->getPt();
  double smallEps = 0.00000001;

  outside = 0;

  assert(segment[2]<nSegment[segment[0]][segment[1]]);  
  Point x0 = getSegment(segment[0], segment[1], segment[2]);
  Point x1 = getSegment(segment[0], segment[1], segment[2]+1);

  double dx[DIM];
  for (i=0;i<DIM;i++)
    dx[i] = x1[i] - x0[i];
  double p1norm = normPoint(p1);
  double p2norm = normPoint(p2);

  double dp[DIM];
  for (i=0;i<DIM;i++)
    dp[i] = p2[i]-p1[i];
  double prodx0dp = vecProd(x0, dp);
  double prodx1dp = vecProd(x1, dp);
  double prod2 = vecProd(dx, dp);

  double dist1;
  double dist2;
  if (fabs(prod2) < smallEps) {
    outside = 1;
  }
  else {
    double d1 = p2norm - p1norm - 2 * prodx0dp;
    double d2 = -p2norm + p1norm + 2 * prodx1dp;

    if (fabs(d1) < smallEps || fabs(d2) < smallEps)
      outside = 1;
    else {

      int tPos = ((sign(d1) * sign(prod2)) > 0);
      int tLess1 = ((sign(d2) * sign(prod2)) > 0);

      if (tPos && tLess1) {
	t = (p2norm - p1norm - 2 * prodx0dp)/(2*prod2);

	if (t<0.0) {
	  printf("Warning(Wells::findMidPoint) t %18.14f < 0\n", t);
	  t = 0.0;
	}
	else if (t>1.0){
	  printf("Warning(Wells::findMidPoint) t %18.14f > 1.0\n", t);
	  t = 1.0;
	}

	double delta = 0.1;
	double x[DIM];
	for (i=0;i<DIM;i++)
	  x[i] = x0[i] + dx[i]*(t+delta);
    
	dist1 = distPoint(p1, x);
	dist2 = distPoint(p2, x);

	increasing = (dist1<dist2);
      }
      else {
	outside = 1;
      }
    }
  }
  if (outside) {
    double xMean[DIM];
    for (i=0;i<DIM;i++)
      xMean[i] = 0.5*(x1[i] + x0[i]);
    dist1 = distPoint(p1, xMean);
    dist2 = distPoint(p2, xMean);
    
    nearest = (dist1<dist2);
  }
}

void Wells::intersect(Vertex * vNew, Vertex * vOld, int segment[])
{ 
  int fullSegment[3];
  int increasing;
  int nearest;
  int outside;
  double t;


  fullSegment[0] = segment[0];
  fullSegment[1] = segment[1];
  fullSegment[2] = findIndex(segment, vOld);

  findMidPoint(fullSegment, vNew, vOld, t, increasing, outside, nearest);

  if (outside) {
    if (nearest) {
      changeSegment(fullSegment, vOld, vNew);
    }
  }
  else { // (t>0.0 && t<1.0)
    if (increasing) {
      splitSegment(fullSegment, t, vOld, vNew);

      fullSegment[2]++;
      merge(vNew, fullSegment);
    }
    else {
      splitSegment(fullSegment, t, vNew, vOld);
      merge(vNew, fullSegment);
    }
  }
}


int Wells::findIndex(int segment[], Vertex * v)
{
  Vertex ** vert = vertInSegment[segment[0]][segment[1]];
  int i;
  if (updateFlag) {
     for (i=startIndex;i<=endIndex;i++) {
      if (vert[i] == v)
	break;
    }
    assert(i<=endIndex);   
  }
  else {
    int n = nSegment[segment[0]][segment[1]];
    for (i=0;i<n;i++) {
      if (vert[i] == v)
	break;
    }
    assert(i<n);
  }

  return i;
}

void Wells::merge(Vertex * vNew, int segment[])
{
  int I = segment[0];
  int J = segment[1];
  int K = segment[2];

  Vertex * v;

  assert(vNew == vertInSegment[I][J][K]);

  if (nSegment[I][J] > 1) {
    // Check upper
    if (K>0) {
      v = vertInSegment[I][J][K-1];
      if (v == vNew) {
	mergeSegment(vNew, I, J, K-1);
	K--;
      }
    }

    // Check lower
    if (K < nSegment[I][J] - 1) {
      v = vertInSegment[I][J][K+1];
      if (v == vNew) {
	mergeSegment(vNew, I, J, K);
      }
    }
  }
}


// Merge segments (I,J,K) and (I,J,K+1)
// Invariant: vertInSegment[I][J][K] = vertInSegment[I][J][K+1] = vNew
// New vertex: vNew
void Wells::mergeSegment(Vertex * vNew, int I, int J, int K)
{
  Point * ws = wellSegment[I][J];
  Vertex ** vert = vertInSegment[I][J];
  int n = nSegment[I][J];

  int segment[] = {I, J, K};

  assert(vert[K] == vert[K+1]);
  assert(vert[K] == vNew);

  if (updateFlag) {
    if (K<startIndex)
      startIndex--;
    if (K<endIndex)
      endIndex--;
  }

  vNew->removeSegment(segment);

  // Remove ws[K+1]
  delete [] ws[K+1];

  for (int i=K+1;i<n-1;i++) {
    ws[i] = ws[i+1];
    vert[i] = vert[i+1];
  }

  vert[n-1] = NULL;
  ws[n-1] = NULL;
  nSegment[I][J]--;
  totalNumberOfSegments--;
}


// Split segment {I=segment[0], J=segment[1], K=segment[2]} into
// two new segments (I,J,K) and (I,J,K+1) with vertices v1 and v2
// respectively.
// The new vertex is the one of v1 and v2 that is different from
// the previous vertex at (I,J,K)
void Wells::splitSegment(int segment[], double t, Vertex * v1,
			 Vertex * v2)
{
  int I = segment[0];
  int J = segment[1];
  int K = segment[2];
  int n = nSegment[I][J];
  n++;

  assert(n<nAlloc); // NBNB Temporary

  Point * ws = wellSegment[I][J];
  Vertex ** vert = vertInSegment[I][J];

  Vertex * v = vert[K];

  int prevSameColour = sameColour(v, I, J);

  if (updateFlag) {
    endIndex++;
  }

  v->removeSegment(segment);
  v1->insertSegment(segment);
  v2->insertSegment(segment);

  for (int i=n-1;i>K+1;i--) {
    ws[i] = ws[i-1];
    vert[i] = vert[i-1];
  }
  ws[K+1] = getPt(segment, t);
  vert[K] = v1;
  vert[K+1] = v2;

  assert(K+1<n);

  nSegment[I][J]++;

  Vertex * vNew;
  if (v != v2) {
    vNew = v2;
    K++;
  }
  else {
    vNew = v1;
  }
  
  assert(vNew == vert[K]);

  int newSameColour = sameColour(vNew, I, J);

  if (prevSameColour != newSameColour) {
    if (K == n-1) {
      int faciesShift = (facies[I][J] != facies[I][J+1]);

      if (faciesShift) { // J == 1
	if (prevSameColour) {
	  N1--;
	}
	else
	  N0--;
	if (newSameColour) {
	  N1++;
	}
	else
	  N0++;
      }
    }
  
    double length = segmentLength(I, J, K);
  
    if (newSameColour)
      L0 -= length;
    else
      L0 += length;
  }
    
  totalNumberOfSegments++;
}


void Wells::changeSegment(int segment[], Vertex * from, Vertex * to)
{
  int I = segment[0];
  int J = segment[1];
  int K = segment[2];

  vertInSegment[I][J][K] = to;
  from->removeSegment(segment);
  to->insertSegment(segment);


  int prevSameColour = sameColour(from, I, J);

  int newSameColour = sameColour(to, I, J);

  if (prevSameColour != newSameColour) {

    double length = segmentLength(I, J, K);
    
    if (K == nSegment[I][J]-1) {
      int faciesShift = (facies[I][J] != facies[I][J+1]);

      if (faciesShift) { // J == 1
	if (newSameColour) {
	  N1++;
	}
	else {
	  N0++;
	}
	if (prevSameColour) {
	  N1--;
	}
	else {
	  N0--;
	}
      }
    }

    if (newSameColour)
      L0 -= length;
    else
      L0 += length;
  }

  // Check if segments should be merged
  merge(to, segment);
}

Point Wells::getPt(int segment[], double t)
{
  Point pt = new double[DIM];

  Point pt0 = getSegment(segment[0], segment[1], segment[2]);
  Point pt1 = getSegment(segment[0], segment[1], segment[2]+1);
  for (int i=0;i<DIM;i++) {
    pt[i] = pt0[i]*(1.0-t) + pt1[i]*t;
  }

  return pt;
}

Point Wells::getSegment(int I, int J, int K)
{
  if (K < nSegment[I][J]) {
    return wellSegment[I][J][K];
  }
  else if (J + 1 < nSegmentInWell[I]){
    return wellSegment[I][J+1][0];
  }
  else {
    return NULL;
  }
}

void Wells::findNumberOfSegments(FILE * ff, int nRecords, int & nLines)
{
  int i;
  char tmp[1000];
  for (i=0;i<nRecords+1;i++) // Heading of file
    fscanf(ff, "%s", tmp);

  double * record = new double[nRecords];
  nLines = 1;
  int status = 1;
  for (i=0;i<nRecords;i++) 
    status = fscanf(ff, "%lf", &record[i]);

  while (status != EOF) {
    nLines++;

    for (i=0;i<nRecords;i++) 
      status = fscanf(ff, "%lf", &record[i]);
  }

  delete [] record;
}

void Wells::getAllSegments(int ** segment, int & nSeg)
{
  nSeg = 0;
  for (int i=0;i<nWell;i++) {
    for (int j=0;j<nSegmentInWell[i]-1;j++) {
      segment[nSeg] = new int[2];
      segment[nSeg][0] = i;
      segment[nSeg][1] = j;
      nSeg++;
    }
  }
}

void Wells::initialiseSegments(Vertex * v)
{
  totalNumberOfSegments = 0;
  totalWellLength = 0.0;
  wellLength = new double *[nWell];
  for (int i=0;i<nWell;i++) {
    wellLength[i] = new double[nSegmentInWell[i]-1];
    for (int j=0;j<nSegmentInWell[i]-1;j++) {
      vertInSegment[i][j][0] = v;
      totalNumberOfSegments++;

      wellLength[i][j] = segmentLength(i,j,0);
      totalWellLength += wellLength[i][j];
      int same = sameColour(v, i, j);
      if (!same)
	L0 += wellLength[i][j];
      if (facies[i][j] != facies[i][j+1]) {
	if (!same)
	  N0++;
	else
	  N1++;
      }
    }
  }
}

int Wells::sameColour(Vertex * v, int I, int J)
{
  return (facies[I][J] == v->getColour());
}

double Wells::segmentLength(int I, int J, int K)
{
  double length = 0.0;
  int segment[] = {I, J, K};

  Point x0 = getSegment(segment[0], segment[1], segment[2]);
  Point x1 = getSegment(segment[0], segment[1], segment[2]+1);

  for (int i=0;i<DIM;i++) {
    double d = x1[i] - x0[i];
    length += d*d;
  }

  length = sqrt(length);

  return length;
}

void Wells::printSegments()
{
  for (int i=0;i<nWell;i++) {
    for (int j=0;j<nSegmentInWell[i];j++) {
      for (int k=0;k<nSegment[i][j];k++) {
	printf("%d %d %d\n", i, j, k);
      }
    }
  }
}


void Wells::checkConsistency(VertexList * vList)
{
  if (nWell == 0) {
    return;
  }

  vList->checkCleared(); // segRemoved[..] = 0

  int nSegments = vList->totalNumberOfSegments();
  if (totalNumberOfSegments != nSegments) {
    printf("Error(Wells::checkConsistency): %s %d %s %d\n",
	   "totalNumberOfSegments in well structure",
	   totalNumberOfSegments,
	   "!= totalNumberOfSegments in vertex structure",
	   nSegments);
    printf("Well structure segments\n");
    printSegments();
    printf("Vertex structure segments\n");
    vList->printSegments();
    assert(0);
  }

  int i;
  int j;
  int k;
  for (i=0;i<nWell;i++) {
    for (j=0;j<nSegmentInWell[i];j++) {
      Vertex * v = vertInSegment[i][j][0];
      for (k=1;k<nSegment[i][j];k++) {
	Vertex * vPrev = v;
	v = vertInSegment[i][j][k];
	if (v == vPrev) {
	  printf("Warning(Wells::checkConsistency): ");
	  printf("Vertex (id %d) is in two neighbouring well segments.\n",
		 v->getId());
	  v->print();
	  printf("Segment %d %d %d\n", i, j, k);
	  assert(0);
	}
      }
    }
  }

  for (i=0;i<nWell;i++) {
    for (j=0;j<nSegmentInWell[i];j++) {
      Point pt = getSegment(i,j,0);
      for (k=0;k<nSegment[i][j];k++) {
	Point ptPrev = pt;
	pt = getSegment(i,j,k+1);
	if (pt == NULL) {
	  break; // Last point in well i
	}

	double midPt[DIM];
	for (int l=0;l<DIM;l++)
	  midPt[l] = 0.5*(pt[l] + ptPrev[l]);

	Vertex * v = vertInSegment[i][j][k];

	if (!v->isNearest(midPt)) {
	  printf("Error(Wells::checkConsistency): midPt (segment %d %d %d) ",
		 i, j, k);
	  printf("not nearest to vertex (id %d)\n", v->getId());
	  assert(0);
	}
      }
    }
  }

  // Check if N0 + N1 = totalNumberOfSegments
  int nOrig = 0;
  int nTotal = 0;
  double epsilon = 0.001; 
  double sumLengthL0 = 0.0;
  for (i=0;i<nWell;i++) {
    for (j=0;j<nSegmentInWell[i]-1;j++) {
      nOrig += (facies[i][j] != facies[i][j+1]);
      nTotal += nSegment[i][j];
      double sumLength = 0.0;
      for (k=0;k<nSegment[i][j];k++) {
	Vertex * v = vertInSegment[i][j][k];
	double length = segmentLength(i, j, k);
	sumLength += length;
	if (!sameColour(v, i, j))
	  sumLengthL0 += length;
      }
      if (fabs(sumLength - wellLength[i][j])>epsilon) {
	printf("Error(Wells::checkConsistency): sumLength (%f) != wellLength[%d][%d] (%f)\n",
	       sumLength, i, j, wellLength[i][j]);
	assert(0);
      }
      
    }
  }

  assert(N0 >= 0);
  assert(N1 >= 0);
  assert(L0 >= 0);

  if (totalNumberOfSegments != nTotal) {
    printf("Error(Wells::checkConsistency): totalNumberOfSegments %d != nTotal %d\n",
	   totalNumberOfSegments, nTotal);
    assert(0);
  }
  if (N0+N1 != nOrig) {
    printf("Error(Wells::checkConsistency): N0 (%d) + N1 (%d) != nOrig (%d)",
	   N0, N1, nOrig);
    assert(0);
  }
  if (fabs(sumLengthL0-L0) > epsilon) {
    printf("Error(Wells::checkConsistency): L0 %f != sumLengthL0 %f\n",
	   L0, sumLengthL0);
  }
}
	   
Vertex * Wells::nextVertexInSegment(Vertex * v, int segment[])
{
  int I = segment[0];
  int J = segment[1];
  int K = findIndex(segment, v);

  Vertex * vRet;

  if (K+1 < nSegment[I][J]) {
    vRet = vertInSegment[I][J][K+1];
  }
  else if (J + 1 < nSegmentInWell[I]){
    vRet = vertInSegment[I][J+1][0];
  }
  else {
    vRet = NULL;
  }

  return vRet;
}

Vertex * Wells::prevVertexInSegment(Vertex * v, int segment[])
{
  int I = segment[0];
  int J = segment[1];
  int K = findIndex(segment, v);

  Vertex * vRet;
  if (K-1 >= 0) {
    vRet = vertInSegment[I][J][K-1];
  }
  else if (J - 1 >= 0){
    vRet = vertInSegment[I][J-1][nSegment[I][J-1]-1];
  }
  else {
    vRet = NULL;
  }
  
  return vRet;
}


void Wells::updateSegment(int segment[], Vertex * vOld,
			  Vertex ** neigh, int nNeigh)
{
  int I = segment[0];
  int J = segment[1];
  int K = segment[2];

  Vertex * vv[100]; // Important to copy vertInSegment into this as
  //                   vertInSegment may change
  
  Vertex ** vert = vertInSegment[I][J];

  int k;

  updateFlag = 1;

  startIndex=K;
  endIndex=K;

  int n = nSegment[I][J];
  changeSegment(segment, vOld, neigh[0]);

  for (int j=1;j<nNeigh;j++) {
    for (k=startIndex;k<=endIndex;k++) {
      vv[k-startIndex] = vert[k];
    }
    int m = endIndex - startIndex + 1;
    for (k=0;k<m;k++) {
      n = nSegment[I][J];
      if (neigh[j] != vv[k]) {
	intersect(neigh[j], vv[k], segment);
      }
    }
  }

  updateFlag = 0;
}

void Wells::finalMerge(int segment[], int N)
{
  int I = segment[0];
  int J = segment[1];
  int K = segment[2];
  int fullSegment[] = {I, J, K};

  int nPrev = nSegment[I][J];
  for (int k=K;k<K+N;k++) {
    int kModified = k - (nPrev - nSegment[I][J]);
    fullSegment[2] = kModified;
    merge(vertInSegment[I][J][kModified], fullSegment);
  }

}

void Wells::printLogInfo(FILE * file)
{
  int i;
  int j;

  fprintf(file, "\n# wells %d\nNames: ", nWell);
  for (i=0;i<nWell;i++)
    fprintf(file, "%s ", wellNames[i]);
  fprintf(file, "\n");
  
  fprintf(file, "Total well length: %8.2f\n", totalWellLength);
  fprintf(file, "N0 %d\nN1 %d\nL0 %8.2f\nL0-Rel %6.4f\n",
	  N0, N1, L0, L0/totalWellLength);

  double * faciesProp = new double[nFacies];
  double ** faciesWellProps = new double * [nWell];
  for (j=0;j<nWell;j++)
    faciesWellProps[j] = new double[nFacies];

  double * wLength = new double [nWell];

  for (i=0;i<nFacies;i++) {
    faciesProp[i] = 0;
    for (j=0;j<nWell;j++)
      faciesWellProps[j][i] = 0;
  }

  for (i=0;i<nWell;i++) {
    wLength[i] = 0.0;
    for (int j=0;j<nSegmentInWell[i]-1;j++) {
      faciesProp[facies[i][j]] += wellLength[i][j];
      faciesWellProps[i][facies[i][j]] += wellLength[i][j];
      wLength[i] += wellLength[i][j];
    }
  }
  //  fprintf(file, "Zone number %d\n", zoneNumber);

  for (int j=0;j<nWell;j++) {
    fprintf(file, "Facies proportion in well %s:\n",
	    wellNames[j]);
    fprintf(file, "Facies  proportion\n");
    for (i=0;i<nFacies;i++) {
      //      if (i>0)
      fprintf(file, "%d \t%6.4f\n", i, //faciesTableInv[i],
	      faciesWellProps[j][i]/wLength[j]);    
      //      else
      //	fprintf(file, "%d \tBACK \t%6.4f\n", i,
      //		faciesWellProps[j][i]/wLength[j]);
    }
  }
  
  fprintf(file, "Facies proportion in wells:\n");
  fprintf(file, "Facies  ID    proportion\n");
  for (i=0;i<nFacies;i++) {
    //    if (i>0)
    fprintf(file, "%d \t%6.4f\n", i, //faciesTableInv[i],
	    faciesProp[i]/totalWellLength);    
    //    else
    //      fprintf(file, "%d \tBACK \t%6.4f\n", i,
    //	      faciesProp[i]/totalWellLength);
  }

  fprintf(file, "Total well length: %8.2f\n", totalWellLength);
  fprintf(file, "N0 %d\nN1 %d\nL0 %8.2f\nL0-Rel %6.4f\n",
	  N0, N1, L0, L0/totalWellLength);

  delete [] wLength;
  delete [] faciesProp;
  for (j=0;j<nWell;j++)
    delete [] faciesWellProps[j];
  delete [] faciesWellProps;
}


void
Wells::writeTransformedWells()
{
  int i;
  int j;
  char catalogue[] = "../dataNew/";
  char name[1000];
  char number[10];

  for (i=0;i<nWell;i++) {
    strcpy(name, catalogue);
    sprintf(number, "%d", i+1);
    strcat(name, number);
    strcat(name, ".well");
    
    FILE * file = fopen(name, "w");
    fprintf(file, "   X         Y         Z    facies\n");
    for (j=0;j<nSegmentInWell[i];j++) {
      double z = wellSegment[i][j][0][2]/zAnisotropy;
      fprintf(file, "%8.4f %8.4f %8.4f  %d\n",
	      wellSegment[i][j][0][0], wellSegment[i][j][0][1], 
	      z, facies[i][j]);
    }
    fclose(file);
  }
}
