/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* header file for the adaptive triangular metropolis sampling                                                   */
/* based on B Cai, R Meyer, and F Perron.                                                                        */
/*          Metropolis-Hastings algorithms with adaptive proposals.                                              */
/*          Statistics and Computing, 18-4 (2008), 421433.                                                      */
/* by Jaeger Jonathan , Universit de Lige, Belgium                                                            */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

#include <R.h>
#include <Rinternals.h>
#include <Rmath.h>

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* random generation for a triangular distribution rtri(lower, mode, upper, n, r)                                */
/* arguments:                                                                                                    */
/*   - lower is the lower bound of the triangular distribution,                                                  */
/*   - mode is the mode of the triangular distribution,                                                          */
/*   - upper is the upper bound of the triangular distribution,                                                  */
/*   - n is the number of random generation,                                                                     */
/*   - r stores the random generation (of lenght n)                                                              */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void rtri(double lower, double mode, double upper, int n, double *r)
{
	int l, length_i, length_j;

	double upper_minus_mode  = upper - mode;
	double upper_minus_lower = upper - lower;
	double mode_minus_lower  = mode - lower;
	
	int *i, *j;
	double *seq_along, *r_int;

	i = (int*)calloc(n, sizeof(int));
	j = (int*)calloc(n, sizeof(int));

	seq_along = (double*)calloc(n, sizeof(double));
	r_int     = (double*)calloc(n, sizeof(double));

	for (l = 0; l < n; l++)
	{
		seq_along[l] = l;

		GetRNGstate();
		r_int[l] = runif(0.0, 1.0);
		PutRNGstate();
	}

	length_i = 0;
	length_j = 0;

	if (lower != mode)
	{
		for (l = 0; l < n; l++)
		{
			if ((lower + sqrt(r_int[l] * upper_minus_lower * mode_minus_lower)) <= mode)
			{
				i[length_i] = seq_along[l];
				length_i += 1;
			}
			if ((upper - sqrt((1-r_int[l]) * upper_minus_lower * upper_minus_mode)) > mode)
			{
				j[length_j] = seq_along[l];
				length_j +=1;
			}
		}
	} else 
	{
		for (l = 0; l < n; l++)
		{
			if ((lower + sqrt(r_int[l] * upper_minus_lower * mode_minus_lower)) < mode)
			{
				i[length_i] = seq_along[l];
				length_i += 1;
			}
			if ((upper - sqrt((1-r_int[l]) * upper_minus_lower * upper_minus_mode)) >= mode)
			{
				j[length_j] = seq_along[l];
				length_j +=1;
			}
		}
	}

	if (length_i != 0)
	{
		for (l = 0; l < length_i; l++)
		{
			r[i[l]] = lower + sqrt(r_int[i[l]] * upper_minus_lower * mode_minus_lower);
		}
	}
	if (length_j != 0)
	{
		for (l = 0; l < length_j; l++)
		{
			r[j[l]] = upper - sqrt((1-r_int[j[l]]) * upper_minus_lower * upper_minus_mode);
		}
	}

	free(i);
	free(j);
	free(seq_along);
	free(r_int);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* density function of a triangular distribution dtri(lower, mode, upper, x, d)                                  */
/* arguments:                                                                                                    */
/*   - lower is the lower bound of the triangular distribution,                                                  */
/*   - mode is the mode of the triangular distribution,                                                          */
/*   - upper is the upper bound of the triangular distribution,                                                  */
/*   - x is the quantile where the density as to be calculated                                                   */
/*   - d stores the value of the density function                                                                */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void dtri(double lower, double mode, double upper, double x, double *d)
{
	double upper_minus_mode, upper_minus_lower, mode_minus_lower;

	*d = 0.0;
	
	if ((x >= lower) && (x <= upper))
	{
		upper_minus_mode  = upper - mode;
		upper_minus_lower = upper - lower;
		mode_minus_lower  = mode - lower;

		if ((lower < mode) & (mode < upper)){
			if (x <= mode)
			{
				*d = 2*(x-lower)/((mode_minus_lower)*(upper_minus_lower));
			} else
			{
				*d = 2*(upper-x)/((upper_minus_mode)*(upper_minus_lower));
			}
		} else
		{
			if (lower == mode)
			{
				*d = 2*(upper-x)/((upper_minus_lower)*(upper_minus_lower));
			}
			if (mode == upper)
			{
				*d = 2*(x-lower)/((upper_minus_lower)*(upper_minus_lower));
			}
		}
	}
}


/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* random generation for a mixture of triangular distributions rmtri(x, m, r)                                    */
/* arguments:                                                                                                    */
/*   - x is the set of "knots",                                                                                  */
/*   - m is the number of "knots",                                                                               */
/*   - r stores the random generation of a mixture of triangular distributions.                                  */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void rmtri(double *x, int m, double *r)
{
	int l;
	int m_plus_one = m + 1;
	double inverse_m = 1.0/((double)m);
	double v;

	double *weight;
	double *cumsum_weight;
	double *r_int;
	double *X;

	weight        = (double*)calloc(m_plus_one, sizeof(double));
	cumsum_weight = (double*)calloc(m_plus_one, sizeof(double));
	r_int         = (double*)calloc(1, sizeof(double));
	X             = (double*)calloc(3*(m-1), sizeof(double));

	/*************************************************************************************************************/
	/* initialising the weights                                                                                  */
	/*************************************************************************************************************/

		for (l = 0; l < m_plus_one; l++)
		{
			weight[l] = inverse_m;
		}
		weight[1]     *= 0.5;
		weight[(m-1)] *= 0.5;

	/*************************************************************************************************************/
	/* vector of the cumulative sums of the weights                                                              */
	/*************************************************************************************************************/

		cumsum_weight[0] = weight[0];
		for (l = 1; l < m_plus_one; l++)
		{
			cumsum_weight[l] = cumsum_weight[(l-1)] + weight[l];
		}

	/*************************************************************************************************************/
	/* generation of a uniform random variable [0;1]                                                             */
	/*************************************************************************************************************/

		GetRNGstate();
		v = runif(0.0, 1.0);
		PutRNGstate();

	/*************************************************************************************************************/
	/* generation from the mixture of exponential/triangular distributions                                       */
	/*************************************************************************************************************/

		if (v < cumsum_weight[0])
		{
			GetRNGstate();
			*r = - rexp((x[1] - x[0])) + x[0];
			PutRNGstate();
		}
		if (v > cumsum_weight[(m-1)])
		{
			GetRNGstate();
			*r = rexp((x[(m-2)] - x[(m-3)])) + x[m-2];
			PutRNGstate();
		}
		if ((v >= cumsum_weight[0]) & (v <= cumsum_weight[(m-1)]))
		{
			for (l=0; l<3*(m-1); l++)
			{
				if (l == 0)
				{
					X[l] = x[0];
				}
				if ((l >= 1) & (l <= (m-2)))
				{
					X[l] = x[(l-1)];
				}
				if ((l >= (m-1)) & (l <= (2*m-3)))
				{
					X[l] = x[(l-(m-1))];
				}
				if ((l >= (2*m-2)) & (l <= (3*m-5)))
				{
					X[l] = x[(l-(2*m-3))];
				}
				if (l == (3*m-4))
				{
					X[l] = x[(m-2)];
				}
			}
			for (l = 0; l < (m-1); l++)
			{
				if ((v >= cumsum_weight[l]) & (v < cumsum_weight[(l+1)]))
				{
					rtri(X[l], X[(l+(m-1))], X[(l+(2*m-2))], 1, r_int);
					*r = *r_int;
					l = m-1;
				}
			}
		}

	free(weight);
	free(cumsum_weight);
	free(r_int);
	free(X);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* density function of a mixture of triangular distributions dmtri(x, m, r)                                      */
/* arguments:                                                                                                    */
/*   - x is the set of "knots",                                                                                  */
/*   - m is the number of "knots",                                                                               */
/*   - y is the quantile where the density as to be calculated                                                   */
/*   - d stores the value of the density function                                                                */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void dmtri(double *x, int m, double y, double *d)
{
	int l;
	int m_plus_one = m + 1;
	double inverse_m = 1/((double)m);
	double *weight;
	double *X;
	double *d_int;
	double *vec_density;

	weight      = (double*)calloc(m_plus_one, sizeof(double));
	X           = (double*)calloc(3*(m-1), sizeof(double));
	d_int       = (double*)calloc(1, sizeof(double));
	vec_density = (double*)calloc(m_plus_one, sizeof(double));

	/*************************************************************************************************************/
	/* initialising the weights                                                                                  */
	/*************************************************************************************************************/

		for (l = 0; l < m_plus_one; l++)
		{
			weight[l] = inverse_m;
		}
		weight[1]     *= 0.5;
		weight[(m-1)] *= 0.5;

	/*************************************************************************************************************/
	/* matrix X                                                                                                  */
	/*************************************************************************************************************/

		for (l = 0; l < 3*(m-1); l++)
		{
			if (l == 0)
			{
				X[l] = x[0];
			}
			if ((l >= 1) & (l <= (m-2)))
			{
				X[l] = x[(l-1)];
			}
			if ((l >= (m-1)) & (l <= (2*m-3)))
			{
				X[l] = x[(l-(m-1))];
			}
			if ((l >= (2*m-2)) & (l <= (3*m-5)))
			{
				X[l] = x[(l-(2*m-3))];
			}
			if (l == (3*m-4))
			{
				X[l] = x[(m-2)];
			}
		}

	/*************************************************************************************************************/
	/* value of each triangular component of the mixture                                                         */
	/*************************************************************************************************************/

		for (l=0; l<(m-1); l++)
		{
			dtri(X[l], X[l+(m-1)], X[l+(2*m-2)], y, d_int);
			vec_density[(l+1)] = *d_int;
		}

	/*************************************************************************************************************/
	/* value of the exponential components                                                                       */
	/*************************************************************************************************************/

		if (y >= x[0])
		{
			vec_density[0] = 0.0;
		} else
		{
			vec_density[0] = (exp((y-x[0])/(x[1]-x[0])))/(x[1]-x[0]);
		}
		if (y <= x[(m-2)])
		{
			vec_density[m] = 0.0;
		} else
		{
			vec_density[m] = (exp((x[(m-2)]-y)/(x[(m-2)]-x[(m-3)])))/(x[(m-2)]-x[(m-3)]);
		}

	/*************************************************************************************************************/
	/* value of the density                                                                                      */
	/*************************************************************************************************************/

		*d = 0.0;

		for (l = 0; l < m_plus_one; l++)
		{
			*d += weight[l]*vec_density[l];
		}

	free(weight);
	free(X);
	free(d_int);
	free(vec_density);
}