/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* header file for log conditional posterior distributions                                                       */
/* One-compartment intravenous bolus model                                                                       */
/* by Jaeger Jonathan , Universit de Lige, Belgium                                                            */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <R.h>
#include <Rmath.h>

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for tau, gamma                                                         */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_tau_gamma(int N, int I, double I_moins_un_demi,
							int M, int M_sq,
							double *prec_c_prior, double *prec_c_prior_mu_c_prior,
							double *P_11, double *P_00, double *P_0110,
							double sum_y_sq, double *tBB, double *tBy,
							double a_tau_post_moins_un, double b_tau,
							double a_gamma_moins_un, double b_gamma,
							double tau, double log_tau,
							double gamma, double log_gamma,
							double *lpk_vect,
							double *lp)
{
	int i, i_two;
	int i_M;
	int m1, m2;
	int m1_M;
	int indice_m;

	double ke_i, ke_i_sq;
	double inv_V_i, inv_V_i_sq;

	double gamma_over_V_i_sq;

	double tau_over_V_i, tau_over_V_i_sq;

	double logdetV1 = 0.0, logdetV2 = 0.0;
	double sign_detV1 = 0.0, sign_detV2 = 0.0;
	double intermedprod1 = 0.0, intermedprod2 = 0.0;

	double *V1_i, *v1_i, *V2_i, *v2_i;
	double *logdetV1_i, *logdetV2_i;
	double *sign_detV1_i, *sign_detV2_i;
	double *intermedprod1_i, *intermedprod2_i;

	V1_i = (double*)calloc(M_sq, sizeof(double));
	v1_i = (double*)calloc(M, sizeof(double));
	V2_i = (double*)calloc(M_sq, sizeof(double));
	v2_i = (double*)calloc(M, sizeof(double));
	logdetV1_i = (double*)calloc(1, sizeof(double));
	logdetV2_i = (double*)calloc(1, sizeof(double));
	sign_detV1_i = (double*)calloc(1, sizeof(double));
	sign_detV2_i = (double*)calloc(1, sizeof(double));
	intermedprod1_i = (double*)calloc(1, sizeof(double));
	intermedprod2_i = (double*)calloc(1, sizeof(double));

	if ((tau <= 0.0) | (gamma <= 0.0))
	{
		*lp = -1e12;
	} else
	{
		/*********************************************************************************************************/
		/* loop for all the subject                                                                              */
		/*********************************************************************************************************/

			for (i = 0; i < I; i++)
			{
				i_M = i*M;
				i_two = i*2;

				/*************************************************************************************************/
				/* parameters for subject i                                                                      */
				/*************************************************************************************************/

					ke_i    = exp(lpk_vect[i_two]);
					ke_i_sq = ke_i*ke_i;

					inv_V_i    = exp(-lpk_vect[(i_two + 1)]);
					inv_V_i_sq = inv_V_i*inv_V_i;

					gamma_over_V_i_sq = gamma*inv_V_i_sq;

					tau_over_V_i    = tau*inv_V_i;
					tau_over_V_i_sq = tau*inv_V_i_sq;

				/*************************************************************************************************/
				/* matrices V2_i, V1_i, vectors v2_i et v1_i                                                     */
				/*************************************************************************************************/

					for (m1 = 0; m1 < M; m1++)
					{
						m1_M = m1*M;
						for (m2 = 0; m2 < M; m2++)
						{
							indice_m = m1_M + m2;
							V1_i[indice_m] = gamma_over_V_i_sq*(P_11[indice_m] + ke_i*P_0110[indice_m] + ke_i_sq*P_00[indice_m]) +
											 prec_c_prior[indice_m];
							V2_i[indice_m] = V1_i[indice_m] +
											 tau_over_V_i_sq*tBB[indice_m];
						}
						v1_i[m1] = prec_c_prior_mu_c_prior[m1];
						v2_i[m1] = v1_i[m1] +
								   tau_over_V_i*tBy[(i_M + m1)];
					}

				/*************************************************************************************************/
				/* log of the det of V1_i et V2_i                                                                */
				/* t(v1_i)*solveV1_i*v1_i et t(v2_i)*solveV2_i*v2_i                                              */
				/*************************************************************************************************/
				
					(*logdetV1_i) = 0.0;
					(*logdetV2_i) = 0.0;
					(*sign_detV1_i) = 1.0;
					(*sign_detV2_i) = 1.0;

					(*intermedprod1_i) = 0.0;
					(*intermedprod2_i) = 0.0;

					logdet_et_prod(V1_i, v1_i, M, logdetV1_i, sign_detV1_i, intermedprod1_i);
					logdet_et_prod(V2_i, v2_i, M, logdetV2_i, sign_detV2_i, intermedprod2_i);

					logdetV1 += (*logdetV1_i);
					logdetV2 += (*logdetV2_i);
					
					sign_detV1 += (*sign_detV1_i);
					sign_detV2 += (*sign_detV2_i);

					intermedprod1 += (*intermedprod1_i);
					intermedprod2 += (*intermedprod2_i);
			}
			
		/*********************************************************************************************************/
		/* log posterior conditional distribution                                                                */
		/*********************************************************************************************************/

			if ((sign_detV1 <= I_moins_un_demi) | (sign_detV2 <= I_moins_un_demi))
			{
				*lp = -1e12;
			} else
			{
				*lp = a_tau_post_moins_un*log_tau - tau*(b_tau + 0.5*sum_y_sq) +
					  a_gamma_moins_un*log_gamma - b_gamma*gamma -
					  0.5*(intermedprod1 - logdetV1 - intermedprod2 + logdetV2);
			}
	}

	free(V1_i);
	free(v1_i);
	free(V2_i);
	free(v2_i);
	free(logdetV1_i);
	free(logdetV2_i);
	free(sign_detV1_i);
	free(sign_detV2_i);
	free(intermedprod1_i);
	free(intermedprod2_i);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for lpk_vect                                                           */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_lpk_vect(int N, int I,
						   int M, int M_sq,
						   double *prec_c_prior, double *prec_c_prior_mu_c_prior,
						   double *P_11, double *P_00, double *P_0110,
						   double *tBB, double *tBy,
						   double tau,
						   double gamma,
						   double *lpk_mean, double *a_tau_lpk_post, double *b_tau_lpk, double max_tau_lpk,
						   int subject, double *lpk_vect_prop, double *lpk_vect_old,
						   double *lp)
{
	int i, l, i_two;
	int subject_M, subject_two;
	int m1, m2;
	int m1_M;
	int indice_m;

	double ke_subject, ke_subject_sq;
	double inv_V_subject, inv_V_subject_sq;

	double gamma_over_V_subject_sq;

	double tau_over_V_subject, tau_over_V_subject_sq;

	double *b_tau_lpk_post;
	double *lpk_vect;

	double *V1_subject, *v1_subject, *V2_subject, *v2_subject;
	double *logdetV1_subject, *logdetV2_subject;
	double *sign_detV1_subject, *sign_detV2_subject;
	double *intermedprod1_subject, *intermedprod2_subject;

	b_tau_lpk_post = (double*)calloc(2, sizeof(double));
	lpk_vect = (double*)calloc((2*I), sizeof(double));

	V1_subject = (double*)calloc(M_sq, sizeof(double));
	v1_subject = (double*)calloc(M, sizeof(double));
	V2_subject = (double*)calloc(M_sq, sizeof(double));
	v2_subject = (double*)calloc(M, sizeof(double));
	logdetV1_subject = (double*)calloc(1, sizeof(double));
	logdetV2_subject = (double*)calloc(1, sizeof(double));
	sign_detV1_subject = (double*)calloc(1, sizeof(double));
	sign_detV2_subject = (double*)calloc(1, sizeof(double));
	intermedprod1_subject = (double*)calloc(1, sizeof(double));
	intermedprod2_subject = (double*)calloc(1, sizeof(double));

	subject_M = subject*M;
	subject_two = subject*2;
		
	/*************************************************************************************************************/
	/* vector of individual lpk parameter                                                                        */
	/*************************************************************************************************************/
	
		for (i = 0; i < I; i++)
		{
			i_two = i*2;
			for (l = 0; l < 2; l++)
			{
				lpk_vect[(i_two + l)] = lpk_vect_old[(i_two + l)];
			}
		}
		for (l = 0; l < 2; l++)
		{
			lpk_vect[(subject_two + l)] = lpk_vect_prop[(subject_two + l)];
		}
		
	/*************************************************************************************************************/
	/* pk parameters for subject subject                                                                         */
	/*************************************************************************************************************/

		ke_subject    = exp(lpk_vect[subject_two]);
		ke_subject_sq = ke_subject*ke_subject;

		inv_V_subject    = exp(-lpk_vect[(subject_two + 1)]);
		inv_V_subject_sq = inv_V_subject*inv_V_subject;

		gamma_over_V_subject_sq = gamma*inv_V_subject_sq;

		tau_over_V_subject    = tau*inv_V_subject;
		tau_over_V_subject_sq = tau*inv_V_subject_sq;

	/*************************************************************************************************************/
	/* matrices V2_subject, V1_subject, vectors v2_subject et v1_subject                                         */
	/*************************************************************************************************************/

		for (m1 = 0; m1 < M; m1++)
		{
			m1_M = m1*M;
			for (m2 = 0; m2 < M; m2++)
			{
				indice_m = m1_M + m2;
				V1_subject[indice_m] = gamma_over_V_subject_sq*(P_11[indice_m] + ke_subject*P_0110[indice_m] + ke_subject_sq*P_00[indice_m]) +
									   prec_c_prior[indice_m];
				V2_subject[indice_m] = V1_subject[indice_m] +
									   tau_over_V_subject_sq*tBB[indice_m];
			}
			v1_subject[m1] = prec_c_prior_mu_c_prior[m1];
			v2_subject[m1] = v1_subject[m1] +
							 tau_over_V_subject*tBy[(subject_M + m1)];
		}

	/*************************************************************************************************************/
	/* log of the det of V1_subject and V2_subject                                                               */
	/* t(v1_subject)*solveV1_subject*v1_subject and t(v2_subject)*solveV2_subject*v2_subject                     */
	/*************************************************************************************************************/

		*logdetV1_subject = 0.0;
		*logdetV2_subject = 0.0;
		*sign_detV1_subject = 1.0;
		*sign_detV2_subject = 1.0;

		*intermedprod1_subject = 0.0;
		*intermedprod2_subject = 0.0;

		logdet_et_prod(V1_subject, v1_subject, M, logdetV1_subject, sign_detV1_subject, intermedprod1_subject);
		logdet_et_prod(V2_subject, v2_subject, M, logdetV2_subject, sign_detV2_subject, intermedprod2_subject);
	
	/*************************************************************************************************************/
	/* log posterior condition distribution                                                                      */
	/*************************************************************************************************************/

		if (((*sign_detV1_subject) < 0.5) | ((*sign_detV2_subject) < 0.5))
		{
			*lp = -1e12;
		} else
		{
			*lp = 0.0;
			for (l = 0; l < 2; l++)
			{
				b_tau_lpk_post[l] = 0.0;
				for (i = 0; i < I; i++)
				{
					b_tau_lpk_post[l] += ((lpk_vect[(i*2 + l)] - lpk_mean[l])*(lpk_vect[(i*2 + l)] - lpk_mean[l]));
				}
				b_tau_lpk_post[l] *= 0.5;
				b_tau_lpk_post[l] += b_tau_lpk[l];
			
				*lp += (-a_tau_lpk_post[l]*log(b_tau_lpk_post[l]) + pgamma(max_tau_lpk, a_tau_lpk_post[l], 1.0/b_tau_lpk_post[l], 1, 1));
			}
			*lp += 0.5*(((*logdetV1_subject) - (*intermedprod1_subject) - (*logdetV2_subject) + (*intermedprod2_subject)));
		}

	free(b_tau_lpk_post);
	free(lpk_vect);

	free(V1_subject);
	free(v1_subject);
	free(V2_subject);
	free(v2_subject);
	free(logdetV1_subject);
	free(logdetV2_subject);
	free(sign_detV1_subject);
	free(sign_detV2_subject);
	free(intermedprod1_subject);
	free(intermedprod2_subject);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for lpk_mean                                                           */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_lpk_mean(int I,
						   double *mu_lpk, double *prec_lpk,
						   double *a_tau_lpk_post, double *b_tau_lpk, double max_tau_lpk,
						   double *lpk_mean_prop, double *lpk_mean_old, int indice,
						   double *lpk_vect,
						   double *lp)
{
	int i, l;
	

	double *lpk_mean;
	double *b_tau_lpk_post;

	lpk_mean = (double*)calloc(2, sizeof(double));
	b_tau_lpk_post = (double*)calloc(2, sizeof(double));
	
	for (l = 0; l < 2; l++)
	{
		lpk_mean[l] = lpk_mean_old[l];
	}
	lpk_mean[indice] = lpk_mean_prop[indice];

	for (l = 0; l < 2; l++)
	{
		b_tau_lpk_post[l] = 0.0;
		for (i = 0; i < I; i++)
		{
			b_tau_lpk_post[l] += ((lpk_vect[(i*2 + l)] - lpk_mean[l])*(lpk_vect[(i*2 + l)] - lpk_mean[l]));
		}
		b_tau_lpk_post[l] *= 0.5;
		b_tau_lpk_post[l] += b_tau_lpk[l];
	}

	*lp = 0.0;
	for (l = 0; l < 2; l++)
	{
		*lp += (-a_tau_lpk_post[l]*log(b_tau_lpk_post[l]) +
			   pgamma(max_tau_lpk, a_tau_lpk_post[l], 1.0/b_tau_lpk_post[l], 1, 1) -
			   0.5*prec_lpk[l]*(lpk_mean[l] - mu_lpk[l])*(lpk_mean[l] - mu_lpk[l]));
	}

	free(lpk_mean);
	free(b_tau_lpk_post);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for lpk_tau                                                            */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_lpk_tau(int I, double double_I,
						  double *mu_lpk, double *prec_lpk,
						  double *a_tau_lpk_post_moins_un, double *b_tau_lpk, double max_tau_lpk,
						  double *lpk_tau_prop, double *log_lpk_tau_prop,
						  double *lpk_tau_old, double *log_lpk_tau_old,
						  int indice,
						  double *lpk_vect,
						  double *lp)
{
	int i, l;
	int i_two_plus_l;

	double *lpk_tau;
	double *log_lpk_tau;
	
	lpk_tau = (double*)calloc(2, sizeof(double));
	log_lpk_tau = (double*)calloc(2, sizeof(double));
	
	for (l = 0; l < 2; l++)
	{
		lpk_tau[l]     = lpk_tau_old[l];
		log_lpk_tau[l] = log_lpk_tau_old[l];
	}
	lpk_tau[indice]     = lpk_tau_prop[indice];
	log_lpk_tau[indice] = log_lpk_tau_prop[indice];

	double *sum_i_lpk_vect;
	double *sum_i_lpk_vect_sq;

	sum_i_lpk_vect = (double*)calloc(2, sizeof(double));
	sum_i_lpk_vect_sq = (double*)calloc(2, sizeof(double));

	if ((lpk_tau[0] < 0.0) | (lpk_tau[1] < 0.0) | (lpk_tau[0] > max_tau_lpk) | (lpk_tau[1] > max_tau_lpk))
	{
		*lp = -1e12;
	} else
	{
		for (l = 0; l < 2; l++)
		{
			sum_i_lpk_vect[l]    = 0.0;
			sum_i_lpk_vect_sq[l] = 0.0;
			for (i = 0; i < I; i++)
			{
				i_two_plus_l = i*2 + l;
				sum_i_lpk_vect[l]    += lpk_vect[i_two_plus_l];
				sum_i_lpk_vect_sq[l] += lpk_vect[i_two_plus_l]*lpk_vect[i_two_plus_l];
			}
		}

		*lp = 0.0;
		for (l = 0; l < 2; l++)
		{
			*lp += (a_tau_lpk_post_moins_un[l]*log_lpk_tau[l] - (b_tau_lpk[l] + 0.5*sum_i_lpk_vect_sq[l])*lpk_tau[l] +
					0.5*((lpk_tau[l]*sum_i_lpk_vect[l] + prec_lpk[l]*mu_lpk[l])*(lpk_tau[l]*sum_i_lpk_vect[l] + prec_lpk[l]*mu_lpk[l])/(double_I*lpk_tau[l] + prec_lpk[l]) - log(double_I*lpk_tau[l] + prec_lpk[l])));
		}
	}
	
	free(lpk_tau);
	free(log_lpk_tau);
	free(sum_i_lpk_vect);
	free(sum_i_lpk_vect_sq);
}