/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* header file for log conditional posterior distributions                                                       */
/* Two-compartment intravenous bolus model                                                                       */
/* by Jaeger Jonathan , Universit de Lige, Belgium                                                            */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <R.h>
#include <Rmath.h>

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for tau, gamma                                                         */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_tau_gamma_lkcp(int N, int I, double I_moins_un_demi,
								 int M1, int M2, int M, int M_sq,
								 double *prec_c_prior, double *prec_c_prior_mu_c_prior,
								 double *P_11, double *P_10, double *P_01, double *P_00, double *P_0110,
								 double sum_y_sq, double *tBB, double *tBy,
								 double *mu_lpk, double *prec_lpk,
								 double a_tau_post_moins_un, double b_tau,
								 double *a_gamma_vect_moins_un, double *b_gamma_vect,
								 double tau, double log_tau,
								 double *gamma_vect_prop, double *log_gamma_vect_prop,
								 double *gamma_vect_old, double *log_gamma_vect_old, int indice,
								 double *lpk_mean_prop, double *lpk_mean_old,
								 double *lpk_vect,
								 double *lp)
{
	int l;
	int i, i_three;
	int i_M1;
	int m1, m2;
	int m1_M, m1_M1, m1_M2;
	int m1_moins_M1, m1_moins_M1_M2;
	int indice_m_1, indice_m_2, indice_m_3;
	
	double indic_lkcp_lkpc = 1.0;

	double kcp;
	double kpc_i, kpc_i_sq;
	double ke_i_plus_kcp, ke_i_plus_kcp_sq;
	double inv_V_i, inv_V_i_sq;

	double gamma_1_over_V_i_sq;
	double gamma_1_kpc_i_over_V_i_sq;
	double gamma_1_kpc_i_sq_over_V_i_sq;
	
	double gamma_2_kcp;
	double gamma_2_kcp_sq;

	double tau_over_V_i, tau_over_V_i_sq;

	double logdetV1 = 0.0, logdetV2 = 0.0;
	double sign_detV1 = 0.0, sign_detV2 = 0.0;
	double intermedprod1 = 0.0, intermedprod2 = 0.0;

	double *V1_i, *v1_i, *V2_i, *v2_i;
	double *logdetV1_i, *logdetV2_i;
	double *sign_detV1_i, *sign_detV2_i;
	double *intermedprod1_i, *intermedprod2_i;

	V1_i = (double*)calloc(M_sq, sizeof(double));
	v1_i = (double*)calloc(M, sizeof(double));
	V2_i = (double*)calloc(M_sq, sizeof(double));
	v2_i = (double*)calloc(M, sizeof(double));
	logdetV1_i = (double*)calloc(1, sizeof(double));
	logdetV2_i = (double*)calloc(1, sizeof(double));
	sign_detV1_i = (double*)calloc(1, sizeof(double));
	sign_detV2_i = (double*)calloc(1, sizeof(double));
	intermedprod1_i = (double*)calloc(1, sizeof(double));
	intermedprod2_i = (double*)calloc(1, sizeof(double));
	
	double *gamma_vect, *log_gamma_vect;

	gamma_vect     = (double*)calloc(2, sizeof(double));
	log_gamma_vect = (double*)calloc(2, sizeof(double));
	
	for (l = 0; l < 2; l++)
	{
		gamma_vect[l]     = gamma_vect_old[l];
		log_gamma_vect[l] = log_gamma_vect_old[l];
	}
	gamma_vect[indice]     = gamma_vect_prop[indice];
	log_gamma_vect[indice] = log_gamma_vect_prop[indice];
	
	if (lpk_mean_prop[0] < lpk_mean_old[2])
	{
		indic_lkcp_lkpc = -1.0;
	}
	for (i = 0; i < I; i++)
	{
		if (lpk_mean_prop[0] < lpk_vect[(3*i + 1)])
		{
			indic_lkcp_lkpc = -1.0;
		}
	}
	if (lpk_mean_prop[0] < -3.0)
	{
		indic_lkcp_lkpc = -1.0;
	}	

	if ((indic_lkcp_lkpc <= 0) | (tau <= 0.0) | (gamma_vect[0] <= 0.0) | (gamma_vect[1] <= 0.0))
	{
		*lp = -1e15;
	} else
	{

		kcp = exp(lpk_mean_prop[0]);
		gamma_2_kcp    = gamma_vect[1]*kcp;
		gamma_2_kcp_sq = gamma_2_kcp*kcp;

		/*********************************************************************************************************/
		/* loop for all the subject                                                                              */
		/*********************************************************************************************************/

			for (i = 0; i < I; i++)
			{
				i_M1 = i*M1;
				i_three = i*3;

				/*************************************************************************************************/
				/* parameters for subject i                                                                      */
				/*************************************************************************************************/

					ke_i_plus_kcp    = exp(lpk_vect[i_three]) + kcp;
					ke_i_plus_kcp_sq = ke_i_plus_kcp*ke_i_plus_kcp;
					
					kpc_i    = exp(lpk_vect[(i_three + 1)]);
					kpc_i_sq = kpc_i*kpc_i;
					
					inv_V_i    = exp(-lpk_vect[(i_three + 2)]);
					inv_V_i_sq = inv_V_i*inv_V_i;

					gamma_1_over_V_i_sq          = gamma_vect[0]*inv_V_i_sq;
					gamma_1_kpc_i_over_V_i_sq    = gamma_1_over_V_i_sq*kpc_i;
					gamma_1_kpc_i_sq_over_V_i_sq = gamma_1_kpc_i_over_V_i_sq*kpc_i;
					
					tau_over_V_i    = tau*inv_V_i;
					tau_over_V_i_sq = tau*inv_V_i_sq;

				/*************************************************************************************************/
				/* matrices V2_i, V1_i, vectors v2_i et v1_i                                                     */
				/*************************************************************************************************/

					for (m1 = 0; m1 < M1; m1++)
					{
						m1_M = m1*M;
						m1_M1 = m1*M1;
						m1_M2 = m1*M2;
						for (m2 = 0; m2 < M1; m2++)
						{
							indice_m_1 = m1_M + m2;
							indice_m_2 = m1_M1 + m2;
							V1_i[indice_m_1] = gamma_1_over_V_i_sq*(P_11[indice_m_2] + ke_i_plus_kcp*P_0110[indice_m_2] + ke_i_plus_kcp_sq*P_00[indice_m_2]) +
											   gamma_2_kcp_sq*P_00[indice_m_2] +
											   prec_c_prior[indice_m_1];
							V2_i[indice_m_1] = V1_i[indice_m_1] +
											   tau_over_V_i_sq*tBB[indice_m_2];
						}
						for (m2 = M1; m2 < M; m2++)
						{
							indice_m_1 = m1_M + m2;
							indice_m_2 = m1_M2 + (m2 - M1);
							indice_m_3 = m2*M + m1;
							V1_i[indice_m_1] = - gamma_1_kpc_i_over_V_i_sq*(P_01[indice_m_2] + ke_i_plus_kcp*P_00[indice_m_2]) -
											   gamma_2_kcp*(P_10[indice_m_2] + kpc_i*P_00[indice_m_2]);
							V1_i[indice_m_3] = V1_i[indice_m_1];
							V2_i[indice_m_1] = V1_i[indice_m_1];
							V2_i[indice_m_3] = V1_i[indice_m_1];
						}
						v1_i[m1] = prec_c_prior_mu_c_prior[m1];
						v2_i[m1] = v1_i[m1] +
								   tau_over_V_i*tBy[(i_M1 + m1)];
					}

					for (m1 = M1; m1 < M; m1++)
					{
						m1_M = m1*M;
						m1_moins_M1 = m1 - M1;
						m1_moins_M1_M2 = m1_moins_M1*M2;
						for (m2 = M1; m2 < M; m2++)
						{
							indice_m_1 = m1_M + m2;
							indice_m_2 = m1_moins_M1_M2 + (m2 - M1);
							V1_i[indice_m_1] = gamma_vect[1]*(P_11[indice_m_2] + kpc_i*P_0110[indice_m_2] + kpc_i_sq*P_00[indice_m_2]) +
											   gamma_1_kpc_i_sq_over_V_i_sq*P_00[indice_m_2] +
											   prec_c_prior[indice_m_1];
							V2_i[indice_m_1] = V1_i[indice_m_1];
						}
						v1_i[m1] = prec_c_prior_mu_c_prior[m1];
						v2_i[m1] = v1_i[m1];
					}

				/*************************************************************************************************/
				/* log of the det of V1_i et V2_i                                                                */
				/* t(v1_i)*solveV1_i*v1_i et t(v2_i)*solveV2_i*v2_i                                              */
				/*************************************************************************************************/
				
					(*logdetV1_i) = 0.0;
					(*logdetV2_i) = 0.0;
					(*sign_detV1_i) = 1.0;
					(*sign_detV2_i) = 1.0;

					(*intermedprod1_i) = 0.0;
					(*intermedprod2_i) = 0.0;

					logdet_et_prod(V1_i, v1_i, M, logdetV1_i, sign_detV1_i, intermedprod1_i);
					logdet_et_prod(V2_i, v2_i, M, logdetV2_i, sign_detV2_i, intermedprod2_i);

					logdetV1 += (*logdetV1_i);
					logdetV2 += (*logdetV2_i);
					
					sign_detV1 += (*sign_detV1_i);
					sign_detV2 += (*sign_detV2_i);

					intermedprod1 += (*intermedprod1_i);
					intermedprod2 += (*intermedprod2_i);
			}
			
		/*********************************************************************************************************/
		/* log posterior conditional distribution                                                                */
		/*********************************************************************************************************/

			if ((sign_detV1 <= I_moins_un_demi) | (sign_detV2 <= I_moins_un_demi))
			{
				*lp = -1e15;
			} else
			{
				*lp = a_tau_post_moins_un*log_tau - tau*(b_tau + 0.5*sum_y_sq) -
					  0.5*(intermedprod1 - logdetV1 - intermedprod2 + logdetV2) -
					  0.5*prec_lpk[0]*(lpk_mean_prop[0] - mu_lpk[0])*(lpk_mean_prop[0] - mu_lpk[0]);
				for (l = 0; l < 2; l++)
				{
					*lp += (a_gamma_vect_moins_un[l]*log_gamma_vect[l] - b_gamma_vect[l]*gamma_vect[l]);
				}
			}
	}

	free(V1_i);
	free(v1_i);
	free(V2_i);
	free(v2_i);
	free(logdetV1_i);
	free(logdetV2_i);
	free(sign_detV1_i);
	free(sign_detV2_i);
	free(intermedprod1_i);
	free(intermedprod2_i);
	
	free(gamma_vect);
	free(log_gamma_vect);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for lpk_vect                                                           */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_lpk_vect(int N, int I, int I_3,
						   int M1, int M2, int M, int M_sq,
						   double *prec_c_prior, double *prec_c_prior_mu_c_prior,
						   double *P_11, double *P_10, double *P_01, double *P_00, double *P_0110,
						   double *tBB, double *tBy,
						   double tau,
						   double *gamma_vect,
						   double *lpk_mean, double *a_tau_lpk_post, double *b_tau_lpk, double max_tau_lpk,
						   int subject, double *lpk_vect_prop, double *lpk_vect_old,
						   double *lp)
{
	int i, l, i_three;
	int subject_M1, subject_three;
	int m1, m2;
	int m1_M, m1_M1, m1_M2;
	int m1_moins_M1, m1_moins_M1_M2;
	int indice_m_1, indice_m_2, indice_m_3;
	
	double indic = 1.0;
	
	double kcp;
	double kpc_subject, kpc_subject_sq;
	double ke_subject_plus_kcp, ke_subject_plus_kcp_sq;
	double inv_V_subject, inv_V_subject_sq;

	double gamma_1_over_V_subject_sq;
	double gamma_1_kpc_subject_over_V_subject_sq;
	double gamma_1_kpc_subject_sq_over_V_subject_sq;
	
	double gamma_2_kcp;
	double gamma_2_kcp_sq;

	double tau_over_V_subject, tau_over_V_subject_sq;

	double *b_tau_lpk_post;
	double *lpk_vect;

	double *V1_subject, *v1_subject, *V2_subject, *v2_subject;
	double *logdetV1_subject, *logdetV2_subject;
	double *sign_detV1_subject, *sign_detV2_subject;
	double *intermedprod1_subject, *intermedprod2_subject;

	b_tau_lpk_post = (double*)calloc(3, sizeof(double));
	lpk_vect = (double*)calloc(I_3, sizeof(double));

	V1_subject = (double*)calloc(M_sq, sizeof(double));
	v1_subject = (double*)calloc(M, sizeof(double));
	V2_subject = (double*)calloc(M_sq, sizeof(double));
	v2_subject = (double*)calloc(M, sizeof(double));
	logdetV1_subject = (double*)calloc(1, sizeof(double));
	logdetV2_subject = (double*)calloc(1, sizeof(double));
	sign_detV1_subject = (double*)calloc(1, sizeof(double));
	sign_detV2_subject = (double*)calloc(1, sizeof(double));
	intermedprod1_subject = (double*)calloc(1, sizeof(double));
	intermedprod2_subject = (double*)calloc(1, sizeof(double));

	subject_M1 = subject*M1;
	subject_three = subject*3;
		
	/*************************************************************************************************************/
	/* vector of individual lpk parameter                                                                        */
	/*************************************************************************************************************/
	
		for (i = 0; i < I; i++)
		{
			i_three = i*3;
			for (l = 0; l < 3; l++)
			{
				lpk_vect[(i_three + l)] = lpk_vect_old[(i_three + l)];
			}
		}
		for (l = 0; l < 3; l++)
		{
			lpk_vect[(subject_three + l)] = lpk_vect_prop[(subject_three + l)];
			if (lpk_vect[(subject_three + l)] < -4.0)
			{
				indic = -1.0;
			}
		}
		
	/*************************************************************************************************************/
	/* lkcp_subject > lkpc_subject                                                                               */
	/*************************************************************************************************************/
	
		if (lpk_mean[0] < lpk_vect[(subject_three + 1)])
		{
			indic = -1.0;
		}
		if (indic < 0.0)
		{
			*lp = -1e15;
		} else
		{

			/*****************************************************************************************************/
			/* pk parameters for subject subject                                                                 */
			/*****************************************************************************************************/

				kcp = exp(lpk_mean[0]);
				gamma_2_kcp    = gamma_vect[1]*kcp;
				gamma_2_kcp_sq = gamma_2_kcp*kcp;

				ke_subject_plus_kcp    = exp(lpk_vect[subject_three]) + kcp;
				ke_subject_plus_kcp_sq = ke_subject_plus_kcp*ke_subject_plus_kcp;

				kpc_subject    = exp(lpk_vect[(subject_three + 1)]);
				kpc_subject_sq = kpc_subject*kpc_subject;

				inv_V_subject    = exp(-lpk_vect[(subject_three + 2)]);
				inv_V_subject_sq = inv_V_subject*inv_V_subject;

				gamma_1_over_V_subject_sq                = gamma_vect[0]*inv_V_subject_sq;
				gamma_1_kpc_subject_over_V_subject_sq    = gamma_1_over_V_subject_sq*kpc_subject;
				gamma_1_kpc_subject_sq_over_V_subject_sq = gamma_1_kpc_subject_over_V_subject_sq*kpc_subject;
		
				tau_over_V_subject    = tau*inv_V_subject;
				tau_over_V_subject_sq = tau*inv_V_subject_sq;


			/*****************************************************************************************************/
			/* matrices V2_subject, V1_subject, vectors v2_subject et v1_subject                                 */
			/*****************************************************************************************************/

				for (m1 = 0; m1 < M1; m1++)
				{
					m1_M = m1*M;
					m1_M1 = m1*M1;
					m1_M2 = m1*M2;
					for (m2 = 0; m2 < M1; m2++)
					{
						indice_m_1 = m1_M + m2;
						indice_m_2 = m1_M1 + m2;
						V1_subject[indice_m_1] = gamma_1_over_V_subject_sq*(P_11[indice_m_2] + ke_subject_plus_kcp*P_0110[indice_m_2] + ke_subject_plus_kcp_sq*P_00[indice_m_2]) +
												 gamma_2_kcp_sq*P_00[indice_m_2] +
												 prec_c_prior[indice_m_1];
						V2_subject[indice_m_1] = V1_subject[indice_m_1] +
												 tau_over_V_subject_sq*tBB[indice_m_2];
					}
					for (m2 = M1; m2 < M; m2++)
					{
						indice_m_1 = m1_M + m2;
						indice_m_2 = m1_M2 + (m2 - M1);
						indice_m_3 = m2*M + m1;
						V1_subject[indice_m_1] = - gamma_1_kpc_subject_over_V_subject_sq*(P_01[indice_m_2] + ke_subject_plus_kcp*P_00[indice_m_2]) -
												 gamma_2_kcp*(P_10[indice_m_2] + kpc_subject*P_00[indice_m_2]);
						V1_subject[indice_m_3] = V1_subject[indice_m_1];
						V2_subject[indice_m_1] = V1_subject[indice_m_1];
						V2_subject[indice_m_3] = V1_subject[indice_m_1];
					}
					v1_subject[m1] = prec_c_prior_mu_c_prior[m1];
					v2_subject[m1] = v1_subject[m1] +
									 tau_over_V_subject*tBy[(subject_M1 + m1)];
				}

				for (m1 = M1; m1 < M; m1++)
				{
					m1_M = m1*M;
					m1_moins_M1 = m1 - M1;
					m1_moins_M1_M2 = m1_moins_M1*M2;
					for (m2 = M1; m2 < M; m2++)
					{
						indice_m_1 = m1_M + m2;
						indice_m_2 = m1_moins_M1_M2 + (m2 - M1);
						V1_subject[indice_m_1] = gamma_vect[1]*(P_11[indice_m_2] + kpc_subject*P_0110[indice_m_2] + kpc_subject_sq*P_00[indice_m_2]) +
												 gamma_1_kpc_subject_sq_over_V_subject_sq*P_00[indice_m_2] +
												 prec_c_prior[indice_m_1];
						V2_subject[indice_m_1] = V1_subject[indice_m_1];
					}
					v1_subject[m1] = prec_c_prior_mu_c_prior[m1];
					v2_subject[m1] = v1_subject[m1];
				}


			/*****************************************************************************************************/
			/* log of the det of V1_subject and V2_subject                                                       */
			/* t(v1_subject)*solveV1_subject*v1_subject and t(v2_subject)*solveV2_subject*v2_subject             */
			/*****************************************************************************************************/

				*logdetV1_subject = 0.0;
				*logdetV2_subject = 0.0;
				*sign_detV1_subject = 1.0;
				*sign_detV2_subject = 1.0;

				*intermedprod1_subject = 0.0;
				*intermedprod2_subject = 0.0;

				logdet_et_prod(V1_subject, v1_subject, M, logdetV1_subject, sign_detV1_subject, intermedprod1_subject);
				logdet_et_prod(V2_subject, v2_subject, M, logdetV2_subject, sign_detV2_subject, intermedprod2_subject);
			
			/*****************************************************************************************************/
			/* log posterior condition distribution                                                              */
			/*****************************************************************************************************/

				if (((*sign_detV1_subject) < 0.5) | ((*sign_detV2_subject) < 0.5))
				{
					*lp = -1e15;
				} else
				{
					*lp = 0.5*(((*logdetV1_subject) - (*intermedprod1_subject) - (*logdetV2_subject) + (*intermedprod2_subject)));
					for (l = 0; l < 3; l++)
					{
						b_tau_lpk_post[l] = 0.0;
						for (i = 0; i < I; i++)
						{
							b_tau_lpk_post[l] += ((lpk_vect[(i*3 + l)] - lpk_mean[(l + 1)])*(lpk_vect[(i*3 + l)] - lpk_mean[(l + 1)]));
						}
						b_tau_lpk_post[l] *= 0.5;
						b_tau_lpk_post[l] += b_tau_lpk[l];
					
						*lp += (-a_tau_lpk_post[l]*log(b_tau_lpk_post[l]) + pgamma(max_tau_lpk, a_tau_lpk_post[l], 1.0/b_tau_lpk_post[l], 1, 1));
					}
				}
		}

	free(b_tau_lpk_post);
	free(lpk_vect);

	free(V1_subject);
	free(v1_subject);
	free(V2_subject);
	free(v2_subject);
	free(logdetV1_subject);
	free(logdetV2_subject);
	free(sign_detV1_subject);
	free(sign_detV2_subject);
	free(intermedprod1_subject);
	free(intermedprod2_subject);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for lpk_mean                                                           */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_lpk_mean(int I,
						   double *mu_lpk, double *prec_lpk,
						   double *a_tau_lpk_post, double *b_tau_lpk, double max_tau_lpk,
						   double *lpk_mean_prop, double *lpk_mean_old, int indice,
						   double *lpk_vect,
						   double *lp)
{
	int i, l;

	double *lpk_mean;
	double *b_tau_lpk_post;
	
	lpk_mean = (double*)calloc(4, sizeof(double));
	b_tau_lpk_post = (double*)calloc(3, sizeof(double));
	
	for (l = 0; l < 4; l++)
	{
		lpk_mean[l] = lpk_mean_old[l];
	}
	lpk_mean[indice] = lpk_mean_prop[indice];
	
	if ((lpk_mean[0] < lpk_mean[2]) | (lpk_mean[1] < -3.0) | (lpk_mean[3] < -3.0))
	{
		*lp = -1e15;
	} else
	{

		for (l = 0; l < 3; l++)
		{
			b_tau_lpk_post[l] = 0.0;
			for (i = 0; i < I; i++)
			{
				b_tau_lpk_post[l] += ((lpk_vect[(i*3 + l)] - lpk_mean[(l + 1)])*(lpk_vect[(i*3 + l)] - lpk_mean[(l + 1)]));
			}
			b_tau_lpk_post[l] *= 0.5;
			b_tau_lpk_post[l] += b_tau_lpk[l];
		}

		*lp = 0.0;
		for (l = 0; l < 3; l++)
		{
			*lp += (-a_tau_lpk_post[l]*log(b_tau_lpk_post[l]) +
				   pgamma(max_tau_lpk, a_tau_lpk_post[l], 1.0/b_tau_lpk_post[l], 1, 1) -
				   0.5*prec_lpk[(l + 1)]*(lpk_mean[(l + 1)] - mu_lpk[(l + 1)])*(lpk_mean[(l + 1)] - mu_lpk[(l + 1)]));
		}
	}

	free(lpk_mean);
	free(b_tau_lpk_post);
}

/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* log conditional posterior distribution for lpk_tau                                                            */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

static void lpost_lpk_tau(int I, double double_I,
						  double *mu_lpk, double *prec_lpk,
						  double *a_tau_lpk_post_moins_un, double *b_tau_lpk, double max_tau_lpk,
						  double *lpk_tau_prop, double *log_lpk_tau_prop,
						  double *lpk_tau_old, double *log_lpk_tau_old,
						  int indice,
						  double *lpk_vect,
						  double *lp)
{
	int i, l;
	int i_three_plus_l;
	

	double *lpk_tau;
	double *log_lpk_tau;
	
	lpk_tau = (double*)calloc(3, sizeof(double));
	log_lpk_tau = (double*)calloc(3, sizeof(double));
	
	for (l = 0; l < 3; l++)
	{
		lpk_tau[l]     = lpk_tau_old[l];
		log_lpk_tau[l] = log_lpk_tau_old[l];
	}
	lpk_tau[indice]     = lpk_tau_prop[indice];
	log_lpk_tau[indice] = log_lpk_tau_prop[indice];

	double *sum_i_lpk_vect;
	double *sum_i_lpk_vect_sq;

	sum_i_lpk_vect = (double*)calloc(3, sizeof(double));
	sum_i_lpk_vect_sq = (double*)calloc(3, sizeof(double));
	

	if ((lpk_tau[indice] < 0.0) | ((lpk_tau[indice] > max_tau_lpk)))
	{
		*lp = -1e15;
	} else
	{
		for (l = 0; l < 3; l++)
		{
			sum_i_lpk_vect[l]    = 0.0;
			sum_i_lpk_vect_sq[l] = 0.0;
			for (i = 0; i < I; i++)
			{
				i_three_plus_l = i*3 + l;
				sum_i_lpk_vect[l]    += lpk_vect[i_three_plus_l];
				sum_i_lpk_vect_sq[l] += lpk_vect[i_three_plus_l]*lpk_vect[i_three_plus_l];
			}
		}

		*lp = 0.0;
		for (l = 0; l < 3; l++)
		{
			*lp += (a_tau_lpk_post_moins_un[l]*log_lpk_tau[l] - (b_tau_lpk[l] + 0.5*sum_i_lpk_vect_sq[l])*lpk_tau[l] +
					0.5*((lpk_tau[l]*sum_i_lpk_vect[l] + prec_lpk[(l + 1)]*mu_lpk[(l + 1)])*(lpk_tau[l]*sum_i_lpk_vect[l] + prec_lpk[(l + 1)]*mu_lpk[(l + 1)])/(double_I*lpk_tau[l] + prec_lpk[(l + 1)]) - log(double_I*lpk_tau[l] + prec_lpk[(l + 1)])));
		}
	}
	
	free(lpk_tau);
	free(log_lpk_tau);
	free(sum_i_lpk_vect);
	free(sum_i_lpk_vect_sq);
}