/*****************************************************************************************************************/
/*****************************************************************************************************************/
/* header file for matrix calculation                                                                            */
/* by Jaeger Jonathan , Universit de Lige, Belgium                                                            */
/*****************************************************************************************************************/
/*****************************************************************************************************************/

#include <R.h>
#include <Rinternals.h>
#include <Rmath.h> 
#include <R_ext/Lapack.h>

typedef double * vector;


/*****************************************************************************************************************/
/* matrix product                                                                                                */
/* arguments:                                                                                                    */
/*   - a matrix of dimension nra, nca,                                                                           */
/*   - b matrix of dimension nrb, ncb,                                                                           */
/*   - z product of matrices a and b (output).                                                                   */
/*****************************************************************************************************************/

static void prod_mat(double *a, int nra, int nca,
					 double *b, int nrb, int ncb,
					 double *z)
{

	double one = 1.0, zero = 0.0;
	char *transa = "N", *transb = "N";

	F77_CALL(dgemm)(transa, transb, &nra, &ncb, &nca, &one,
		            a, &nra, b, &nrb, &zero, z, &nra);
}

/*****************************************************************************************************************/
/* cholesky decomposition                                                                                        */
/* arguments:                                                                                                    */
/*   - a matrix of dimension m, m, (input & output)                                                              */
/*   - m dimension of matrix a.                                                                                  */
/*****************************************************************************************************************/

static void matchol(double *a, int m)
{
	int i, j, info;
	char *ul = "Upper";

	for (j=0; j<m; j++)
	{
		for (i=0; i<j; i++)
		{
			a[j + i*m] = 0;
		}
	}

	F77_CALL(dpotrf)(ul, &m, a, &m, &info);
}

/*****************************************************************************************************************/
/* inverse of a matrix                                                                                           */
/* arguments:                                                                                                    */
/*   - a matrix of dimension m, m, (input & output)                                                              */
/*   - m dimension of matrix a.                                                                                  */
/*****************************************************************************************************************/

static void matinvchol(double *a, int m)
{
	int i, j, info, j_m;
	char *ul = "Upper";

	for (j=0; j<m; j++)
	{
		for (i=0; i<j; i++)
		{
			a[j + i*m] = 0;
		}
	}

	F77_CALL(dpotrf)(ul, &m, a, &m, &info);
	F77_CALL(dpotri)(ul, &m, a, &m, &info);

	for (j=0; j<m; j++)
	{
		j_m = j*m;
		for (i=0; i<j; i++)
		{
			a[j + i*m] = a[i + j_m];
		}
	}
}


/*****************************************************************************************************************/
/* determinant of a symmetric positive definite matrix                                                           */
/* arguments:                                                                                                    */
/*   - determ of the matrix sig, (output)                                                                        */
/*   - sig matrix of dimension dd, dd                                                                            */
/*   - dd dimension of matrix sig,                                                                               */
/*   - ind_log incator for the log of the determinant (int_log = 1 => log),                                      */
/*   - sign_determ sign of the determinant of matrix sig.                                                        */
/*****************************************************************************************************************/

static void detmatsym(double *determ, double *sig, int dd, int ind_log, double *sign_determ)
{
	int r, s, r_dd, indice;
	double *intermedsig;
	
	intermedsig = (double*)calloc(dd * dd, sizeof(double));

	for (r=0; r<dd; r++){
		r_dd = r*dd;
		for (s=0; s<dd; s++){
			indice = r_dd + s;
			intermedsig[indice] = sig[indice];
		}
	}

	matchol(intermedsig, dd);

	*sign_determ = 1.0;
	if (ind_log==1){
		*determ = 0.0;
		for (r=0; r<dd; r++){
			indice = r + r*dd;
			if(intermedsig[indice]>0.0){
				*determ += 2*log(intermedsig[indice]);
			} else {
				*sign_determ = -1.0;
				r = dd;
			}
		}
	} else {
		*determ = 1.0;
		for (r=0; r<dd; r++){
			indice = r + r*dd;
			if(intermedsig[indice]>0.0){
				*determ = *determ * intermedsig[indice];
			} else {
				*sign_determ = -1.0;
				r = dd;
			}
		}
		*determ = *determ * (*determ);
	}

	free(intermedsig);
}


/*****************************************************************************************************************/
/* random generation of a multivariate normal distribution                                                       */
/* arguments:                                                                                                    */
/*   - mean is the mean vector of dimension d,                                                                   */
/*   - sig matrix of dimension d, d,                                                                             */
/*   - d dimension of mean & sig,                                                                                */
/*   - gen random generation of the multivariate normal distribution.                                            */
/*****************************************************************************************************************/

static void multnorm(double *mean, double *sig, int d, double *gen)
{
	int k, l, k_d, indice;
	double *intermedmn1, *intermedmn2;
	double *sauvegardesig;

	intermedmn1 = (double*)calloc(d,sizeof(double));
	intermedmn2 = (double*)calloc(d,sizeof(double));
	sauvegardesig = (double*)calloc(d*d,sizeof(double));

	for(k=0; k<d; k++){
		k_d = k*d;
		for(l=0; l<d; l++){
			indice = k_d + l;
			sauvegardesig[indice] = sig[indice];
		}
	}
	for (l=0; l<d; l++){
		GetRNGstate();
		intermedmn1[l] = rnorm(0.0, 1.0);
		PutRNGstate();
	}

	matchol(sig, d);
	prod_mat(intermedmn1, 1, d,
		     sig, d, d,
			 intermedmn2);
	for (l=0; l<d; l++){
		gen[l] = mean[l] + intermedmn2[l];
	}

	for(k=0; k<d; k++){
		k_d = k*d;
		for(l=0; l<d; l++){
			indice = k_d + l;
			sig[indice] = sauvegardesig[indice];
		}
	}

	free(intermedmn1);
	free(intermedmn2);
	free(sauvegardesig);
}

/*****************************************************************************************************************/
/* log of a determinant & matrix product                                                                         */
/* arguments:                                                                                                    */
/*   - M matrix of dimension m, m,                                                                               */
/*   - v vector of dimension m,                                                                                  */
/*   - logdet log of the determinant of M,                                                                       */
/*   - signdet sign of the determinant of M,                                                                     */
/*   - prod is t(v)%*%solve(M)%*%v.                                                                              */
/*****************************************************************************************************************/

static void logdet_et_prod(double *M, double *v, int m, double *logdet, double *signdet, double *prod)
{
	int i, j, info, r, indice, j_m;
	char *ul = "Upper";

	double *intprod1;

	intprod1 = (double*)calloc(m,sizeof(double));

	/* cholesky transformation                                                                                   */

	for (j=0; j<m; j++)
	{
		for (i=0; i<j; i++)
		{
			M[j + i*m] = 0.0;
		}
	}

	F77_CALL(dpotrf)(ul, &m, M, &m, &info);

	/* log of the determinant                                                                                    */

	*signdet = 1.0;
	*logdet = 0.0;

	for (r=0; r<m; r++)
	{
		indice = r + r*m;
		if(M[indice]>0.0)
		{
			*logdet += 2*log(M[indice]);
		} else {
			*signdet = -1.0;
			r = m;
		}
	}

	/* inverse of matrix M                                                                                       */

	F77_CALL(dpotri)(ul, &m, M, &m, &info);

	for (j=0; j<m; j++)
	{
		j_m = j*m;
		for (i=0; i<j; i++)
		{
			M[j + i*m] = M[i + j_m];
		}
	}

	/* matrix product t(v)%*%solve(M)%*%v                                                                        */

	prod_mat(v, 1, m,
			 M, m, m,
			 intprod1);
	prod_mat(intprod1, 1, m,
			 v, m, 1,
			 prod);

	free(intprod1);
}