###################################################################################################################
# author: Jona Cederbaum
# date: 08.04.2015
##################################################################################################################
# description: eigen decomposition of the covariance estimator evaluated on a fine grid and prediction of
## the FPC weights.
##################################################################################################################
eigen_decomp<-function(cov_B,cov_C,cov_U,sigmasq_int_hat,my_grid,var_level,N_B,N_C,N_U,
                       curve_info,nums_subject,nums_word,index,I,J,n,sigmasq_hat,use_RI){
  
  ##################################################
  # compute interval length of Riemann approximation
  ##################################################
  interv<-my_grid[2]-my_grid[1]
  
  ####################
  # extract components
  ####################
  t<-curve_info$t
  
  ######################################################
  # estimation of eigenvalues via spectral decomposition
  ######################################################
  # Note: if covariances matrices are not symmetric due to numerical inaccuracies
  # they are thus forced to be symmetric
  cov_B<-symmpart(cov_B)
  my_eigen_B<-eigen(cov_B,symmetric=TRUE)
  cov_B<-NULL
  
  
  lamB_hat<-my_eigen_B$values*interv              # resacaled eigenvalues
  neg_lamB<-which(lamB_hat<10^(-10)*max(lamB_hat))   
  
  if(length(neg_lamB)>0)
    warning(paste0(length(neg_lamB)," negative eigenvalues of B are truncated"))
  lamB_hat[neg_lamB]<-0                           # set negative eigen values to zero
  
  
  if(!use_RI){
    
    cov_C<-symmpart(cov_C)
    my_eigen_C<-eigen(cov_C,symmetric=TRUE)
    cov_C<-NULL
    
    lamC_hat<-my_eigen_C$values*interv               # resacaled eigenvalues
    neg_lamC<-which(lamC_hat<10^(-10)*max(lamC_hat))  
    
    if(length(neg_lamC)>0)
      warning(paste0(length(neg_lamC)," negative eigenvalues of C are truncated"))
    lamC_hat[neg_lamC]<-0                            # set negative eigen values to zero  
  }else{
    my_eigen_C<-eigen(cov_C)
    lamC_hat<-0
    neg_lamC<-NA
    symm_antisymm_C<-NA
  }
  
  cov_U<-symmpart(cov_U)
  my_eigen_U<-eigen(cov_U,symmetric=TRUE)
  cov_U<-NULL
  
  lamU_hat<-my_eigen_U$values*interv               # resacaled eigenvalues
  neg_lamU<-which(lamU_hat<10^(-10)*max(lamU_hat))  
  
  if(length(neg_lamU)>0)
    warning(paste0(length(neg_lamU)," negative eigenvalues of U are truncated"))
  lamU_hat[neg_lamU]<-0                            # set negative eigen values to zero  
  
  
  ###############################################
  # compute total variance and variance explained
  ###############################################
  total_var<-sum(lamB_hat*(lamB_hat>0)) + sum(lamC_hat*(lamC_hat>0)) + sum(lamU_hat*(lamU_hat>0)) + sigmasq_int_hat
  
  ##############################################
  # specify number of components to keep
  ## (depends on var_level and N if it is not NA)
  #############################################
  if(is.na(N_B)|is.na(N_C)|is.na(N_U)){  # if at least N_B OR N_U are NA
    prop<-N_B<-N_C<-N_U<-0
    while(prop<var_level){
      if(!use_RI){
        lam_all<-c(lamB_hat[N_B+1],lamC_hat[N_C+1],lamU_hat[N_U+1])
        N_all<-c(N_B,N_C,N_U)
        maxi<-which.max(lam_all)
        N_all[maxi]<- N_all[maxi]+1
        N_B<-N_all[1]
        N_C<-N_all[2]
        N_U<-N_all[3]
        prop<-(sum(lamB_hat[seq(len=N_B)])+sum(lamC_hat[seq(len=N_C)])+
                 sum(lamU_hat[seq(len=N_U)])+sigmasq_int_hat)/total_var
      }else{
        lam_all<-c(lamB_hat[N_B+1],lamU_hat[N_U+1])
        N_all<-c(N_B,N_U)
        maxi<-which.max(lam_all)
        N_all[maxi]<- N_all[maxi]+1
        N_B<-N_all[1]
        N_U<-N_all[2]
        prop<-(sum(lamB_hat[seq(len=N_B)])+sum(lamU_hat[seq(len=N_U)])+sigmasq_int_hat)/total_var
        
      }
    }
  }
  
  ##########################################
  # the following makes only sense 
  # if at least one score is to be estimated 
  # now N_B, N_C and N_U known
  ##########################################
  if(N_B!=0|N_C!=0|N_U!=0){
    
    #####################################################
    # truncate eigen values to level of explained variance
    ######################################################
    lamB_hat<-lamB_hat[seq(len=N_B),drop=FALSE]     # truncated eigenvalues for B
    lamC_hat<-lamC_hat[seq(len=N_C),drop=FALSE]     # truncated eigenvalues for C
    lamU_hat<-lamU_hat[seq(len=N_U),drop=FALSE]     # truncated eigenvalues for curve-specific deviations
    var_explained<-(sum(lamB_hat)+sum(lamC_hat)+sum(lamU_hat)+
                      sigmasq_int_hat)/total_var  # variance explained 
    
    ###############################
    # truncate and NPC
    # if one chosen eigenvalue is 0
    ###############################
    if(N_B>0){
      while(lamB_hat[N_B]<10^(-8)){
        N_B<-N_B-1
        if(N_B>0){
          lamB_hat<-lamB_hat[seq(len=N_B),drop=FALSE]
        }else{
          lamB_hat<-0
        }
        if(N_B==0) break  # if this was the only eigenvalue than stop
      }
    }
    
    if(N_C>0){
      while(lamC_hat[N_C]<10^(-8)){
        N_C<-N_C-1
        if(N_C>0){
          lamC_hat<-lamC_hat[seq(len=N_C),drop=FALSE]  
        }else{
          lamC_hat<-0
        }        
        if(N_C==0) break  # if this was the only eigenvalue than stop
      }
    }
    
    if(N_U>0){
      while(lamU_hat[N_U]<10^(-8)){
        N_U<-N_U-1
        if(N_U>0){
          lamU_hat<-lamU_hat[seq(len=N_U),drop=FALSE]  
        }else{
          lamU_hat<-0
        }        
        if(N_U==0) break  # if this was the only eigenvalue than stop
      }
    }
    
    ##############################
    ##############################
    # estimation of eigenfunctions
    ##############################
    ##############################
    
    # rescale eigenfunctions
    
    ######################
    # eigenfunctions for B
    ######################
    if(N_B!=0){  
      phiB_hat_grid<-(1/sqrt(interv))*my_eigen_B$vectors[,seq(len=N_B),drop=FALSE]  # rescaled phiB_hat_grid
      # evaluate on original data points
      phiB_hat_orig<-matrix(NA,ncol=N_B,nrow=length(unlist(t)))
      for(k in 1:N_B){
        phiB_hat_orig[,k]<-approx(x=my_grid,y=phiB_hat_grid[,k],xout=unlist(t),method="linear")$y
      }
    }else{# if N_B=0  
      phiB_hat_grid<-matrix(0,0,0)
      phiB_hat_orig<-matrix(0,0,0)
    }
    
    my_eigen_B<-NULL
    
    ######################
    # eigenfunctions for C
    ######################
    if(N_C!=0){  
      phiC_hat_grid<-(1/sqrt(interv))*my_eigen_C$vectors[,seq(len=N_C),drop=FALSE]  # rescaled phiC_hat_grid (every entry)
      # evaluate on original data points
      phiC_hat_orig<-matrix(NA,ncol=N_C,nrow=length(unlist(t)))
      for(k in 1:N_C){
        phiC_hat_orig[,k]<-approx(x=my_grid,y=phiC_hat_grid[,k],xout=unlist(t),method="linear")$y
      }
    }else{# if N_C=0  
      phiC_hat_grid<-matrix(0,0,0)
      phiC_hat_orig<-matrix(0,0,0)
    }  
    
    my_eigen_C<-NULL
    
    ######################
    # eigenfunctions for U
    ######################
    if(N_U!=0){
      phiU_hat_grid<-(1/sqrt(interv))*my_eigen_U$vectors[,seq(len=N_U),drop=FALSE]  # rescaled phiU_hat_grid
      # evaluate on original data points
      phiU_hat_orig<-matrix(NA,ncol=N_U,nrow=length(unlist(t)))
      
      for(k in 1:N_U){
        phiU_hat_orig[,k]<-approx(x=my_grid,y=phiU_hat_grid[,k],xout=unlist(t),method="linear")$y
      }
    }else{# if N_U=0
      phiU_hat_grid<-matrix(0,0,0)
      phiU_hat_orig<-matrix(0,0,0)
    }
    
    my_eigen_U<-NULL
    
    
    index<-NULL
    
    ########################################################
    # Preparations for estimation of the scores
    ########################################################
    
    #####################################
    # compute covariance of basis weights
    #####################################
    cat("compute covariance of scores","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
    if(!use_RI){
      N<-I*N_B+J*N_C+n*N_U  # I*N_B + J*N_C + n*N_U
    }else{
      N<-I*N_B+n*N_U  # I*N_B  + n*N_U
    }
    T_all<-nrow(curve_info)
    if(N_B>0){
      G_B<-diag(rep(lamB_hat,times=I))
    }else{
      G_B<-matrix(NA,ncol=0,nrow=0)
    }
    if(N_C>0){
      G_C<-diag(rep(lamC_hat,times=J))
    }else{
      G_C<-matrix(NA,ncol=0,nrow=0)
    }
    if(N_U>0){
      G_U<-diag(rep(lamU_hat,times=n))
    }else{
      G_U<-matrix(NA,ncol=0,nrow=0)
    }
    G<-bdiag(G_B,G_C,G_U) # is a sparse matrix
    
    ######################
    # invert covariance
    # of scores
    ######################
    
    cat("invert covariance of scores","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
    G_inverse<-try(solve(G,sparse=TRUE))
    G<-NULL
    
    # only continue if G can be inverted
    if(class(G_inverse)[1]!="try-error"){
      #############################
      # construct design matrix phi
      #############################
      #####################################
      # for speaker/first grouping variable
      #####################################
      if(N_B>0){
        help_blocks<-data.table(subject_long=curve_info$subject_long,phiB_hat_orig)
        blocks<-list()
        for(i in 1:I){
          blocks[[i]]<-as.matrix(subset(help_blocks,subset=subject_long==i,select=-subject_long),ncol=N_B)
        }
        # combine blocks
        phiB_block<-bdiag(blocks)
      }else{
        phiB_block<-matrix(0,0,0)
      }
      
      ###################################
      #for items/second grouping variable
      ###################################
      if(N_C>0){
        help_blocks<-data.table(subject_long=curve_info$subject_long,
                                word_long=curve_info$word_long,phiC_hat_orig)
        blocks<-list()
        for(i in 1:I){
          blocks[[i]]<-list()
          for(j in 1:J){
            blocks[[i]][[j]]<-as.matrix(subset(help_blocks,subset=subject_long==i&word_long==j,
                                               select=-c(subject_long,word_long)),ncol=N_C)
          }
          blocks[[i]]<-bdiag(blocks[[i]])
        }
        phiC_block<-do.call("rBind",blocks)
      }else{
        phiC_block<-matrix(0,0,0)
      }
      
      ############
      # for curves
      ############
      if(N_U>0){
        help_blocks<-data.table(n_long=curve_info$n_long,phiU_hat_orig)
        blocks<-list()
        for(i in 1:n){
          blocks[[i]]<-as.matrix(subset(help_blocks,subset=n_long==i,select=-n_long),ncol=N_U)    
        }
        phiU_block<-bdiag(blocks)
      }else{
        phiU_block<-matrix(0,0,0)
      }
      
      help_blocks<-NULL
      blocks<-NULL
      
      if(N_B>0){
        if(N_C>0){
          if(N_U>0){
            phi_all<-cBind(phiB_block,phiC_block,phiU_block) # if N_B>0, N_C>0, and N_U>0
          }else{
            phi_all<-cBind(phiB_block,phiC_block) # if N_B>0, N_C>0, but N_U=0
          }
        }else{
          if(N_U>0){
            phi_all<-cBind(phiB_block,phiU_block) # if N_B>0 and N_U>0, but N_C=0
          }else{
            phi_all<-phiB_block  #if only N_B>0
          }
        }
      }else{
        if(N_C>0){
          if(N_U>0){
            phi_all<-cBind(phiC_block,phiU_block) #if N_C>0 and N_U>0, but N_B=0
          }else{
            phi_all<-phiC_block # if only N_C>0
          }
        }else{
          phi_all<-phiU_block # if only N_U>0
        }
      }
      
      #################
      # compute bracket
      ################# 
      cat("compute bracket","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
      bracket<-sigmasq_hat*G_inverse+crossprod(phi_all)
      
      cat("svd","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
      svd<-svd(bracket,nu=0,nv=0)  # extract singular values (correspond to eigenvalues if existent)
      
      cat("cond","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
      cond<-abs(max(svd$d))/abs(min(svd$d)) # compute condition number
      
      cat("scores","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
      if(cond<=1e+10){  #if well conditioned
        xi_all_hat<-solve(bracket,t(phi_all)%*%curve_info$y_tilde) # compute scores via solving the system of equations
      }else{# if near to singular, use ginv()
        bracket_inverse<-ginv(matrix(bracket,nrow=nrow(bracket),ncol=ncol(bracket)))
        xi_all_hat<-bracket_inverse%*%t(phi_all)%*%curve_info$y_tilde # compute scores using ginv
      }
      
      ######################
      # compute scores
      # for regular or schur
      ######################
      
      #######################
      # determine which
      # scores belong to what
      #######################
      cat("which scores belong to what","; time ",format(Sys.time(),"%a %b %d %X"),"\n",sep="")
      if(N_B>0){  
        xiB_hat<-matrix(xi_all_hat[1:(N_B*I)],ncol=N_B,byrow=T)
        if(N_C>0){
          xiC_hat<-matrix(xi_all_hat[(N_B*I+1):(N_B*I+N_C*J)],ncol=N_C,byrow=T)
          if(N_U>0){
            xiU_hat<-matrix(xi_all_hat[(N_B*I+N_C*J+1):N],ncol=N_U,byrow=T)
          }else{ # if N_U=0
            xiU_hat<-rep(NA,n)
          }
        }else{ # if N_C=0
          if(use_RI){
            xiC_hat<-NA
          }else{
            xiC_hat<-rep(NA,J)  
          }
          
          if(N_U>0){
            xiU_hat<-matrix(xi_all_hat[(N_B*I+1):N],ncol=N_U,byrow=T)
          }else{
            xiU_hat<-rep(NA,n) # if N_C and N_U are 0
          }
        }
      }else{# if N_B=0
        xiB_hat<-rep(NA,I)
        if(N_C>0){
          xiC_hat<-matrix(xi_all_hat[1:(N_C*J)],ncol=N_C,byrow=T)
          if(N_U>0){
            xiU_hat<-matrix(xi_all_hat[(N_C*J+1):N],ncol=N_U,byrow=T)
          }else{# if N_B and N_U are 0
            xiU_hat<-rep(NA,n)
          }
        }else{# if N_B and N_C are 0
          if(use_RI){
            xiC_hat<-NA
          }else{
            xiC_hat<-rep(NA,J)  
          }
          
          xiU_hat<-matrix(xi_all_hat[1:N],ncol=N_U,byrow=T)
        }
      }
      
      
      phiB_hat_orig<-NULL
      phiC_hat_orig<-NULL
      phiU_hat_orig<-NULL
      
    }else{# if G cannot be inverted
      warning("scores cannot be computed due to inversion of G")
      xiB_hat<-NA
      xiC_hat<-NA
      xiU_hat<-NA
    }
    
  }else{# if N_B, N_C=0, and N_U =0
    xiB_hat<-NA
    xiC_hat<-NA
    xiU_hat<-NA
    phiB_hat_grid<-NA
    phiC_hat_grid<-NA
    phiU_hat_grid<-NA
    neg_lamB<-NA
    neg_lamC<-NA
    neg_lamU<-NA
    print(warning("no PCs chosen at all"))
  }
  
  gc()
  
  ###########################################################################################################  
  ########################
  # Output: Return results
  ########################
  results<-list(phiB_hat_grid=phiB_hat_grid,phiC_hat_grid=phiC_hat_grid,phiU_hat_grid=phiU_hat_grid,
                lamB_hat=lamB_hat,lamC_hat=lamC_hat,lamU_hat=lamU_hat,N_B=N_B,N_C=N_C,N_U=N_U,    
                total_var=total_var,var_explained=var_explained,xiB_hat=xiB_hat,xiC_hat=xiC_hat,xiU_hat=xiU_hat)
  return(results)  
}
######################################################################################################################## 
