#################################################
# HMM-BASED SEGMENTATION OF MORTALITY SURFACES  #
#                                               #
# E-M algorithm for estimating the parameters   #
# of a hidden Markov model with K states        #
#                                               #
# INPUT:                                        #
#  Dat (dataframe)                              #
#  K (numer of latent states)                   #
#  d (covergence tolerance)                     #
#                                               #
# OUTPUT:                                       #
#  beta.reg (array of state-specific reg. coef) #
#  Pi (vector init. prob. of the Markov chain)  #
#  PI (matrix transition prob. of the Markov c.)#
#  l.new (log-likelihood at the MLEs)           #
#  post.pi (array of posterior probabilities)   #
#                                               #
#  during exectution, the increasing values of  #
#  the log-likelihood at each iteration are     #
#  printed                                      #
#                                               #
#  NOTE: the output depends on the initial      #
#        values and the algorithm may diverge   #
#        to spurious solution (see Section 6:   #
#        "Computational details")               #                               #
#################################################

# input: data
Dat <- read.table("Dat.txt") 
################################################
# data frame Dat includes Italian person years #
# (pop) and death counts by tumor (deaths),    #
# clustered by year, age (in years) and sex    #
# (1=male, 2=female)                           #
################################################
# input: number of latent states
K <- 3 
# input: covergence tolarance
d <- 10^-8
# define response vector y and covariates array x 
T <- length(min(Dat$year):max(Dat$year))       
n <- nrow(Dat)/T    
y <- array(Dat$deaths,dim=c(n,T))

mo <- lm(1:(n*T) ~ Dat$pop + Dat$age + as.factor(Dat$sex))
X <- model.matrix(mo)
H <- ncol(X) # number of columns of the design matrix

x <- array(X,dim=c(n,T,H))

###################
# initialization  #
###################

beta.reg <- array(dim=c(K,H-1)) # H-1 because of the offset
fit 	<- array(dim=c(n,T,K))
pred <-array(dim=c(n,T,K))
W <- array(dim=c(n,T,K))

#  (random weights)
for(i in 1:n){
  for(t in 1:T){
    ww <- runif(K)
    W[i,t,] <- ww/sum(ww)
  }
}

for(k in 1:K){
  mod <- glm(deaths ~ offset(log(pop)) + age + as.factor(sex), 
             weights=as.vector(W[,,k]),data=Dat,family="poisson")
  beta.reg[k,] <- coef(mod)
  pred[,,k] <- predict.glm(mod,type="response")
  fit[,,k] <- dpois(Dat$deaths,pred[,,k])
}


Pi <-runif(K)
Pi <- Pi/sum(Pi)

PI <- matrix(runif(K*K),K,K)
for(k in 1:K){
  PI[k,] <- PI[k,]/sum(PI[k,])
}
#Baum-Welc quantities as defined in the Appendix
psi   <- array(dim=c(n,T,K))
alpha <- array(dim=c(n,T,K))
C	<- array(dim=c(n,T))
Beta <- array(dim=c(n,T,K))

post.pi <- array(dim=c(n,T,K)) # univariate post. probabilities
post.bi.pi <- array(dim=c(n,T-1,K,K)) # bivariate postirior probabilities

step=1

l.new=1
l.old=0

l0=0
l <- list()
while(abs((l.new-l.old)/(l.new-l0)) > d){

  ###############
  # E step	    #
  ###############
  
  for(i in 1:n){
    
    psi[i,1,] <- Pi	
    C[i,1] <- sum(psi[i,1,]*fit[i,1,])
    alpha[i,1,] <- (psi[i,1,]*fit[i,1,])/C[i,1] 
    for(t in 2:T){
      for(k in 1:K){
        psi[i,t,k] <- sum(alpha[i,t-1,]*PI[,k])
      }
      C[i,t] <- sum(psi[i,t,]*fit[i,t,])
      alpha[i,t,] <- (psi[i,t,]*fit[i,t,])/C[i,t] 
      
    }
    
    Beta[i,T,] <- 1/C[i,T]
    
    for(t in (T-1):1){
      
      for(k in 1:K){
        
        Beta[i,t,k] <- (sum(PI[k,]*fit[i,t+1,]*Beta[i,t+1,]))/C[i,t]
        
      }
    }
  }
  
  l[[step]] <- sum(log(C)) # loglikelihood reached at the step
  l.new <- l[[step]]
  l.old <- ifelse(step==1,l0,l[[step-1]])
  
  print(l[[step]])
  
  for(i in 1:n){
    for(k in 1:K){
      for(t in 1:T){
        post.pi[i,t,k] <- (alpha[i,t,k]*Beta[i,t,k])/sum(alpha[i,t,]*Beta[i,t,])		
      }
    }
    
    
    for(t in 1:(T-1)){
      for(h in 1:K){
        for(k in 1:K){
          post.bi.pi[i,t,h,k]   <- alpha[i,t,h]*PI[h,k]*fit[i,t+1,k]*Beta[i,t+1,k] 
        }
      }
    }
  }
  
  W <- post.pi
  
  ##########
  # M step #
  ##########
  
  for(k in 1:K){
    
    Pi[k] <- sum(post.pi[,1,k])/sum(post.pi[,1,])
    for(h in 1:K){
      
      PI[h,k] <- sum(post.bi.pi[,,h,k])/sum(post.bi.pi[,,h,]) 
      
    }
  }
  
  
  for(k in 1:K){
    
    mod <- glm(deaths ~ offset(log(pop)) + age + as.factor(sex),
               weights=as.vector(W[,,k]),data=Dat,family="poisson")
    beta.reg[k,] <- coef(mod)
    pred[,,k] <- predict.glm(mod,type="response")
    fit[,,k] <- dpois(Dat$deaths,pred[,,k])
  }
  step=step+1
} 

beta.reg #(array of state-specific reg. coef)
Pi       #(vector init. prob. of the Markov chain)
PI       #(matrix transition prob. of the Markov c.)
l.new    #(log-likelihood at the MLEs)
post.pi  #(array of posterior probabilities)


