# Packages
using DataFrames       # counts require DataFrames
using Distributions
using JLD              # save files: save("output.jld", "output", output)
# using KernelDensity  # KernelDensity.PyPlot_init()
# using PyPlot
# using Plots          # pyplot()
# using Gadfly


# Gibbs sampling for latent variables a
function sample_a(mu::Float64, sigma::Float64, link::Int64)
  if link==1
    s = rand(TruncatedNormal(mu, sigma, 0, Inf))
  else
    s = rand(TruncatedNormal(mu, sigma, -Inf, 0))
  end
  return s
end

# Gibbs sampling for phi (for static model and DynII model)
function sample_phi(c::Array{Int64,1}, ssr::Array{Int64,1}, ssc::Array{Int64,1},
  phi::Array{Float64,1}, a1::Array{Float64,1}, vphi0::Float64, T::Int64)

  G = maximum(c)              # no. of clusters in DP
  for g in 1:G
    cg = find(c.==g)         # find subjects in group g
    ir = findin(ssr, cg)     # find row indices in group g
    ic = findin(ssc, cg)     # find column indices in group g
    rc = intersect(ir, ic)   # entries: both row & column indices are in group g
    rcc = union(ir, ic)
    rcc = setdiff(rcc, rc)   # entries: only row or only column in group g
    ic = intersect(rcc, ic)
    ir = intersect(rcc, ir)
    v = 1/(1/vphi0 + 4T*length(rc) + T*length(rcc))
    mu = v * (2sum(a1[rc]) + sum(a1[ir] - T*phi[c[ssc[ir]]]) + sum(a1[ic] - T*phi[c[ssr[ic]]]))
    phi[g] = rand(Normal(mu, sqrt(v)))
  end
  return phi
end

# Gibbs sampling for phi (for DynI model)
function sample_phi_dyn(c::Array{Int64,2}, ssr::Array{Int64,1},
  ssc::Array{Int64,1}, phi::Array{Float64,1}, a1::Array{Float64,2},
  vphi0::Float64)

  n, T = size(c)
  G = maximum(c)              # no. of clusters in DP
  for g in 1:G
    precision = 1/vphi0
    mu = 0.0
    for t in 1:T
      cg = find(c[:, t].==g)   # find subjects in group g
      ir = findin(ssr, cg)     # find row indices in group g
      ic = findin(ssc, cg)     # find column indices in group g
      rc = intersect(ir, ic)   # entries: both row & column indices are in group g
      rcc = union(ir, ic)
      rcc = setdiff(rcc, rc)   # entries: only row or only column in group g
      ic = intersect(rcc, ic)
      ir = intersect(rcc, ir)
      precision += 4length(rc) + length(rcc)
      mu += (2sum(a1[rc, t]) + sum(a1[ir, t] - phi[c[ssc[ir], t]])
             + sum(a1[ic, t] - phi[c[ssr[ic], t]]))
    end
    phi[g] = rand(Normal(mu/precision, 1/sqrt(precision)))
  end
  return phi
end

# Gibbs sampling for c and phi  (for static model and DynII model)
function sample_c(i::Int64, a1::Array{Float64,1}, ssr::Array{Int64,1},
  ssc::Array{Int64,1}, c::Array{Int64,1}, phi::Array{Float64,1},
  alpha::Float64, vphi0::Float64, T::Int64)

  n = length(c)
  ir = find(ssr.==i)      # entries s.t row index is i
  ic = find(ssc.==i)      # entries s.t column index is i
  asum = sum(a1[ir] - T*phi[c[ssc[ir]]]) + sum(a1[ic] - T*phi[c[ssr[ic]]])
  phi_idx = sort(unique(c))
  n_c = counts(c, 1:maximum(c))
  if n_c[c[i]]==1
    deleteat!(phi,c[i])          # remove the c[i] mixture component and its index
    deleteat!(phi_idx,c[i])
  end
  deleteat!(c,i)                 # remove ith element
  unique_elem = unique(c)
  c = indexin(c, unique_elem)    # renumber the components so that they range from 1 to max number of components
  phi_idx = indexin(phi_idx, unique_elem)
  phi = phi[sortperm(phi_idx)]
  K = length(unique_elem)
  n_c = counts(c, 1:K)
  p = Array(Float64, K+1)
  v = 1/(T*(n-1) + 1/vphi0)
  p[1:K] = log(n_c) + asum*phi - 0.5T*(n-1)*(phi.^2)
  p[K+1] = log(alpha) + 0.5log(v) - 0.5log(vphi0) + 0.5v*asum^2
  p = exp(p-maximum(p))
  p /= sum(p)
  ci = rand(Categorical(p))
  insert!(c, i, ci)
  if ci == (K+1)
    push!(phi, rand(Normal(v*asum, sqrt(v))))
  end
  return c, phi
end

# Gibbs sampling for c and phi  (for DynI model)
function sample_c_dyn(i::Int64, t::Int64, a1::Array{Float64,2}, ssr::Array{Int64,1},
  ssc::Array{Int64,1}, c::Array{Int64,2}, phi::Array{Float64,1},
  alpha::Float64, vphi0::Float64)

  n, T = size(c)
  ir = find(ssr.==i)      # entries s.t row index is i
  ic = find(ssc.==i)      # entries s.t column index is i
  asum = sum(a1[ir, t] - phi[c[ssc[ir], t]]) + sum(a1[ic, t] - phi[c[ssr[ic], t]])
  phi_idx = sort(unique(c))
  n_c = counts(c, 1:maximum(c))
  if n_c[c[i, t]]==1
    deleteat!(phi, c[i, t])          # remove the c[i] mixture component and its index
    deleteat!(phi_idx, c[i, t])
  end
  c[i,t] = 0                 # remove ith element
  unique_elem = unique(c)
  unique_elem = setdiff(unique_elem, 0)
  c = indexin(c, unique_elem)    # renumber the components so that they range from 1 to max number of components
  phi_idx = indexin(phi_idx, unique_elem)
  phi = phi[sortperm(phi_idx)]
  K = length(unique_elem)
  n_c = counts(c, 1:K)
  p = Array(Float64, K+1)
  v = 1/(T*(n-1) + 1/vphi0)
  p[1:K] = log(n_c) + asum*phi - 0.5*(n-1)*phi.^2
  p[K+1] = log(alpha) + 0.5log(v) - 0.5log(vphi0) + 0.5v*asum^2
  p = exp(p-maximum(p))
  p /= sum(p)
  c[i,t] = rand(Categorical(p))
  if c[i,t] == (K+1)
    push!(phi, rand(Normal(v*asum, sqrt(v))))
  end
  return c, phi
end

# Gibbs sampling for z
function sampleAlg2_z(i::Int64, a1::Array{Float64,1}, z::Array{Int64,1},
  ssr::Array{Int64,1}, ssc::Array{Int64,1}, nu::Float64,
  beta::Array{Float64,1}, vbeta0::Float64, T::Int64)

  n = length(z)
  ri = find(ssr.==i)            # indices of elements in row i
  ci = find(ssc.==i)            # indices of elements in column i
  ai = a1[union(ri,ci)]
  jindex = union(ssc[ri], ssr[ci])

  beta_idx = sort(unique(z))
  n_z = counts(z, 1:maximum(z))
  if n_z[z[i]]==1
    deleteat!(beta, z[i])
    deleteat!(beta_idx, z[i])
  end
  deleteat!(z,i)                 # remove ith element
  unique_elem = unique(z)
  z = indexin(z, unique_elem)
  beta_idx = indexin(beta_idx, unique_elem)
  beta = beta[sortperm(beta_idx)]
  K = length(unique_elem)
  n_z = counts(z, 1:K)
  p = Array(Float64, K+1)
  I = collect(1:n)
  deleteat!(I,i)
  for k = 1:K
    p[k] = (log(n_z[k]) + sum(ai[z[indexin(jindex, I)].==k])*beta[k]
           - 0.5T*n_z[k]*beta[k]^2)
  end
  p[K+1] = log(nu)
  p = exp(p-maximum(p))
  p /= sum(p)
  zi = rand(Categorical(p))
  insert!(z, i, zi)
  if zi == K+1
    push!(beta, rand(Normal(0.0, sqrt(vbeta0))))
  end
  return z, beta
end

# Gibbs sampling for z
function sampleAlg3_z(i::Int64, a1::Array{Float64,1}, z::Array{Int64,1}, ssr::Array{Int64,1},
    ssc::Array{Int64,1}, nu::Float64, vbeta0::Float64, T::Int64)

    n = length(z)                    # total no. of actors
    num = length(a1)                 # total no. of possible links
    ri = find(ssr.==i)               # indices of links in row i
    ci = find(ssc.==i)               # indices of links in column i
    W = union(ri, ci)                # indices of links in row i or column i
    jindex = union(ssc[ri], ssr[ci]) # indices of actors interacting with actor i (correspond to W)
    Wc = setdiff(1:num, W)           # indices of links not involving actor i

    deleteat!(z,i)                  # remove ith element
    unique_elem = unique(z)         # unique elements in z[-i]
    K = length(unique_elem)         # no. of unique elements in z[-i]
    z = indexin(z, unique_elem)     # renumber components from 1:K
    n_z_i = counts(z, 1:K)          # no. of actors assigned to each comp (exclude actor i)
    p = Array(Float64, K+1)         # probability vector

    # posterior of beta
    I = collect(1:n)
    deleteat!(I,i)                  # remove actor i
    Ir = indexin(ssr[Wc], I)        # renumber for extracting purpose (ith element is gone)
    Ic = indexin(ssc[Wc], I)        # now z is a vector of length (n-1)
    ZWc = z[Ir]                     # comp of one side of each link
    ZWc[z[Ir].!=z[Ic]] = 0
    SWc = Array(Float64,K)          # posterior means of beta based on links excluding actor i
    for k in 1:K
        SWc[k] = sum((a1[Wc])[ZWc.==k])
    end
    pWc = 1/vbeta0 + T*counts(ZWc,1:K) # posterior precisions

    for k in 1:K
        pnew = pWc[k] + T*n_z_i[k]    # new posterior precisions
        Snew = SWc[k] + sum((a1[W])[z[indexin(jindex, I)].==k])  # new posterior means
        p[k] = log(n_z_i[k]) + 0.5log(pWc[k]/pnew) - 0.5SWc[k]^2/pWc[k] + 0.5Snew^2/pnew
    end
    p[K+1] = log(nu)
    p = exp(p-maximum(p))
    p /= sum(p)
    zi = rand(Categorical(p))
    insert!(z, i, zi)
    return z
end

function sample_DPprecision(n::Int64, L::Int64, alpha::Float64,
  a_alpha::Float64, b_alpha::Float64)
  # L = no. of clusters in DP
  eta = rand(Beta(alpha+1, n))      # sample eta
  p = 1/(1 + n * (b_alpha - log(eta)) / (a_alpha + L - 1))
  r = rand(Bernoulli(p))
  if r==1
    alpha = rand(Gamma(a_alpha + L, 1/(b_alpha-log(eta))))
  else
    alpha = rand(Gamma(a_alpha + L - 1, 1/(b_alpha-log(eta))))
  end
  return alpha
end

function sample_eta(y::Array{Int64,2}, a::Array{Float64,2},
  ssr::Array{Int64,1}, ssc::Array{Int64,1}, c::Array{Int64,1}, phi::Array{Float64,1},
  Z::Array{Int64,1}, beta::Array{Float64,1}, veta0::Float64)

  num, T = size(y)
  betaZ = zeros(num)
  betaZ[Z.!=0] = beta[Z[Z.!=0]]
  v_eta = 1/(1/veta0 + sum(y[:, 1:(T-1)].^2))
  mu_eta = v_eta*(sum(y[:, 1:(T-1)].*a[:,2:end])- vecdot(sum(y[:, 1:(T-1)], 2),
           phi[c[ssr]] + phi[c[ssc]] + betaZ))
  eta = rand(Normal(mu_eta, sqrt(v_eta)))
  return eta
end

function Comm(y::Array{Int64,2}, N::Int64, burnin::Int64, thin::Int64,
  gamma0::Float64, v0::Float64, initcomp::Int64, seed::Int64)
  # assume y is symmetric (undirected) adjacency matrix

  n = size(y, 1)                             # no. of subjects
  id = find(triu(ones(n,n), 1).==1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n,n), 1).==1) # row indices and column indices
  num = length(id)                           # no. of non-diagonal elements
  y = y[id]                                  # convert y to vector (upper-triangle only)

  # Hyperparameters
  a_nu = b_nu = gamma0
  vbeta0 = v0

  # save samples
  num_save = Int((N-burnin)/thin)
  nu_save = Array(Float64, num_save)
  z_save = Array(Int64, num_save, n)
  beta_save = Array(Array{Float64,1}, num_save)

  it_save = 0
  srand(seed)                                   # set random seed

  # Initialization
  a = zeros(num)
  for i in 1:num
    a[i] = sample_a(0.0, 1.0, y[i]);
  end

  K = initcomp                           # no. of initial clusters in DP (communities)
  nu = 1.0                               # DP hyperparameter
  z = rand(Categorical(1/K*ones(K)), n)  # z: vector of length n indicating cluster membership of each subject
  z = indexin(z, unique(z))              # so that numbering of components start from 1 to max no. of unique components
  beta = randn(maximum(z))               # initialize the mixture components

  for it in 1:N

    # sample a
    for i in 1:num
      mu_a_i = 0.0
      if z[ssr[i]]==z[ssc[i]]
        mu_a_i += beta[z[ssr[i]]]
      end
      a[i] = sample_a(mu_a_i, 1.0, y[i])
    end

    # sample z
    for i in 1:n
      z, beta = sampleAlg2_z(i, a, z, ssr, ssc, nu, beta, vbeta0, 1)
    end
    K = maximum(z)
    Z = z[ssr]
    Z[z[ssr].!=z[ssc]]=0

    # sample beta
    P = counts(Z, 1:K) + 1/vbeta0    # precision
    S = Array(Float64,K)
    for k in 1:K
      S[k] = sum(a[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # sample nu
    nu = sample_DPprecision(n, K, nu, a_nu, b_nu)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      nu_save[it_save] = copy(nu)
      z_save[it_save,:] = copy(z)
      beta_save[it_save] = copy(beta)
    end

    if mod(it,1000)==0        # print every 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  output = Dict("nu_save" => nu_save, "z_save" => z_save, "beta_save" => beta_save)
  return output
end

function Pop(y::Array{Int64,2}, N::Int64, burnin::Int64, thin::Int64,
  gamma0::Float64, v0::Float64, initcomp::Int64, seed::Int64)
  # assume y is symmetric (undirected) adjacency matrix

  n = size(y, 1)                             # no. of subjects
  id = find(triu(ones(n,n), 1).==1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n,n), 1).==1) # row indices and column indices
  num = length(id)                           # no. of non-diagonal elements
  y = y[id]                                  # convert y to vector (upper-triangle only)

  # Hyperparameters
  a_alpha = b_alpha = gamma0
  vphi0 = v0

  # save samples
  num_save = Int((N-burnin)/thin)
  alpha_save = Array(Float64, num_save)
  c_save = Array(Int64, num_save, n)
  phi_save = Array(Array{Float64,1}, num_save)
  it_save = 0
  srand(seed)                           # set random seed

  # Initialization
  a = zeros(num)
  for i in 1:num
    a[i] = sample_a(0.0, 1.0, y[i]);
  end

  L = initcomp                           # no. of initial clusters in DP (popularity)
  alpha = 1.0                            # DP hyperparameter
  c = rand(Categorical(1/L*ones(L)), n)  # c: vector of length n indicating cluster membership of each subject
  c = indexin(c, unique(c))
  phi = randn(maximum(c))                # initialize the mixture components

  for it in 1:N

    # sample a
    for i in 1:num
      mu_a_i = phi[c[ssr[i]]] + phi[c[ssc[i]]]
      a[i] = sample_a(mu_a_i, 1.0, y[i])
    end

    # Sample c
    for i in 1:n
      c, phi = sample_c(i, a, ssr, ssc, c, phi, alpha, vphi0, 1)
    end

    # sample phi
    phi = sample_phi(c, ssr, ssc, phi, a, vphi0, 1)

    # sample alpha
    alpha = sample_DPprecision(n, maximum(c), alpha, a_alpha, b_alpha)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      alpha_save[it_save] = copy(alpha)
      c_save[it_save,:] = copy(c)
      phi_save[it_save] = copy(phi)
    end

    if mod(it,1000)==0     # print every 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  output = Dict("alpha_save" => alpha_save,  "c_save" => c_save, "phi_save" => phi_save)
  return output
end

function Static(y::Array{Int64,2}, N::Int64, burnin::Int64, thin::Int64,
  gamma0::Float64, v0::Float64, initcomp::Int64, seed::Int64, x...)
  # assume y is symmetric (undirected) adjacency matrix
  # x = (clp, phi) is a tuple

  n = size(y, 1)                             # no. of subjects
  id = find(triu(ones(n,n), 1).==1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n,n), 1).==1) # row indices and column indices
  num = length(id)                           # no. of non-diagonal elements
  y = y[id]                                  # convert y to vector (upper-triangle only)

  # Hyperparameters
  a_nu = b_nu = a_alpha = b_alpha = gamma0
  vbeta0 = vphi0 = v0

  # save samples
  num_save = Int((N-burnin)/thin)
  nu_save = Array(Float64, num_save)
  alpha_save = Array(Float64, num_save)
  c_save = Array(Int64, num_save, n)
  z_save = Array(Int64, num_save, n)
  beta_save = Array(Array{Float64,1}, num_save)
  phi_save = Array(Array{Float64,1}, num_save)

  it_save = 0
  srand(seed)                                   # set random seed

  # Initialization
  a = zeros(num)
  for i in 1:num
    a[i] = sample_a(0.0, 1.0, y[i]);
  end

  K = initcomp                           # no. of initial clusters in DP (communities)
  nu = 1.0                               # DP hyperparameter
  z = rand(Categorical(1/K*ones(K)), n)  # z: vector of length n indicating cluster membership of each subject
  z = indexin(z, unique(z))              # so that numbering of components start from 1 to max no. of unique components
  beta = randn(maximum(z))               # initialize the mixture components

  alpha = 1.0                            # DP hyperparameter
  if length(x)>0
    c = copy(x[1])
    L = maximum(c)
    phi = copy(x[2])
  else
    L = initcomp                           # no. of initial clusters in DP (popularity)
    c = rand(Categorical(1/L*ones(L)), n)  # c: vector of length n indicating cluster membership of each subject
    c = indexin(c, unique(c))
    phi = randn(maximum(c))                # initialize the mixture components
  end

  for it in 1:N

    # sample a
    for i in 1:num
      mu_a_i = phi[c[ssr[i]]] + phi[c[ssc[i]]]
      if z[ssr[i]]==z[ssc[i]]
        mu_a_i += beta[z[ssr[i]]]
      end
      a[i] = sample_a(mu_a_i, 1.0, y[i])
    end

    # sample z
    a1 = a - phi[c[ssr]] - phi[c[ssc]]
    for i in 1:n
      z, beta = sampleAlg2_z(i, a1, z, ssr, ssc, nu, beta, vbeta0, 1)
    end
    K = maximum(z)
    Z = z[ssr]
    Z[z[ssr].!=z[ssc]]=0

    # sample beta
    P = counts(Z, 1:K) + 1/vbeta0    # precision
    S = Array(Float64,K)
    for k in 1:K
      S[k] = sum(a1[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # sample nu
    nu = sample_DPprecision(n, K, nu, a_nu, b_nu)

    # Sample c
    betaZ = zeros(num)
    betaZ[Z.!=0] = beta[Z[Z.!=0]]
    a1 = a - betaZ
    for i in 1:n
      c, phi = sample_c(i, a1, ssr, ssc, c, phi, alpha, vphi0, 1)
    end

    # sample phi
    phi = sample_phi(c, ssr, ssc, phi, a1, vphi0, 1)

    # sample alpha
    alpha = sample_DPprecision(n, maximum(c), alpha, a_alpha, b_alpha)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      nu_save[it_save] = copy(nu)
      alpha_save[it_save] = copy(alpha)
      z_save[it_save,:] = copy(z)
      c_save[it_save,:] = copy(c)
      beta_save[it_save] = copy(beta)
      phi_save[it_save] = copy(phi)
    end

    if mod(it,1000)==0           # print everytime 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  output = Dict("nu_save" => nu_save, "alpha_save" => alpha_save, "z_save" => z_save,
  "c_save" => c_save, "beta_save" => beta_save, "phi_save" => phi_save)
  return output
end

function Static_fix(y::Array{Int64,2}, N::Int64, burnin::Int64, thin::Int64,
  gamma0::Float64, v0::Float64, seed::Int64, c::Array{Int64,1},
  z::Array{Int64,1}, phi::Array{Float64,1}, beta::Array{Float64,1})
  # assume y is symmetric (undirected) adjacency matrix
  # x = (clp, phi) is a tuple

  n = size(y, 1)                             # no. of subjects
  id = find(triu(ones(n,n), 1).==1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n,n), 1).==1) # row indices and column indices
  num = length(id)                           # no. of non-diagonal elements
  y = y[id]                                  # convert y to vector (upper-triangle only)

  # Hyperparameters
  a_nu = b_nu = a_alpha = b_alpha = gamma0
  vbeta0 = vphi0 = v0

  # save samples
  num_save = Int((N-burnin)/thin)
  beta_save = Array(Float64, num_save, maximum(z))
  phi_save = Array(Float64, num_save, maximum(c))
  it_save = 0
  srand(seed)                                   # set random seed

  # Initialization
  a = zeros(num)
  for i in 1:num
    a[i] = sample_a(0.0, 1.0, y[i]);
  end
  K = maximum(z)                # no. of clusters in DP (communities)
  Z = z[ssr]
  Z[z[ssr].!=z[ssc]]=0

  for it in 1:N

    # sample a
    for i in 1:num
      mu_a_i = phi[c[ssr[i]]] + phi[c[ssc[i]]]
      if z[ssr[i]]==z[ssc[i]]
        mu_a_i += beta[z[ssr[i]]]
      end
      a[i] = sample_a(mu_a_i, 1.0, y[i])
    end

    # sample beta
    a1 = a - phi[c[ssr]] - phi[c[ssc]]
    P = counts(Z, 1:K) + 1/vbeta0    # precision
    S = Array(Float64,K)
    for k in 1:K
      S[k] = sum(a1[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # Sample c
    betaZ = zeros(num)
    betaZ[Z.!=0] = beta[Z[Z.!=0]]
    a1 = a - betaZ
    phi = sample_phi(c, ssr, ssc, phi, a1, vphi0, 1)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      beta_save[it_save,:] = copy(beta)
      phi_save[it_save,:] = copy(phi)
    end

    if mod(it,1000)==0           # print everytime 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  return beta_save, phi_save
end

function DynI(Y::Array{Array{Int64,2},1}, N::Int64, burnin::Int64,
  thin::Int64, gamma0::Float64, v0::Float64, initcomp::Int, seed::Int64)
  # assume y is symmetric (undirected) adjacency matrix

  T = length(Y)                                # no. of time points
  n = size(Y[1], 1)                            # no. of subjects
  id = find(triu(ones(n, n), 1).== 1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n, n), 1).== 1) # row indices and column indices
  num = length(id)                             # no. of non-diagonal elements
  y = Array(Int64, num, T)                     # y: matrix contain upper-triangle only
  for t = 1:T
    y[:,t] = (Y[t])[id]
  end

  # Hyperparameters
  a_nu = b_nu = a_alpha = b_alpha = gamma0
  vbeta0 = vphi0 = v0

  # save samples
  num_save = Int((N-burnin)/thin)
  nu_save = Array(Float64, num_save)
  alpha_save = Array(Float64, num_save)
  c_save = Array(Int64, num_save, n, T)
  z_save = Array(Int64, num_save, n)
  beta_save = Array(Array{Float64,1}, num_save)
  phi_save = Array(Array{Float64,1}, num_save)
  it_save = 0
  srand(seed)                                 # set random seed

  # Initialization
  a = zeros(num, T)
  for i in 1:num
    for t in 1:T
      a[i,t] = sample_a(0.0, 1.0, y[i,t]);
    end
  end

  K = initcomp                           # no. of initial clusters in DP (communities)
  nu = 1.0                               # DP hyperparameter
  z = rand(Categorical(1/K*ones(K)), n)  # z: vector of length n indicating cluster membership of each subject
  z = indexin(z, unique(z))
  beta = randn(maximum(z))

  L = initcomp                              # no. of initial clusters in DP (popularity)
  alpha = 1.0                               # DP hyperparameter
  c = rand(Categorical(1/L*ones(L)), n, T)  # c: vector of length n indicating cluster membership of each subject
  c = indexin(c, unique(c))
  phi = randn(maximum(c))

  for it in 1:N

    # sample a
    for i in 1:num
      for t in 1:T
        mu_a_i = phi[c[ssr[i], t]] + phi[c[ssc[i], t]]
        if z[ssr[i]]==z[ssc[i]]
          mu_a_i += beta[z[ssr[i]]]
        end
        a[i, t] = sample_a(mu_a_i, 1.0, y[i, t])
      end
    end

    # sample z
    a1 = squeeze(sum(a - phi[c[ssr,:]] - phi[c[ssc,:]], 2), 2)
    for i in 1:n
      z, beta = sampleAlg2_z(i, a1, z, ssr, ssc, nu, beta, vbeta0, T)
    end
    K = maximum(z)
    Z = z[ssr]
    Z[z[ssr].!=z[ssc]]=0

    # sample beta
    P = T*counts(Z,1:K) + 1/vbeta0
    S = Array(Float64, K)
    for k in 1:K
      S[k] = sum(a1[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # sample nu
    nu = sample_DPprecision(n, K, nu, a_nu, b_nu)

    # Sample c
    betaZ = zeros(num)
    betaZ[Z.!=0] = beta[Z[Z.!=0]]
    a1 = broadcast(-, a, betaZ)
    for i in 1:n
      for t in 1:T
        c, phi = sample_c_dyn(i, t, a1, ssr, ssc, c, phi, alpha, vphi0)
      end
    end

    # sample phi
    phi = sample_phi_dyn(c, ssr, ssc, phi, a1, vphi0)

    # sample alpha
    alpha = sample_DPprecision(n, maximum(c), alpha, a_alpha, b_alpha)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      nu_save[it_save] = copy(nu)
      alpha_save[it_save] = copy(alpha)
      z_save[it_save, :] = copy(z)
      c_save[it_save, :, :] = copy(c)
      beta_save[it_save] = copy(beta)
      phi_save[it_save] = copy(phi)
    end

    if mod(it,1000)==0           # print everytime 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  output = Dict("nu_save" => nu_save, "alpha_save" => alpha_save,
  "z_save" => z_save, "c_save" => c_save, "beta_save" => beta_save,
  "phi_save" => phi_save)
  return output
end

function DynI_fix(Y::Array{Array{Int64,2},1}, N::Int64, burnin::Int64,
  thin::Int64, gamma0::Float64, v0::Float64, seed::Int64, c::Array{Int64,2},
  z::Array{Int64,1}, phi::Array{Float64,1}, beta::Array{Float64,1})
  # assume y is symmetric (undirected) adjacency matrix

  T = length(Y)                                # no. of time points
  n = size(Y[1], 1)                            # no. of subjects
  id = find(triu(ones(n, n), 1).== 1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n, n), 1).== 1) # row indices and column indices
  num = length(id)                             # no. of non-diagonal elements
  y = Array(Int64, num, T)                     # y: matrix contain upper-triangle only
  for t = 1:T
    y[:,t] = (Y[t])[id]
  end

  # Hyperparameters
  a_nu = b_nu = a_alpha = b_alpha = gamma0
  vbeta0 = vphi0 = v0

  # save samples
  num_save = Int((N-burnin)/thin)
  beta_save = Array(Float64, num_save, maximum(clc))
  phi_save = Array(Float64, num_save, maximum(clp))
  it_save = 0
  srand(seed)                                 # set random seed

  # Initialization
  a = zeros(num, T)
  for i in 1:num
    for t in 1:T
      a[i,t] = sample_a(0.0, 1.0, y[i,t]);
    end
  end
  K = maximum(z)
  Z = z[ssr]
  Z[z[ssr].!=z[ssc]]=0
  L = maximum(c)

  for it in 1:N

    # sample a
    for i in 1:num
      for t in 1:T
        mu_a_i = phi[c[ssr[i], t]] + phi[c[ssc[i], t]]
        if z[ssr[i]]==z[ssc[i]]
          mu_a_i += beta[z[ssr[i]]]
        end
        a[i, t] = sample_a(mu_a_i, 1.0, y[i, t])
      end
    end

    # sample beta
    a1 = squeeze(sum(a - phi[c[ssr,:]] - phi[c[ssc,:]], 2), 2)
    P = T*counts(Z,1:K) + 1/vbeta0
    S = Array(Float64, K)
    for k in 1:K
      S[k] = sum(a1[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # Sample c
    betaZ = zeros(num)
    betaZ[Z.!=0] = beta[Z[Z.!=0]]
    a1 = broadcast(-, a, betaZ)

    # sample phi
    phi = sample_phi_dyn(c, ssr, ssc, phi, a1, vphi0)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      beta_save[it_save,:] = copy(beta)
      phi_save[it_save,:] = copy(phi)
    end

    if mod(it,1000)==0           # print everytime 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  return beta_save, phi_save
end

function DynII(Y::Array{Array{Int64,2},1}, N::Int64, burnin::Int64,
  thin::Int64, gamma0::Float64, v0::Float64, initcomp::Int, seed::Int64)
    # assume y is symmetric (undirected) adjacency matrix

  T = length(Y)                                # no. of time points
  n = size(Y[1], 1)                            # no. of subjects
  id = find(triu(ones(n, n), 1).== 1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n, n), 1).== 1) # row indices and column indices
  num = length(id)                             # no. of non-diagonal elements
  y = Array(Int64, num, T)                     # y: matrix contain upper-triangle only
  for t = 1:T
    y[:,t] = (Y[t])[id]
  end

  # Hyperparameters
  a_nu = b_nu = a_alpha = b_alpha = gamma0
  vbeta0 = vphi0 = v0
  veta0 = 1.0

  # save samples
  num_save = Int((N-burnin)/thin)
  nu_save = Array(Float64, num_save)
  eta_save = Array(Float64, num_save)
  alpha_save = Array(Float64, num_save)
  c_save = Array(Int64, num_save, n)
  z_save = Array(Int64, num_save, n)
  beta_save = Array(Array{Float64,1}, num_save)
  phi_save = Array(Array{Float64,1}, num_save)
  it_save = 0
  srand(seed)                                 # set random seed

  # Initialization
  a = zeros(num, T)
  for i in 1:num
    for t in 1:T
      a[i,t] = sample_a(0.0, 1.0, y[i,t]);
    end
  end

  K = initcomp               # no. of initial clusters in DP (communities)
  nu = 1.0             # DP hyperparameter
  z = rand(Categorical(1/K*ones(K)), n)  # z: vector of length n indicating cluster membership of each subject
  z = indexin(z, unique(z))
  beta = randn(maximum(z))

  L = initcomp                # no. of initial clusters in DP (popularity)
  alpha = 1.0          # DP hyperparameter
  c = rand(Categorical(1/L*ones(L)), n)  # c: vector of length n indicating cluster membership of each subject
  c = indexin(c, unique(c))
  phi = randn(maximum(c))
  eta = 0

  for it in 1:N

    # sample a
    for i in 1:num
      for t in 1:T
        mu_a_i = phi[c[ssr[i]]] + phi[c[ssc[i]]]
        if z[ssr[i]]==z[ssc[i]]
          mu_a_i += beta[z[ssr[i]]]
        end
        if t>1
          mu_a_i += eta*y[i, (t-1)]
        end
        a[i, t] = sample_a(mu_a_i, 1.0, y[i, t])
      end
    end

    # sample z
    a1 = squeeze( sum(a - cat(2,zeros(num),eta*y[:,1:(T-1)]), 2), 2)
    a1 = a1 - T*phi[c[ssr]] - T*phi[c[ssc]]
    for i in 1:n
      z, beta = sampleAlg2_z(i, a1, z, ssr, ssc, nu, beta, vbeta0, T)
    end
    K = maximum(z)
    Z = z[ssr]
    Z[z[ssr].!=z[ssc]]=0

    # sample beta
    P = T*counts(Z,1:K) + 1/vbeta0
    S = Array(Float64,K)
    for k in 1:K
      S[k] = sum(a1[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # sample nu
    nu = sample_DPprecision(n, K, nu, a_nu, b_nu)

    # Sample c
    betaZ = zeros(num)
    betaZ[Z.!=0] = beta[Z[Z.!=0]]
    a1 = squeeze( sum(a - cat(2,zeros(num),eta*y[:,1:(T-1)]), 2), 2 )
    a1 -= T*betaZ
    for i in 1:n
      c, phi = sample_c(i, a1, ssr, ssc, c, phi, alpha, vphi0, T)
    end

    # sample phi
    phi = sample_phi(c, ssr, ssc, phi, a1, vphi0, T)

    # sample alpha
    alpha = sample_DPprecision(n, maximum(c), alpha, a_alpha, b_alpha)

    # sample eta
    eta = sample_eta(y, a, ssr, ssc, c, phi, Z, beta, veta0)


    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      nu_save[it_save] = copy(nu)
      eta_save[it_save] = copy(eta)
      alpha_save[it_save] = copy(alpha)
      z_save[it_save, :] = copy(z)
      c_save[it_save, :] = copy(c)
      beta_save[it_save] = copy(beta)
      phi_save[it_save] = copy(phi)
    end

    if mod(it,1000)==0           # print everytime 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  output = Dict("nu_save" => nu_save, "eta_save" => eta_save, "alpha_save" => alpha_save,
  "z_save" => z_save, "c_save" => c_save, "beta_save" => beta_save, "phi_save" => phi_save)
  return output
end

function DynII_fix(Y::Array{Array{Int64,2},1}, N::Int64, burnin::Int64,
  thin::Int64, gamma0::Float64, v0::Float64, seed::Int64, c::Array{Int64,1},
  z::Array{Int64,1}, phi::Array{Float64,1}, beta::Array{Float64,1}, eta::Float64)

  T = length(Y)                                # no. of time points
  n = size(Y[1], 1)                            # no. of subjects
  id = find(triu(ones(n, n), 1).== 1)          # indices of upper-triangle elements
  (ssr, ssc) = findn(triu(ones(n, n), 1).== 1) # row indices and column indices
  num = length(id)                             # no. of non-diagonal elements
  y = Array(Int64, num, T)                     # y: matrix contain upper-triangle only
  for t = 1:T
    y[:,t] = (Y[t])[id]
  end

  # Hyperparameters
  a_nu = b_nu = a_alpha = b_alpha = gamma0
  vbeta0 = vphi0 = v0
  veta0 = 1.0

  # save samples
  num_save = Int((N-burnin)/thin)
  eta_save = Array(Float64, num_save)
  beta_save = Array(Float64, num_save, maximum(clc))
  phi_save = Array(Float64, num_save, maximum(clp))
  it_save = 0
  srand(seed)                                 # set random seed

  # Initialization
  a = zeros(num, T)
  for i in 1:num
    for t in 1:T
      a[i,t] = sample_a(0.0, 1.0, y[i,t]);
    end
  end

  K = maximum(clc)
  Z = z[ssr]
  Z[z[ssr].!=z[ssc]]=0
  L = maximum(clp)

  for it in 1:N

    # sample a
    for i in 1:num
      for t in 1:T
        mu_a_i = phi[c[ssr[i]]] + phi[c[ssc[i]]]
        if z[ssr[i]]==z[ssc[i]]
          mu_a_i += beta[z[ssr[i]]]
        end
        if t>1
          mu_a_i += eta*y[i, (t-1)]
        end
        a[i, t] = sample_a(mu_a_i, 1.0, y[i, t])
      end
    end

    # sample beta
    a1 = squeeze( sum(a - cat(2,zeros(num),eta*y[:,1:(T-1)]), 2), 2)
    a1 = a1 - T*phi[c[ssr]] - T*phi[c[ssc]]
    P = T*counts(Z,1:K) + 1/vbeta0
    S = Array(Float64,K)
    for k in 1:K
      S[k] = sum(a1[Z.==k])
    end
    beta = rand(MvNormal(S./P, 1./sqrt(P)))

    # sample phi
    betaZ = zeros(num)
    betaZ[Z.!=0] = beta[Z[Z.!=0]]
    a1 = squeeze( sum(a - cat(2,zeros(num),eta*y[:,1:(T-1)]), 2), 2 )
    a1 -= T*betaZ
    phi = sample_phi(c, ssr, ssc, phi, a1, vphi0, T)

    # sample eta
    eta = sample_eta(y, a, ssr, ssc, c, phi, Z, beta, veta0)

    # Save parameters
    if (it > burnin && mod(it,thin)==0)
      it_save += 1
      eta_save[it_save] = copy(eta)
      beta_save[it_save,:] = copy(beta)
      phi_save[it_save,:] = copy(phi)
    end

    if mod(it,1000)==0           # print everytime 1000 iterations completed
      println("seed ", seed, ": completed ", it, " iterations")
    end
  end
  return eta_save, beta_save, phi_save
end
