
library(MCMCpack)

## Read the data in
dat <- read.csv(file="~/StatModelingPaper/dentpar.csv")

## Keep only the complete cases
dat <- dat[complete.cases(dat),]


## Variables in the dataset include:

## T - Binary treatment indicator (1 if low SES, 0 if high)
## M1 - Continuous mediator 1 (dental visit frequency per year)
## M2 - Continuous mediator 2 (oral hygience index OHI)
## Y - Continuous outcome (number of decayed, missing or filled teeth)
## X1, X2 and X3 - list of confounders: normal birth weight, sex and race  
## ID - Subject ID, needed for joint prediction of M1(0) and M1(1). If not available, can add it as row number


## First, fit regression models for mediators M1, M2 and log transformed outcome Y

model.m1 <- lm(M1 ~ T + X1 + X2 + X3, data = dat)
model.m2 <- lm(M2 ~ T + M1 + X1 + X2 + X3 + T*M1, data = dat)
model.y <- lm(log(Y+1) ~ T + M1 + M2 + X1 + X2 + X3 + T*M1 , data = dat)


## Bayesian model for mediator M1, the list of confounders includes X1, X2 and X3
## burnin is number of burnin iterations, mcmc is number of posterior draws from M1 model
## mubeta and Vbeta are prior mean and variance for fixed effects in M1 model
## r and R are prior parameters for covariance matrix of random effects in M1 model
## nu and delta are prior parameters for residual error variance in M1 model
bayes.m1 <- MCMChregress(fixed = M1 ~ T + X1 + X2 + X3 ,
                         random = ~1+T,
                         group="ID",
                         data=dat,
                         burnin=1000,  
                         mcmc=10000,
                         thin=1,
                         verbose=0, 
                         mubeta=0,
                         Vbeta=1000,
                         r=2,
                         R=matrix(c(1,0,0,1),2,2),
                         nu=0.001,
                         delta=0.001
                         )
summary(bayes.m1$mcmc)
plot(bayes.m1$mcmc)
raftery.diag(bayes.m1$mcmc)


## Bayesian model for mediator M2, the list of confounders includes X1, X2 and X3
## burnin is number of burnin iterations, mcmc is number of posterior draws from M2 model
## b0 and B0 are prior mean and precision for coefficients in M2 model
## c0 and d0 are prior parameters for residual error variance in M2 model
bayes.m2 <- MCMCregress(M2 ~ T + M1 + X1 + X2 + X3 + T*M1, 
                        data=dat,
                        burnin=1000,  
                        mcmc=10000,
                        thin=1,
                        verbose=0, 
                        b0 = 0, 
                        B0 = 0.001, 
                        c0 = 0.001, 
                        d0 = 0.001
                        )
summary(bayes.m2)
plot(bayes.m2)
raftery.diag(bayes.m2)

## Bayesian model for log transformed outcome Y, the list of confounders includes X1, X2 and X3
## burnin is number of burnin iterations, mcmc is number of posterior draws from Y model
## b0 and B0 are prior mean and precision for coefficients in Y model
## c0 and d0 are prior parameters for residual error variance in Y model
bayes.y <- MCMCregress(log(Y+1) ~ T + M1 + M2 + X1 + X2 + X3 + T*M1 , 
                       data=dat,
                       burnin=1000,
                       mcmc=10000,
                       thin=1,
                       verbose=0,
                       b0=0, 
                       B0=0.001,
                       c0=0.001,
                       d0=0.001
                       )
summary(bayes.y)
plot(bayes.y)
raftery.diag(bayes.y)



## Function to calculate potential outcomes based on above Bayesian model fit, where
## model.m1, model.m2, model.y are regression model fits for M1, M2 and Y,
## and bayes.m1, bayes.m2, bayes.y are Bayesian model fits for M1, M2 and Y
## sims is the number of potential mediators and outcome (for each subject). It is the same as the number of posterior draws from M1, M2 and Y model. 
## treat.name is name of treatment variable in the dataset
## med.name is the name of mediator 1 M1 in the dataset
## med2.name is the name of mediator 2 M2 in the dataset
## conf.level is the confidence level of Bayesian credible interval

bayes.ordered <- function(model.m1, model.m2, model.y, bayes.m1, bayes.m2, bayes.y, sims, 
                          treat = "treat.name", mediator = "med.name", mediator2 ="med2.name",
                          conf.level = .95){
  
  
  m1.data <- cbind(1,model.matrix(model.y)[,c(attr(model.m1$terms, "term.labels"))])
  m2.data <- cbind(1,model.matrix(model.y)[,c(attr(model.m2$terms, "term.labels"))])
  y.data  <- model.matrix(model.y)
  
  
  # Numbers of observations in the data
  n <- nrow(m1.data)
  
  # Number of predictors in M1 model (excluding intercept)
  n.m1 <- ncol(m1.data)-1
 
  
  ######################################################
  ##  M1 Predictions
  ##  M1(0) and M1(1) are correlated
  ##  So they are predicted together 
  ##  and will be used to predict M2 and Y together
  ######################################################
  m1mat.c <- m1mat.t <- m1.data
  
  m1mat.c[,treat] <- 0
  m1mat.t[,treat] <- 1
 
  m1.fixed.0 <- bayes.m1$mcmc[,1:(n.m1+1)] %*% t(m1mat.c)
  m1.fixed.1 <- bayes.m1$mcmc[,1:(n.m1+1)] %*% t(m1mat.t)
  
  error <- matrix(rnorm(n*sims,0,sd=sqrt(bayes.m1$mcmc[,c(2*n+n.m1+6)])),nrow=sims)
  
  # M1(0) prediction
  PredictM1.0 <- m1.fixed.0 + as.matrix(bayes.m1$mcmc[,(n.m1+2):(n.m1+n+1)]) + error
  
  # M1(1) prediction
  PredictM1.1 <- m1.fixed.1 + as.matrix(bayes.m1$mcmc[,(n.m1+2):(n.m1+n+1)]) + as.matrix(bayes.m1$mcmc[,(n.m1+n+2):(n.m1+2*n+1)]) + error
  

  
  #####################################
  ##  Outcome Y Predictions
  #####################################
  
  
  ## Y(t0, M1(t1), M2(t2,M1(t3)))
  ## t0, t1, t2, t3 can each be 0 or 1
  ## So total combinations of all t's for the Y potential outcomes is 2*2*2*2=16
  
  Y.potential <- array(NA, dim = c(n, sims, 2, 2, 2, 2))
  ## in the array, 
  ## first 2 is for t0
  ## second 2 is for t1
  ## third 2 is for t2
  ## fourth 2 is for t3
  
  
  for (t0 in 0:1){
    for (t1 in 0:1){
      for (t2 in 0:1){
        for (t3 in 0:1){
          ## 	Y.potential[,,(t0+1),(t1+1),(t2+1),(t3+1)] is used to save potential outcomes of Y
          for (j in 1:sims){
            ## set M1(t3) values
            PredictM1.t3 <- PredictM1.1[j,]*t3 + PredictM1.0[j,]*(1-t3)
            
            ## using M1(t3) and t2 to predict M2(t2, M1(t3))
            ## From mcmc of M2 model parameter posteriors and the design matrix m2.data
            ## where m2.data's t2 column is replaced with t2, and M1 column is replaced with M1(t3)--PredictM1.t3
            ## also interaction term T*M1 is replaced with t2*M1(t3)
            pred.data.m1 <- m2.data
            pred.data.m1[,treat] <- t2
            pred.data.m1[,mediator] <- PredictM1.t3
            pred.data.m1[,paste(treat,":",mediator,sep='')] <- t2*PredictM1.t3
      
            muM2 <- bayes.m2[j,1:ncol(pred.data.m1)] %*% t(pred.data.m1)
            error.2 <- rnorm(n,0,sd=sqrt(bayes.m2[j,(1+ncol(pred.data.m1))]))
            PredictM2 <- muM2 + error.2
            
            ## using t0, M1(t1), and the predicted M2(t2, M1(t3)) to predict Y
            ## From mcmc of Y model parameter posteriors and the design matrix y.data
            ## where y.data's T column is replaced with t0, and M1 is replaced by M1(t1), M2 is replaced with Predicted M2(t2, M1(t3))
            ## and interaction T*M1 is replaced with t0*M1(t1)
            
            ## first, set M1(t1) value
            PredictM1.t1 <- PredictM1.1[j,]*t1 + PredictM1.0[j,]*(1-t1)
            ## then get design matrix of y.data, replace treat, mediator and mediator2 with t0, M1(t1), and M2(t2,M1(t3)), interaction of T*M1 with t0*M1(t1)
            pred.data.m2 <- y.data
            pred.data.m2[,treat] <- t0
            pred.data.m2[,mediator] <- PredictM1.t1
            pred.data.m2[,mediator2] <- t(PredictM2)
            pred.data.m2[,paste(treat,":",mediator,sep='')] <- t0*PredictM1.t1
            
            
            ## From mcmc posterior of Y model parameter, and above updated design matrix, predict the Y potential outcome
            muY <- bayes.y[j,1:ncol(pred.data.m2)] %*% t(pred.data.m2)
            error.y <- rnorm(n,0,sd=sqrt(bayes.y[j,(1+ncol(pred.data.m2))]))
            PredictY <- muY + error.y 
            
            Y.potential[,j,(t0+1),(t1+1),(t2+1),(t3+1)] <- PredictY
          }
        }
      }
    }
  }		

  ## Potential outcomes averaged over the population -- which is number of patients --n, the first dimension of the array
  Y.potential.estimate <- apply(Y.potential,c(2:6),mean)


  ## direct effect: t0=1 vs. t0=0, with t1,t2,t3 fixed
  d <- array(NA,c(sims,8))
  
  i <- 0
  for (t1 in 0:1){
    for (t2 in 0:1){
      for (t3 in 0:1){
        i <- i + 1
        d[,i] <- Y.potential.estimate[,2,(t1+1),(t2+1),(t3+1)]-Y.potential.estimate[,1,(t1+1),(t2+1),(t3+1)]
      }
    }
  }
  
  colnames(d)<-c("000","001","010","011","100","101","110","111")
  
  
  ## indirect effect through M1 only: t1=1 vs. t1=0, with t0,t2,t3 fixed
  m1 <- array(NA,c(sims,8))
  
  i <- 0
  for (t0 in 0:1){
    for (t2 in 0:1){
      for (t3 in 0:1){
        i <- i + 1
        m1[,i] <- Y.potential.estimate[,(t0+1),2,(t2+1),(t3+1)]-Y.potential.estimate[,(t0+1),1,(t2+1),(t3+1)]
      }
    }
  }
  
  colnames(m1)<-c("000","001","010","011","100","101","110","111")
  

  ## indirect effect through M2 only: t2=1 vs. t2=0, with t0,t1,t3 fixed
  m2 <- array(NA,c(sims,8))
  
  i <- 0
  for (t0 in 0:1){
    for (t1 in 0:1){
      for (t3 in 0:1){
        i <- i + 1
        m2[,i] <- Y.potential.estimate[,(t0+1),(t1+1),2,(t3+1)]-Y.potential.estimate[,(t0+1),(t1+1),1,(t3+1)]
      }
    }
  }
  
  colnames(m2)<-c("000","001","010","011","100","101","110","111")
  

  ## indirect effect through both M1 and M2: t3=1 vs. t3=0, with t0,t1,t2 fixed
  m12 <- array(NA,c(sims,8))
  
  i <- 0
  for (t0 in 0:1){
    for (t1 in 0:1){
      for (t2 in 0:1){
        i <- i + 1
        m12[,i] <- Y.potential.estimate[,(t0+1),(t1+1),(t2+1),2]-Y.potential.estimate[,(t0+1),(t1+1),(t2+1),1]
      }
    }
  }
  
  colnames(m12)<-c("000","001","010","011","100","101","110","111")
  

  ## Total causal effect: t0=t1=t2=t3=1 vs. t0=t1=t2=t3=0
  t <- Y.potential.estimate[,2,2,2,2]-Y.potential.estimate[,1,1,1,1]
  

  ###########################################################################################
  ## Summary of all direct and indirect effects -- posterior mean and credible interval(CI)##
  ###########################################################################################
  
  low <- (1 - conf.level)/2
  high <- 1 - low
  
  t.ci <- quantile(t,probs=c(low,high))
  d.ci <- apply(d,2,function(x) quantile(x,probs=c(low,high)))
  m1.ci <- apply(m1,2,function(x) quantile(x,probs=c(low,high)))
  m2.ci <- apply(m2,2,function(x) quantile(x,probs=c(low,high)))
  m12.ci <- apply(m12,2,function(x) quantile(x,probs=c(low,high)))
  
  t.mean <- mean(t)
  d.mean <- apply(d,2,mean)
  m1.mean <- apply(m1,2,mean)
  m2.mean <- apply(m2,2,mean)
  m12.mean <- apply(m12,2,mean)
  
  t.var <- var(t)
  d.var <- apply(d,2,var)
  m1.var <- apply(m1,2,var)
  m2.var <- apply(m2,2,var)
  m12.var <- apply(m12,2,var)
  
  
  out <- list(t.mean=t.mean, d.mean=d.mean, m1.mean=m1.mean, m2.mean=m2.mean, m12.mean=m12.mean,
              t.ci=t.ci, d.ci=d.ci, m1.ci=m1.ci, m2.ci=m2.ci, m12.ci=m12.ci,
              t.var=t.var, d.var=d.var, m1.var=m1.var, m2.var=m2.var, m12.var=m12.var,
              treat=treat, mediator=mediator, mediator2=mediator2, 
              conf.level=conf.level,
              nobs=n, sims=sims)
  
  return(out)

}






result <- bayes.ordered(model.m1, model.m2, model.y,
                       bayes.m1, bayes.m2, bayes.y, 
                       sims = 10000, 
                       treat = "T", mediator = "M1", mediator2 = "M2",
                       conf.level = 0.95)

result

## The result of Bayesian analysis include:
## t.mean, d.mean, m1.mean, m2.mean, m12.mean -- posterior mean for total causal effect, direct effect, indirect effect through M1 alone, through M2 alone, and through both M1 and M2
## t.ci, d.ci, m1.ci, m2.ci, m12.ci -- Bayesian credible interval for total causal effect, direct effect, indirect effect through M1 alone, through M2 alone, and through both M1 and M2
## t.var, d.var, m1.var, m2.var, m12.var -- posterior variance for total causal effect, direct effect, indirect effect through M1 alone, through M2 alone, and through both M1 and M2
## nobs -- number of observations (subjects) in the dataset
## sims -- number of potential mediators and outcome for each subject
