################################################################################################
######  Defining responses and covariates from credit card data ################################
################################################################################################

library(AER)
data("CreditCard")
zeros <- which(CreditCard$reports == 0)
nonzero <- which(CreditCard$reports != 0)

n.cc <- length(CreditCard$reports)
reports.star <- CreditCard$reports
reports.star[nonzero] <- log(reports.star[nonzero] - runif(length(nonzero)))
save(reports.star, file = "CChurdle_reports.RData")

intercept.cc <- rep(1, n.cc)
share <- CreditCard$share
owner <- as.numeric(CreditCard$owner) - 1
accounts <- CreditCard$active

################################################################################################
######  Running R2OpenBUGS for Quantile Regression  ############################################
################################################################################################

library(R2OpenBUGS)
library(coda)

CC.data <- list("reports.star", "n.cc", "tau", "intercept.cc", "share", "owner", "accounts")

# initial values chosen from prior run at the median
CC.initials <- list(
  list(beta = c(0.1, -3, -0.5, 0.03), gamma = c(-1.7, -5, -0.5, 0.1), delta = 2.1,
       w = rep(0.5, n.cc)),
  list(beta = c(-0.2, -5, -1, 0), gamma = c(-2, -7.5, -1, 0), delta = 1.5, w = rep(1, n.cc)),
  list(beta = c(0.5, -1, 0, 0.1), gamma = c(-1, -2.5, 0, 0.2), delta = 3, w = rep(2, n.cc)))

CC.keeps <- list("beta", "gamma", "delta", "pi", "w")

###### 5th percentile ######

tau <- 0.05

CC.05 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
                 n.iter = 7500, n.chains = 3, n.thin = 100, n.burnin = 500,
                 parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.05, file = "CC05.RData")
rm(CC.05)

###### 10th percentile ######

tau <- 0.1

CC.10 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
                 n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
                 parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.10, file = "CC10.RData")
rm(CC.10)

###### 15th percentile ######

tau <- 0.15

CC.15 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.15, file = "CC15.RData")
rm(CC.15)

###### 20th percentile ######

tau <- 0.2

CC.20 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.20, file = "CC20.RData")
rm(CC.20)

###### 25th percentile ######

tau <- 0.25

CC.25 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.25, file = "CC25.RData")
rm(CC.25)

###### 30th percentile ######

tau <- 0.3

CC.30 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.30, file = "CC30.RData")
rm(CC.30)

###### 35th percentile ######

tau <- 0.35

CC.35 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.35, file = "CC35.RData")
rm(CC.35)

###### 40th percentile ######

tau <- 0.4

CC.40 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.40, file = "CC40.RData")
rm(CC.40)

###### 45th percentile ######

tau <- 0.45

CC.45 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 40, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.45, file = "CC45.RData")
rm(CC.45)

###### 50th percentile ######

tau <- 0.5

CC.50 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 40, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.50, file = "CC50.RData")
rm(CC.50)

###### 55th percentile ######

tau <- 0.55

CC.55 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 40, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.55, file = "CC55.RData")
rm(CC.55)

###### 60th percentile ######

tau <- 0.6

CC.60 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.60, file = "CC60.RData")
rm(CC.60)

###### 65th percentile ######

tau <- 0.65

CC.65 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.65, file = "CC65.RData")
rm(CC.65)

###### 70th percentile ######

tau <- 0.7

CC.70 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.70, file = "CC70.RData")
rm(CC.70)

###### 75th percentile ######

tau <- 0.75

CC.75 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.75, file = "CC75.RData")
rm(CC.75)

###### 80th percentile ######

tau <- 0.8

CC.80 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.80, file = "CC80.RData")
rm(CC.80)

###### 85th percentile ######

tau <- 0.85

CC.85 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.85, file = "CC85.RData")
rm(CC.85)

###### 90th percentile ######

tau <- 0.9

CC.90 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.90, file = "CC90.RData")
rm(CC.90)

###### 95th percentile ######

tau <- 0.95

CC.95 <- bugs(model.file = "CChurdle.txt", data = CC.data, inits = CC.initials,
              n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
              parameters.to.save = CC.keeps, DIC = FALSE)
save(CC.95, file = "CC95.RData")
rm(CC.95)

################################################################################################
######  Mean models for credit card data  ######################################################
################################################################################################

reports <- CreditCard$reports

library(R2jags)

CC.meandata <- list("reports", "n.cc", "intercept.cc", "share", "owner", "accounts")

CC.meaninitials <- list(
  list(beta = c(0.1, -3, -0.5, 0.03), gamma = c(-1.7, -5, -0.5, 0.1)),
  list(beta = c(-0.2, -5, -1, 0), gamma = c(-2, -7.5, -1, 0)),
  list(beta = c(0.5, -1, 0, 0.1), gamma = c(-1, -2.5, 0, 0.2)))

CC.meankeeps <- c("beta", "gamma")

CC.meaninitialsNH <- list(
  list(beta = c(0.1, -3, -0.5, 0.03)),
  list(beta = c(-0.2, -5, -1, 0)),
  list(beta = c(0.5, -1, 0, 0.1)))

CC.meankeepsNH <- c("beta")

PhurdleCC <- jags(model.file = "hurdlePmeanCC.txt", data = CC.meandata,
                  inits = CC.meaninitials, n.iter = 25500, n.chains = 3,
                  n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeeps)

NBhurdleCC <- jags(model.file = "hurdleNBmeanCC.txt", data = CC.meandata,
                   inits = CC.meaninitials, n.iter = 25500, n.chains = 3,
                   n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeeps)

PnhCC <- jags(model.file = "PmeanCC.txt", data = CC.meandata,
              inits = CC.meaninitialsNH, n.iter = 25500, n.chains = 3,
              n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeepsNH)

NBnhCC <- jags(model.file = "NBmeanCC.txt", data = CC.meandata,
               inits = CC.meaninitialsNH, n.iter = 25500, n.chains = 3,
               n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeepsNH)

save(PhurdleCC, file = "Phurdlecc.RData")
save(NBhurdleCC, file = "NBhurdlecc.RData")
save(PnhCC, file = "Pnhcc.RData")
save(NBnhCC, file = "NBnhcc.RData")

################################################################################################
######  Predictions for hypothetical people  ###################################################
################################################################################################

person1 <- c(1, mean(share), 0, mean(accounts))
person2 <- c(1, quantile(share, 0.9), 1, quantile(accounts, 0.9))
person3 <- c(1, quantile(share, 0.25), 1, quantile(accounts, 0.25))

load("CC05.RData")
load("CC10.RData")
load("CC15.RData")
load("CC20.RData")
load("CC25.RData")
load("CC30.RData")
load("CC35.RData")
load("CC40.RData")
load("CC45.RData")
load("CC50.RData")
load("CC55.RData")
load("CC60.RData")
load("CC65.RData")
load("CC70.RData")
load("CC75.RData")
load("CC80.RData")
load("CC85.RData")
load("CC90.RData")
load("CC95.RData")

p1quantiles <- vector("numeric", length = 19)
p1quantiles[1] <- sum(CC.05$summary[1:4, 1] * person1)
p1quantiles[2] <- sum(CC.10$summary[1:4, 1] * person1)
p1quantiles[3] <- sum(CC.15$summary[1:4, 1] * person1)
p1quantiles[4] <- sum(CC.20$summary[1:4, 1] * person1)
p1quantiles[5] <- sum(CC.25$summary[1:4, 1] * person1)
p1quantiles[6] <- sum(CC.30$summary[1:4, 1] * person1)
p1quantiles[7] <- sum(CC.35$summary[1:4, 1] * person1)
p1quantiles[8] <- sum(CC.40$summary[1:4, 1] * person1)
p1quantiles[9] <- sum(CC.45$summary[1:4, 1] * person1)
p1quantiles[10] <- sum(CC.50$summary[1:4, 1] * person1)
p1quantiles[11] <- sum(CC.55$summary[1:4, 1] * person1)
p1quantiles[12] <- sum(CC.60$summary[1:4, 1] * person1)
p1quantiles[13] <- sum(CC.65$summary[1:4, 1] * person1)
p1quantiles[14] <- sum(CC.70$summary[1:4, 1] * person1)
p1quantiles[15] <- sum(CC.75$summary[1:4, 1] * person1)
p1quantiles[16] <- sum(CC.80$summary[1:4, 1] * person1)
p1quantiles[17] <- sum(CC.85$summary[1:4, 1] * person1)
p1quantiles[18] <- sum(CC.90$summary[1:4, 1] * person1)
p1quantiles[19] <- sum(CC.95$summary[1:4, 1] * person1)

p2quantiles <- vector("numeric", length = 19)
p2quantiles[1] <- sum(CC.05$summary[1:4, 1] * person2)
p2quantiles[2] <- sum(CC.10$summary[1:4, 1] * person2)
p2quantiles[3] <- sum(CC.15$summary[1:4, 1] * person2)
p2quantiles[4] <- sum(CC.20$summary[1:4, 1] * person2)
p2quantiles[5] <- sum(CC.25$summary[1:4, 1] * person2)
p2quantiles[6] <- sum(CC.30$summary[1:4, 1] * person2)
p2quantiles[7] <- sum(CC.35$summary[1:4, 1] * person2)
p2quantiles[8] <- sum(CC.40$summary[1:4, 1] * person2)
p2quantiles[9] <- sum(CC.45$summary[1:4, 1] * person2)
p2quantiles[10] <- sum(CC.50$summary[1:4, 1] * person2)
p2quantiles[11] <- sum(CC.55$summary[1:4, 1] * person2)
p2quantiles[12] <- sum(CC.60$summary[1:4, 1] * person2)
p2quantiles[13] <- sum(CC.65$summary[1:4, 1] * person2)
p2quantiles[14] <- sum(CC.70$summary[1:4, 1] * person2)
p2quantiles[15] <- sum(CC.75$summary[1:4, 1] * person2)
p2quantiles[16] <- sum(CC.80$summary[1:4, 1] * person2)
p2quantiles[17] <- sum(CC.85$summary[1:4, 1] * person2)
p2quantiles[18] <- sum(CC.90$summary[1:4, 1] * person2)
p2quantiles[19] <- sum(CC.95$summary[1:4, 1] * person2)

p3quantiles <- vector("numeric", length = 19)
p3quantiles[1] <- sum(CC.05$summary[1:4, 1] * person3)
p3quantiles[2] <- sum(CC.10$summary[1:4, 1] * person3)
p3quantiles[3] <- sum(CC.15$summary[1:4, 1] * person3)
p3quantiles[4] <- sum(CC.20$summary[1:4, 1] * person3)
p3quantiles[5] <- sum(CC.25$summary[1:4, 1] * person3)
p3quantiles[6] <- sum(CC.30$summary[1:4, 1] * person3)
p3quantiles[7] <- sum(CC.35$summary[1:4, 1] * person3)
p3quantiles[8] <- sum(CC.40$summary[1:4, 1] * person3)
p3quantiles[9] <- sum(CC.45$summary[1:4, 1] * person3)
p3quantiles[10] <- sum(CC.50$summary[1:4, 1] * person3)
p3quantiles[11] <- sum(CC.55$summary[1:4, 1] * person3)
p3quantiles[12] <- sum(CC.60$summary[1:4, 1] * person3)
p3quantiles[13] <- sum(CC.65$summary[1:4, 1] * person3)
p3quantiles[14] <- sum(CC.70$summary[1:4, 1] * person3)
p3quantiles[15] <- sum(CC.75$summary[1:4, 1] * person3)
p3quantiles[16] <- sum(CC.80$summary[1:4, 1] * person3)
p3quantiles[17] <- sum(CC.85$summary[1:4, 1] * person3)
p3quantiles[18] <- sum(CC.90$summary[1:4, 1] * person3)
p3quantiles[19] <- sum(CC.95$summary[1:4, 1] * person3)

p1means <- c(sum(PhurdleCC$BUGSoutput$summary[1:4, 1]*person1),
             sum(NBhurdleCC$BUGSoutput$summary[1:4, 1]*person1),
             sum(PnhCC$BUGSoutput$summary[1:4, 1]*person1),
             sum(NBnhCC$BUGSoutput$summary[1:4, 1]*person1))
names(p1means) <- c("Phurd", "NBhurd", "Pnh", "NBnh")

p2means <- c(sum(PhurdleCC$BUGSoutput$summary[1:4, 1]*person2),
             sum(NBhurdleCC$BUGSoutput$summary[1:4, 1]*person2),
             sum(PnhCC$BUGSoutput$summary[1:4, 1]*person2),
             sum(NBnhCC$BUGSoutput$summary[1:4, 1]*person2))
names(p2means) <- c("Phurd", "NBhurd", "Pnh", "NBnh")

p3means <- c(sum(PhurdleCC$BUGSoutput$summary[1:4, 1]*person3),
             sum(NBhurdleCC$BUGSoutput$summary[1:4, 1]*person3),
             sum(PnhCC$BUGSoutput$summary[1:4, 1]*person3),
             sum(NBnhCC$BUGSoutput$summary[1:4, 1]*person3))
names(p3means) <- c("Phurd", "NBhurd", "Pnh", "NBnh")

################################################################################################
######  Defining responses and covariates from boat trip data ##################################
################################################################################################

data("RecreationDemand")
nonzero <- which(RecreationDemand$trips != 0)

n.rec <- length(RecreationDemand$trips)
trips.star <- RecreationDemand$trips
trips.star[nonzero] <- log(trips.star[nonzero] - runif(length(nonzero)))
save(trips.star, file = "rechurdle_trips.RData")

intercept.rec <- rep(1, n.rec)
ski <- as.numeric(RecreationDemand$ski) - 1
fee <- as.numeric(RecreationDemand$userfee) - 1
income <- RecreationDemand$income

################################################################################################
######  QR for boating data  ###################################################################
################################################################################################

rec.data <- list("trips.star", "n.rec", "tau", "intercept.rec", "ski", "fee", "income")

# initial values chosen from prior run at the median
rec.initials <- list(
  list(beta = c(1, 0.2, 1.5, 0), gamma = c(-0.7, 0.5, 28, 0), delta = 1.7,
       w = rep(0.6, n.rec)),
  list(beta = c(0.5, -0.2, 0.4, -0.2), gamma = c(-1.2, 0.1, 4, -0.1), delta = 1.5,
       w = rep(0.01, n.rec)),
  list(beta = c(1.5, 0.6, 2.5, 0.05), gamma = c(-0.3, 1, 75, 0.1), delta = 2,
       w = rep(2.2, n.rec)))

rec.keeps <- list("beta", "gamma", "delta", "pi", "w")

###### 5th percentile ######

tau <- 0.05

rec.05 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 100, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.05, file = "rec05.RData")
rm(rec.05)

###### 10th percentile ######

tau <- 0.1

rec.10 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.10, file = "rec10.RData")
rm(rec.10)

###### 15th percentile ######

tau <- 0.15

rec.15 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.15, file = "rec15.RData")
rm(rec.15)

###### 20th percentile ######

tau <- 0.2

rec.20 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.20, file = "rec20.RData")
rm(rec.20)

###### 25th percentile ######

tau <- 0.25

rec.25 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.25, file = "rec25.RData")
rm(rec.25)

###### 30th percentile ######

tau <- 0.3

rec.30 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.30, file = "rec30.RData")
rm(rec.30)

###### 35th percentile ######

tau <- 0.35

rec.35 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.35, file = "rec35.RData")
rm(rec.35)

###### 40th percentile ######

tau <- 0.4

rec.40 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.40, file = "rec40.RData")
rm(rec.40)

###### 45th percentile ######

tau <- 0.45

rec.45 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 40, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.45, file = "rec45.RData")
rm(rec.45)

###### 50th percentile ######

tau <- 0.5

rec.50 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 40, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.50, file = "rec50.RData")
rm(rec.50)

###### 55th percentile ######

tau <- 0.55

rec.55 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 40, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.55, file = "rec55.RData")
rm(rec.55)

###### 60th percentile ######

tau <- 0.6

rec.60 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.60, file = "rec60.RData")
rm(rec.60)

###### 65th percentile ######

tau <- 0.65

rec.65 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.65, file = "rec65.RData")
rm(rec.65)

###### 70th percentile ######

tau <- 0.7

rec.70 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.70, file = "rec70.RData")
rm(rec.70)

###### 75th percentile ######

tau <- 0.75

rec.75 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 50, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.75, file = "rec75.RData")
rm(rec.75)

###### 80th percentile ######

tau <- 0.8

rec.80 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.80, file = "rec80.RData")
rm(rec.80)

###### 85th percentile ######

tau <- 0.85

rec.85 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 60, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.85, file = "rec85.RData")
rm(rec.85)

###### 90th percentile ######

tau <- 0.9

rec.90 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.90, file = "rec90.RData")
rm(rec.90)

###### 95th percentile ######

tau <- 0.95

rec.95 <- bugs(model.file = "rechurdle.txt", data = rec.data, inits = rec.initials,
               n.iter = 7500, n.chains = 3, n.thin = 80, n.burnin = 500,
               parameters.to.save = rec.keeps, DIC = FALSE)
save(rec.95, file = "rec95.RData")
rm(rec.95)

################################################################################################
######  Mean models for boating data  ##########################################################
################################################################################################

trips <- RecreationDemand$trips

rec.meandata <- list("trips", "n.rec", "intercept.rec", "ski", "fee", "income")

# initial values chosen from prior run at the median
rec.meaninitials <- list(
  list(beta = c(1, 0.2, 1.5, 0), gamma = c(-0.7, 0.5, 28, 0)),
  list(beta = c(0.5, -0.2, 0.4, -0.2), gamma = c(-1.2, 0.1, 4, -0.1)),
  list(beta = c(1.5, 0.6, 2.5, 0.05), gamma = c(-0.3, 1, 75, 0.1)))

Phurdlerec <- jags(model.file = "hurdlePmeanrec.txt", data = rec.meandata,
                   inits = rec.meaninitials, n.iter = 25500, n.chains = 3,
                   n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeeps)

NBhurdlerec <- jags(model.file = "hurdleNBmeanrec.txt", data = rec.meandata,
                    inits = CC.meaninitials, n.iter = 25500, n.chains = 3,
                    n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeeps)

Pnhrec <- jags(model.file = "Pmeanrec.txt", data = rec.meandata,
               inits = CC.meaninitialsNH, n.iter = 25500, n.chains = 3,
               n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeepsNH)

NBnhrec <- jags(model.file = "NBmeanrec.txt", data = rec.meandata,
                inits = CC.meaninitialsNH, n.iter = 25500, n.chains = 3,
                n.thin = 25, n.burnin = 500, parameters.to.save = CC.meankeepsNH)

save(Phurdlerec, file = "Phurdlerec.RData")
save(NBhurdlerec, file = "NBhurdlerec.RData")
save(Pnhrec, file = "Pnhrec.RData")
save(NBnhrec, file = "NBnhrec.RData")

################################################################################################
######  Predictions for hypothetical people  ###################################################
################################################################################################

personA <- c(1, 0, 0, mean(income))
personB <- c(1, 1, 0, mean(income[ski == 1]))
personC <- c(1, 1, 1, quantile(income, 0.75))

load("rec05.RData")
load("rec10.RData")
load("rec15.RData")
load("rec20.RData")
load("rec25.RData")
load("rec30.RData")
load("rec35.RData")
load("rec40.RData")
load("rec45.RData")
load("rec50.RData")
load("rec55.RData")
load("rec60.RData")
load("rec65.RData")
load("rec70.RData")
load("rec75.RData")
load("rec80.RData")
load("rec85.RData")
load("rec90.RData")
load("rec95.RData")

pAquantiles <- vector("numeric", length = 19)
pAquantiles[1] <- sum(rec.05$summary[1:4, 1] * personA)
pAquantiles[2] <- sum(rec.10$summary[1:4, 1] * personA)
pAquantiles[3] <- sum(rec.15$summary[1:4, 1] * personA)
pAquantiles[4] <- sum(rec.20$summary[1:4, 1] * personA)
pAquantiles[5] <- sum(rec.25$summary[1:4, 1] * personA)
pAquantiles[6] <- sum(rec.30$summary[1:4, 1] * personA)
pAquantiles[7] <- sum(rec.35$summary[1:4, 1] * personA)
pAquantiles[8] <- sum(rec.40$summary[1:4, 1] * personA)
pAquantiles[9] <- sum(rec.45$summary[1:4, 1] * personA)
pAquantiles[10] <- sum(rec.50$summary[1:4, 1] * personA)
pAquantiles[11] <- sum(rec.55$summary[1:4, 1] * personA)
pAquantiles[12] <- sum(rec.60$summary[1:4, 1] * personA)
pAquantiles[13] <- sum(rec.65$summary[1:4, 1] * personA)
pAquantiles[14] <- sum(rec.70$summary[1:4, 1] * personA)
pAquantiles[15] <- sum(rec.75$summary[1:4, 1] * personA)
pAquantiles[16] <- sum(rec.80$summary[1:4, 1] * personA)
pAquantiles[17] <- sum(rec.85$summary[1:4, 1] * personA)
pAquantiles[18] <- sum(rec.90$summary[1:4, 1] * personA)
pAquantiles[19] <- sum(rec.95$summary[1:4, 1] * personA)

pBquantiles <- vector("numeric", length = 19)
pBquantiles[1] <- sum(rec.05$summary[1:4, 1] * personB)
pBquantiles[2] <- sum(rec.10$summary[1:4, 1] * personB)
pBquantiles[3] <- sum(rec.15$summary[1:4, 1] * personB)
pBquantiles[4] <- sum(rec.20$summary[1:4, 1] * personB)
pBquantiles[5] <- sum(rec.25$summary[1:4, 1] * personB)
pBquantiles[6] <- sum(rec.30$summary[1:4, 1] * personB)
pBquantiles[7] <- sum(rec.35$summary[1:4, 1] * personB)
pBquantiles[8] <- sum(rec.40$summary[1:4, 1] * personB)
pBquantiles[9] <- sum(rec.45$summary[1:4, 1] * personB)
pBquantiles[10] <- sum(rec.50$summary[1:4, 1] * personB)
pBquantiles[11] <- sum(rec.55$summary[1:4, 1] * personB)
pBquantiles[12] <- sum(rec.60$summary[1:4, 1] * personB)
pBquantiles[13] <- sum(rec.65$summary[1:4, 1] * personB)
pBquantiles[14] <- sum(rec.70$summary[1:4, 1] * personB)
pBquantiles[15] <- sum(rec.75$summary[1:4, 1] * personB)
pBquantiles[16] <- sum(rec.80$summary[1:4, 1] * personB)
pBquantiles[17] <- sum(rec.85$summary[1:4, 1] * personB)
pBquantiles[18] <- sum(rec.90$summary[1:4, 1] * personB)
pBquantiles[19] <- sum(rec.95$summary[1:4, 1] * personB)

pCquantiles <- vector("numeric", length = 19)
pCquantiles[1] <- sum(rec.05$summary[1:4, 1] * personC)
pCquantiles[2] <- sum(rec.10$summary[1:4, 1] * personC)
pCquantiles[3] <- sum(rec.15$summary[1:4, 1] * personC)
pCquantiles[4] <- sum(rec.20$summary[1:4, 1] * personC)
pCquantiles[5] <- sum(rec.25$summary[1:4, 1] * personC)
pCquantiles[6] <- sum(rec.30$summary[1:4, 1] * personC)
pCquantiles[7] <- sum(rec.35$summary[1:4, 1] * personC)
pCquantiles[8] <- sum(rec.40$summary[1:4, 1] * personC)
pCquantiles[9] <- sum(rec.45$summary[1:4, 1] * personC)
pCquantiles[10] <- sum(rec.50$summary[1:4, 1] * personC)
pCquantiles[11] <- sum(rec.55$summary[1:4, 1] * personC)
pCquantiles[12] <- sum(rec.60$summary[1:4, 1] * personC)
pCquantiles[13] <- sum(rec.65$summary[1:4, 1] * personC)
pCquantiles[14] <- sum(rec.70$summary[1:4, 1] * personC)
pCquantiles[15] <- sum(rec.75$summary[1:4, 1] * personC)
pCquantiles[16] <- sum(rec.80$summary[1:4, 1] * personC)
pCquantiles[17] <- sum(rec.85$summary[1:4, 1] * personC)
pCquantiles[18] <- sum(rec.90$summary[1:4, 1] * personC)
pCquantiles[19] <- sum(rec.95$summary[1:4, 1] * personC)

pAmeans <- c(sum(Phurdlerec$BUGSoutput$summary[1:4, 1]*personA),
             sum(NBhurdlerec$BUGSoutput$summary[1:4, 1]*personA),
             sum(Pnhrec$BUGSoutput$summary[1:4, 1]*personA),
             sum(NBnhrec$BUGSoutput$summary[1:4, 1]*personA))
names(pAmeans) <- c("Phurd", "NBhurd", "Pnh", "NBnh")

pBmeans <- c(sum(Phurdlerec$BUGSoutput$summary[1:4, 1]*personB),
             sum(NBhurdlerec$BUGSoutput$summary[1:4, 1]*personB),
             sum(Pnhrec$BUGSoutput$summary[1:4, 1]*personB),
             sum(NBnhrec$BUGSoutput$summary[1:4, 1]*personB))
names(pBmeans) <- c("Phurd", "NBhurd", "Pnh", "NBnh")

pCmeans <- c(sum(Phurdlerec$BUGSoutput$summary[1:4, 1]*personC),
             sum(NBhurdlerec$BUGSoutput$summary[1:4, 1]*personC),
             sum(Pnhrec$BUGSoutput$summary[1:4, 1]*personC),
             sum(NBnhrec$BUGSoutput$summary[1:4, 1]*personC))
names(pCmeans) <- c("Phurd", "NBhurd", "Pnh", "NBnh")

################################################################################################
######  Simulation  ############################################################################
################################################################################################

library(quantreg)
library(lqmm)

# function to compare Bayesian and frequentist hurdle QR for counts with log transform.
# only considering QR portion, not logistic portion.
# generating 100 data sets at each setting.
# comparing prediction error.

bfcompare <- function(quant = 0.5, n = 25, M = 100){
  
  Fprederrors <- matrix(NA, nrow = M, ncol = n)
  Bprederrors <- matrix(NA, nrow = M, ncol = n)
  intercept.sim <- rep(1, n)
  
  for(i in 1:M){
    
    x1 <- rnorm(n, 5, 1)
    x2 <- rnorm(n, 3, 1)
    
    q <- floor(0.5 + 1.5*x1 + 0*x2)
    
    f <- lqm.counts(q ~ x1 + x2, tau = quant)
    
    Fprederrors[i, ] <- f$fitted - q[q !=0] - 0.5
    
    bjit <- runif(n, 0, 1)
    q.transformed <- log(q + bjit)
    
    sim.data <- list("q.transformed", "n", "tau", "intercept.sim", "x1", "x2")
    
    sim.initials <- list(
      list(beta = c(0, 0, -10), delta = 2, w = rep(0.5, n)),
      list(beta = c(0.5, 1.5, 0), delta = 0.5, w = rep(1, n)),
      list(beta = c(1, 3, 10), delta = 3, w = rep(2, n)))
    
    sim.keeps <- c("beta")
    tau <- quant
    
    b <- jags(model.file = "sim2.txt", data = sim.data, inits = sim.initials,
              n.iter = 126000, n.chains = 3, n.thin = 50, n.burnin = 1000,
              parameters.to.save = sim.keeps)
    
    Bestimates <- b$BUGSoutput$summary[1:3, 1]
    Bprederrors[i, ] <- exp(Bestimates[1] + Bestimates[2]*x1[q != 0] +
                              Bestimates[3]*x2[q != 0]) - q[q != 0] - 0.5
  }
  
  list(Fpred = Fprederrors, Bpred = Bprederrors)
}

q50n25 <- bfcompare()
q50n50 <- bfcompare(n = 50)
q50n100 <- bfcompare(n = 100)

q90n25 <- bfcompare(quant = 0.9)
q90n50 <- bfcompare(quant = 0.9, n = 50)
q90n100 <- bfcompare(quant = 0.9, n = 100)

q25n25 <- bfcompare(quant = 0.25)
q25n50 <- bfcompare(quant = 0.25, n = 50)
q25n100 <- bfcompare(quant = 0.25, n = 100)

simpreds <- data.frame(Prediction_Error = c(q25n25$Fpred, q25n25$Bpred,
                                            q25n50$Fpred, q25n50$Bpred,
                                            q25n100$Fpred, q25n100$Bpred,
                                            q50n25$Fpred, q50n25$Bpred,
                                            q50n50$Fpred, q50n50$Bpred,
                                            q50n100$Fpred, q50n100$Bpred,
                                            q90n25$Fpred, q90n25$Bpred,
                                            q90n50$Fpred, q90n50$Bpred,
                                            q90n100$Fpred, q90n100$Bpred),
                       Quantile = rep(c("0.25", "0.50", "0.90"), each = 35000),
                       Sample_Size = rep(c("25", "50", "100"), c(5000, 10000, 20000)),
                       Method = rep(c("Frequentist", "Bayesian", "Frequentist", "Bayesian",
                                      "Frequentist", "Bayesian"), c(2500, 2500,
                                                                    5000, 5000,
                                                                    10000, 10000)))

simpreds$Sample_Size <- factor(simpreds$Sample_Size, levels = c("25", "50", "100"))

save(simpreds, file = "simpreds.RData")

# box plot with whiskers but no outliers
ggplot(data = simpreds) + 
  geom_boxplot(aes(x = Sample_Size, y = Prediction_Error, fill = Method),
               outlier.shape = NA) +
  scale_fill_manual(values = c("dodgerblue", "white")) +
  facet_wrap(~ Quantile, nrow = 1) +
  ylim(c(-1.25, 2.25)) +
  labs(x = "Sample Size", y = "Prediction Error",
       title = "Prediction Error Faceted by Quantile") +
  theme(plot.title = element_text(hjust = 0.5))

# second function to compare intervals with truth known (no log transformation)
# for beta1 (informed prior) and beta2 (uninformed prior)

bfcompare2 <- function(quant = 0.5, n = 25, M = 100){
  
  intercept.sim <- rep(1, n)
  Fbounds <- matrix(NA, nrow = M, ncol = 4)
  Bbounds <- matrix(NA, nrow = M, ncol = 4)
  
  for(i in 1:M){
    
    x1 <- rnorm(n, 5, 1)
    x2 <- rnorm(n, 3, 1)
    
    counts <- floor(0.5 + 1.5*x1 + 0*x2)
    
    freqest <- matrix(NA, nrow = 50, ncol = 4)
    
    for(j in 1:50){
      
      q <- counts + runif(n)
      
      f <- rq(q ~ x1 + x2, tau = quant)
      
      freqest[j, ] <- c(summary(f)$coefficients[2, 2:3],
                        summary(f)$coefficients[3, 2:3])
    }
    Fbounds[i, ] <- colMeans(freqest)
    
    q.transformed <- counts + runif(n)
    
    sim.data <- list("q.transformed", "n", "tau", "intercept.sim", "x1", "x2")
    
    sim.initials <- list(
      list(beta = c(0, 0, -10), delta = 2, w = rep(0.5, n)),
      list(beta = c(0.5, 1.5, 0), delta = 0.5, w = rep(1, n)),
      list(beta = c(1, 3, 10), delta = 3, w = rep(2, n)))
    
    sim.keeps <- c("beta")
    tau <- quant
    
    b <- jags(model.file = "sim2.txt", data = sim.data, inits = sim.initials,
              n.iter = 126000, n.chains = 3, n.thin = 50, n.burnin = 1000,
              parameters.to.save = sim.keeps)
    
    Bbounds[i, ] <- c(b$BUGSoutput$summary[2, c(3, 7)],
                      b$BUGSoutput$summary[3, c(3, 7)])
  }
  list(Fbounds = Fbounds, Bbounds = Bbounds)
}

int25 <- bfcompare2()
int50 <- bfcompare2(n = 50)
int100 <- bfcompare2(n = 100)

save(int25, file = "simint25.RData")
save(int50, file = "simint50.RData")
save(int100, file = "simint100.RData")

intwidths <- data.frame(Width = c(int25$Fbounds[,2] - int25$Fbounds[,1],
                                  int25$Fbounds[,4] - int25$Fbounds[,3],
                                  int25$Bbounds[,2] - int25$Bbounds[,1],
                                  int25$Bbounds[,4] - int25$Bbounds[,3],
                                  int50$Fbounds[,2] - int50$Fbounds[,1],
                                  int50$Fbounds[,4] - int50$Fbounds[,3],
                                  int50$Bbounds[,2] - int50$Bbounds[,1],
                                  int50$Bbounds[,4] - int50$Bbounds[,3],
                                  int100$Fbounds[,2] - int100$Fbounds[,1],
                                  int100$Fbounds[,4] - int100$Fbounds[,3],
                                  int100$Bbounds[,2] - int100$Bbounds[,1],
                                  int100$Bbounds[,4] - int100$Bbounds[,3]),
                        Parameter = rep(c("b1", "b2"), each = 100),
                        Method = rep(c("Frequentist", "Bayesian"), each = 200),
                        Sample_Size = rep(c("25", "50", "100"), each = 400))

intwidths$Sample_Size <- factor(intwidths$Sample_Size, levels = c("25", "50", "100"))
levels(intwidths$Parameter) <- c(expression(beta[1]), expression(beta[2]))

ggplot(data = intwidths) +
  geom_boxplot(aes(x = Sample_Size, y = Width, fill = Method),
               position = position_dodge(0.8)) +
  scale_fill_manual(values = c("dodgerblue", "white")) +
  facet_wrap(~ Parameter, nrow = 1, labeller = label_parsed) +
  labs(x = "Sample Size", title = expression("Distribution of Interval Widths for"~beta[1]~"and"~beta[2])) +
  theme(plot.title = element_text(hjust = 0.5),
        strip.text = element_text(size = 12))

sum(int25$Bbounds[, 1] < 1.5 & int25$Bbounds[, 2] > 1.5)
sum(int25$Fbounds[, 1] < 1.5 & int25$Fbounds[, 2] > 1.5)
sum(int50$Bbounds[, 1] < 1.5 & int50$Bbounds[, 2] > 1.5)
sum(int50$Fbounds[, 1] < 1.5 & int50$Fbounds[, 2] > 1.5)
sum(int100$Bbounds[, 1] < 1.5 & int100$Bbounds[, 2] > 1.5)
sum(int100$Fbounds[, 1] < 1.5 & int100$Fbounds[, 2] > 1.5)
sum(int25$Bbounds[, 3] < 0 & int25$Bbounds[, 4] > 0)
sum(int25$Fbounds[, 3] < 0 & int25$Fbounds[, 4] > 0)
sum(int50$Bbounds[, 3] < 0 & int50$Bbounds[, 4] > 0)
sum(int50$Fbounds[, 3] < 0 & int50$Fbounds[, 4] > 0)
sum(int100$Bbounds[, 3] < 0 & int100$Bbounds[, 4] > 0)
sum(int100$Fbounds[, 3] < 0 & int100$Fbounds[, 4] > 0)
