function ModelChoice(alldata)
%MODELCHOICE Model choice of fault size distribution: McMC simulation algorithm
% model 1 : power law distribution
% model 2 : exponential distribution
% The program requires MATLAB statistical toolbox.

global DATA NDATA PDF_M PARAM TRAINING INCR MODELRND ORDINARYRND PSEUDORND

rand('state',sum(100*clock))

% - Number of iterations at differtent stages of the algorithm ----------------

burnin     = 10000;
training   = 5000; % Must be > 1
step_train = 10;
nsample    = 5000;
step       = 10;

check_burnin = 0; % If 1, write burn-iterations to file

max_rnr = 1000; % Numer of random numbers generated at a time

% - Define parameter values ---------------------------------------------------

PARAM.probsmall  = 0.001;             % p1
PARAM.problarge  = 0.99;              % p2
PARAM.obslimit   = 2;                 % y0, lower limit of observations
PARAM.lower      = 2;                 % eta1, limit of resolution
PARAM.upper      = 20;                % eta2, size > PARAM.upper: observable fault
PARAM.mintrunc   = 1; 		      % delta1, minimum truncation of fault end
PARAM.maxtrunc   = 5;                 % delta2, maximum truncation
PARAM.truncrange = PARAM.maxtrunc-PARAM.mintrunc;
PARAM.maxsize    = 3000000;           % PARAM.upper limit of observable size (censoring)
PARAM.maxsize    = max(max(alldata),PARAM.maxsize);
PARAM.mu_nfaults = [600 300];         % [omega1 omega2]
PARAM.hp1 = [4 4];                    % [alpha1 alpha2]
PARAM.hp2 = [.25 0.005];              % [beta1 beta2]
mu_param    = PARAM.hp1.*PARAM.hp2;
PARAM.delta_mu   = 0.05*mu_param;     % Parameter for transition kernel

PARAM.missing = 1-(PARAM.probsmall==1 | (PARAM.obslimit>PARAM.upper & PARAM.problarge==1));

if(PARAM.probsmall>PARAM.problarge)
  error('Likelihood: p(fault_size) must be an increasing function')
end
if(PARAM.lower>(min(alldata)+PARAM.mintrunc))
  error('choice of lower limit of observations does not correspond with observations');
end


% - Use only data above lower limit -------------------------------------------

DATA  = alldata(alldata>=PARAM.obslimit);
NDATA = length(DATA);
DATA  = reshape(DATA,1,NDATA);

% - Probabilities of observation ----------------------------------------------

pobs1 = P_Obs(1,mu_param(1));
pobs2 = P_Obs(2,mu_param(2));
probabilities = [pobs1 pobs2];

% - Counters of accepts and numbers of each type of McMC-steps ----------------

n_type = 3;
acc = zeros(1,n_type);
n_model = zeros(1,2);

% - Initiate variables --------------------------------------------------------

[TRAINING.param,TRAINING.nfaults] = deal(NaN);

model = unidrnd(2);

for i=1:2
  fault(i) = struct('model',i,'nfaults',[],'param',[],...
		    'pdf_nfaults',[],'pdf_param',[],'pdf_pseudo',NaN,...
		    'likelihood_o',[],'likelihood_u',[]);
  fault(i).param   = MLE_Param(DATA,PARAM.obslimit,i);
  fault(i).nfaults = round(NDATA/probabilities(i));
end

if(PARAM.missing==0) % no missing observations
  [fault(1).nfaults,fault(2).nfaults] = deal(NDATA);
end

fault(1) = CalculatePdfLikelihood(fault(1));
fault(2) = CalculatePdfLikelihood(fault(2));

PDF_M = log([0.5 0.5]);
update_pdf_m = 1;

% ---
% - Start iterations ----------------------------------------------------------
% ---

% - burn-in and training iterations ---

disp(sprintf('%d burn-in iterations and %d training iterations...\n',...
	     burnin,step_train*training));

status = [];
state  = 'Burnin';

RandomNumbers(min(burnin+step_train*training,max_rnr),0)

for iteration = 1:(burnin+step_train*training)
  rnr = rem((iteration-1),max_rnr)+1;
  [fault(1),accept] = UpdateOrdinary(fault(1),rnr);
  [fault(2),accept] = UpdateOrdinary(fault(2),rnr);

  if(mod(iteration,max_rnr)==0 & iteration<(burnin+step_train*training))
    RandomNumbers(max_rnr,0)
  end
  
  j = iteration-burnin*(iteration>burnin);
  
  if(mod(j,100)==0)
    all = ((mod(j,1000)==0) | iteration==burnin | ...
	   iteration==(burnin+step_train*training));
    status = WriteStatus(0,j,all,fault,status,state);
  end

  if(iteration<=burnin | mod(j,step_train)==0)
    if(iteration>burnin)
      j = j/step_train;
    end
    [param1_s(j),param2_s(j),nfaults_s(j),nfaults1_s(j),...
     nfaults2_s(j),l_s(j),f_s(j)] = CollectState(model,fault);
    model_s(j) = 1.5;
  end
    
  if(iteration==burnin)
    if(check_burnin)
      save burnin param1_s param2_s nfaults1_s nfaults2_s l_s f_s
    end
    clear *_s
    state  = 'Training';
    status = [];
  end  
end

% - Prepare training data -> pseudopriors on parameters ---

TRAINING.param = [param1_s' param2_s'];
if(PARAM.missing==1)
  TRAINING.nfaults = [nfaults1_s' nfaults2_s'];
end

n = length(param1_s);
for i=1:2
  INCR.param(i) = (4/3)^0.2*std(TRAINING.param(:,i))/n^0.2;
  if(PARAM.missing==1)
    INCR.nfaults(i) = (4/3)^0.2*std(TRAINING.nfaults(:,i))/n^0.2;
  end
end

clear *_s

% - Prepare pseudopriors on model ---

if(update_pdf_m)
  fault_old = fault;

  n    = length(TRAINING.param(:,1));
  nnn  = min(500,n); % Number of randomly chosen points used to find 
                     % rough estimate of Bayes factor
  done = zeros(1,n);
  nrs  = 1:n;
  
  if(nnn>1)
    for j=1:nnn
      sub = nrs(done==0);
      k   = sub(unidrnd(length(nrs(done==0))));
      done(k) = 1;
      for i=1:2
	if(PARAM.missing==1)
	  fault(i).nfaults = TRAINING.nfaults(k,i);
	else
	  fault(i).nfaults = NDATA;
	end
	fault(i).param = TRAINING.param(k,i);
	fault(i) = CalculatePdfLikelihood(fault(i));
      end
      a(j) = FullLikelihood(fault(1))+fault(1).pdf_nfaults+...
	     fault(1).pdf_param+EmpiricalPdf(fault(2))-...
	     (FullLikelihood(fault(2))+fault(2).pdf_nfaults+...
	      fault(2).pdf_param+EmpiricalPdf(fault(1)));
    end
  else % use only map_estimate to find rough estimate
    for i=1:2
      if(PARAM.missing==1)
	  map_train = Mode(TRAINING.param(:,i),TRAINING.nfaults(:,i));
	  fault(i).nfaults = round(map_train(2));
      else
	  map_train = Mode(TRAINING.param(:,i),[]);
	  fault(i).nfaults = NDATA;
      end
      fault(i).param = map_train(1);
      fault(i) = CalculatePdfLikelihood(fault(i));
    end
    a = FullLikelihood(fault(1))+fault(1).pdf_nfaults+...
                   fault(1).pdf_param+EmpiricalPdf(fault(2))-...
       (FullLikelihood(fault(2))+fault(2).pdf_nfaults+...
	  fault(2).pdf_param+EmpiricalPdf(fault(1)));
  end
  
  a = mean(a);
  
  fault = fault_old;
  clear fault_old
  if(abs(a)<10)
    p = exp(-log(exp(a)+1));
    PDF_M = log([p 1-p]);
  else
    if(a<0)
      p = exp(a);
      PDF_M = [0 a];
    else
      p = exp(-a);
      PDF_M = [-a 0];
    end
  end
  bayes_rough = exp(a);
else
  bayes_rough = NaN;
end

% - sampling iterations ---

status = [];
state  = 'Sampling';

disp(sprintf('%d sampling iterations...\n',step*nsample))

MODELRND = unifrnd(0,1,10,1);
for i=1:10 % "mini-burnin" of model indicator
  [model,fault,accept] = UpdateModel(model,fault,i);
end
[fault(1).pdf_pseudo,fault(2).pdf_pseudo] = deal(NaN);

RandomNumbers(min(step*nsample,max_rnr),1)

for iteration=1:(step*nsample)
  rnr = rem((iteration-1),max_rnr)+1;
  [model,fault,accept(1)]  = UpdateModel(model,fault,rnr);
  [fault(model),accept(2)] = UpdateOrdinary(fault(model),rnr);
  [fault(Swap(model)),accept(3)] = UpdatePseudo(fault(Swap(model)),rnr);

  if(accept(2))
    fault(model).pdf_pseudo = NaN;
  end
  if(accept(3))
    fault(Swap(model)).pdf_pseudo = NaN;
  end
  
  if(mod(iteration,max_rnr)==0 & iteration<(step*nsample))
    RandomNumbers(max_rnr,1)
  end
  
  model_all(iteration) = model;
  n_model(model) = n_model(model)+1;
  acc = acc+accept;

  if(mod(iteration,100)==0)
    all = (mod(iteration,1000)==0 | iteration==(step*nsample));
    status = WriteStatus(model,iteration,all,fault,status,state);
  end

  if(mod(iteration,step)==0)
    j = iteration/step;
    [param1_s(j),param2_s(j),nfaults_s(j),nfaults1_s(j),...
     nfaults2_s(j),l_s(j),f_s(j)] = CollectState(model,fault);
    if(n_model(2)>0)
      bayes_s(j) = n_model(1)/n_model(2)*exp(PDF_M(2)-PDF_M(1));
    else
      bayes_s(j) = inf;
    end
  end
  
end

% - Accept rates --------------------------------------------------------------

acc_rate = acc/(nsample*step);

% - Output --------------------------------------------------------------------

if(n_model(2)>0)
  Bayes = n_model(1)/n_model(2)*exp(PDF_M(2)-PDF_M(1));
else
  Bayes = inf;
end

model_s = model_all(step:step:(step*nsample));

n1 = length(model_s(model_s==1));
n2 = length(model_s(model_s==2));

if(n1>0)
  param1_mean = mean(param1_s(model_s==1));
  n1_mean = mean(nfaults1_s(model_s==1));
  fault_size_1 = [SampleFaultSizeObserved(1,param1_mean) ...
		  SampleFaultSizeUnobserved(round(n1_mean)-NDATA,1,param1_mean)];
else
  [param1_mean,fault_size_1,n1_mean] = deal(NaN);
end
if(n1>1)
  param1_std = std(param1_s(model_s==1))/sqrt(n1);
  n1_std = std(nfaults1_s(model_s==1))/sqrt(n1);
else
  [param1_std,n1_std] = deal(NaN);
end

if(n2>0)
  param2_mean = mean(param2_s(model_s==2));
  n2_mean = mean(nfaults2_s(model_s==2));
  fault_size_2 = [SampleFaultSizeObserved(2,param2_mean) ...
		  SampleFaultSizeUnobserved(round(n2_mean)-NDATA,2,param2_mean)];
else
  [param2_mean,n2_mean,fault_size_2] = deal(NaN);
end
if(n2>1)
  param2_std  = std(param2_s(model_s==2))/sqrt(n2);
  n2_std = std(nfaults2_s(model_s==2))/sqrt(n2);
else
  [param2_std,n2_std] = deal(NaN);
end

nfaults_mean = [mean(nfaults_s) n1_mean n2_mean];
nfaults_std  = [std(nfaults_s) n1_std n2_std];

[b_l,b_u] = ConfidenceInterval([n_model(1) n_model(2)],0.05);
bayes_conf_95 = [b_l b_u]*exp(PDF_M(2)-PDF_M(1));
if(n2>0)
  bayes_sub = n1/n2*exp(PDF_M(2)-PDF_M(1));
else
  bayes_sub = Inf;
end

% - Save results --------------------------------------------------------------

save models model_all
save simulations param1_s param2_s nfaults_s nfaults1_s nfaults2_s l_s f_s bayes_s
save sample fault_size_1 fault_size_2
save results Bayes param1_mean param1_std param2_mean param2_std nfaults_mean ...
    nfaults_std acc_rate bayes_sub bayes_conf_95 bayes_rough n1 n2 PDF_M

% - List results --------------------------------------------------------------

disp(sprintf('\n\n\nSimulation results:\n'))
disp(sprintf('    Estiamted Bayes factor = %.4g\n',Bayes))
disp(sprintf('    Number of realizations from each model = %d %d\n',n_model))
disp(sprintf('    Parameter 1 = %.4f (%.4f)',[param1_mean param1_std]))
disp(sprintf('    Parameter 2 = %.4f (%.4f)\n',[param2_mean param2_std]))
nfaults_mean = round(nfaults_mean);
disp(sprintf('    Number of faults 1 = %d (%.1f)',[nfaults_mean(2) nfaults_std(2)]))
disp(sprintf('    Number of faults 2 = %d (%.1f)\n\n',[nfaults_mean(3) nfaults_std(3)]))
acc_rate


%%% --------------------------------------------------------------------
function fault = CalculatePdfLikelihood(fault_old)
% Calculate all prior pdfs and likelihoods under a model (log-values)
% If all=0, pseudo_pdf is not calculated
    
global PARAM

fault = fault_old;
i = fault.model;

fault.pdf_nfaults  = PdfNofFaults(fault.nfaults,i);
fault.pdf_param    = PdfParam(fault.param,PARAM.hp1(i),PARAM.hp2(i));
fault.likelihood_o = log(PdfObs(i,fault.param));
if(PARAM.missing==1)
  fault.likelihood_u = log(1-P_Obs(i,fault.param));
else
  fault.likelihood_u = 0;
end


%%% --------------------------------------------------------------------
function RandomNumbers(n,all)
% generate all random nubmers in advance

global PARAM INCR TRAINING MODELRND ORDINARYRND PSEUDORND

ORDINARYRND.u = unifrnd(0,1,n,2);
ORDINARYRND.n = normrnd(0,PARAM.delta_mu(1),n,1);
ORDINARYRND.n(:,2) = normrnd(0,PARAM.delta_mu(2),n,1);
if(all)
  MODELRND = unifrnd(0,1,n,1);
  if(PARAM.missing)
    PSEUDORND.sign = 2*unidrnd(2,n,1)-3;
    PSEUDORND.p = poissrnd(INCR.nfaults(1),n,1);
    PSEUDORND.p(:,2) = poissrnd(INCR.nfaults(2),n,1);
  end
  PSEUDORND.n = normrnd(0,INCR.param(1),n,1);
  PSEUDORND.n(:,2) = normrnd(0,INCR.param(2),n,1);
  PSEUDORND.u = unidrnd(length(TRAINING.param(:,1)),n,1);
end


%%% --------------------------------------------------------------------
function [param1_s,param2_s,nfaults_s,nfaults1_s,nfaults2_s,l_s,f_s] = CollectState(model,fault)

global PDF_M

pdf_posterior = PDF_M(model)+FullLikelihood(fault(model))+...
    fault(model).pdf_nfaults+fault(model).pdf_param+...
    EmpiricalPdf(fault(Swap(model)));

param1_s   = fault(1).param;
param2_s   = fault(2).param;
nfaults_s  = fault(model).nfaults;
nfaults1_s = fault(1).nfaults;
nfaults2_s = fault(2).nfaults;
l_s = FullLikelihood(fault(model));
f_s = pdf_posterior;


%%% --------------------------------------------------------------------
function p = EmpiricalPdf(fault)
% log of empirical pdf of (param,nfaults), using kernel density based on
% training samples

global NDATA TRAINING INCR PARAM


if(isnan(TRAINING.param))
  p = 0;
else
  t_p     = TRAINING.param(:,fault.model);
  sigma   = INCR.param(fault.model);
  p_param = normpdf(fault.param-t_p,0,sigma)./normcdf(t_p,0,sigma);

  if(PARAM.missing==1)
    t_k = TRAINING.nfaults(:,fault.model);
    lambda  = INCR.nfaults(fault.model);
    numbers = (1:length(t_k));
    ind1 = numbers(t_k<fault.nfaults);
    ind2 = numbers(t_k==fault.nfaults);
    ind3 = numbers(t_k>fault.nfaults);
    if(length(ind1)>0)
      p_k(ind1) = 0.5*poisspdf(fault.nfaults-t_k(ind1),lambda);
    end
    if(length(ind2)>0)
      p_k(ind2) = 0.5*exp(-lambda)*...
	  (1+poisscdf(t_k(ind2)-NDATA,lambda).^(-1));
    end
    if(length(ind3)>0)
      p_k(ind3) = 0.5*poisspdf(t_k(ind3)-fault.nfaults,lambda)./ ...
	  poisscdf(t_k(ind3)-NDATA,lambda);
    end
  else
    p_k = 1.0;
  end

  mmm = mean(p_param.*p_k');
  if(mmm>0)
    p = log(mmm);
  else
    p = -1000000.0;
  end
end


%%% --------------------------------------------------------------------
function l = FullLikelihood(fault)

global NDATA

l = sum(fault.likelihood_o)+(fault.nfaults-NDATA)*fault.likelihood_u;


%%% --------------------------------------------------------------------
function f = PdfNofFaults(n,model)

global PARAM

f = n*log(PARAM.mu_nfaults(model))-PARAM.mu_nfaults(model)-sum(log(1:n));


%%% --------------------------------------------------------------------
function f = PdfParam(sample,param1,param2)

f = log(gampdf(sample,param1,param2));


%%% --------------------------------------------------------------------
function p = PdfObs(model,param)
% integral of h(x|y)p(y)f(y|param,model)
    
global NDATA DATA PARAM

if(PARAM.truncrange==0)
  p = P_ObsFaultSize(DATA).*PdfFaultSize(DATA,model,param);
else
  numbers = 1:NDATA;  
  ind1 = numbers(DATA<=(PARAM.maxsize-PARAM.truncrange) & PARAM.lower>(DATA+PARAM.mintrunc));
  ind2 = numbers(DATA<=(PARAM.maxsize-PARAM.truncrange) & PARAM.upper>(DATA+PARAM.mintrunc));
  ind3 = numbers(DATA<=(PARAM.maxsize-PARAM.truncrange) & PARAM.upper<(DATA+PARAM.maxtrunc));
  ind4 = numbers(DATA>(PARAM.maxsize-PARAM.truncrange) & PARAM.lower>(PARAM.maxsize+PARAM.mintrunc));
  ind5 = numbers(DATA>(PARAM.maxsize-PARAM.truncrange) & PARAM.upper>(PARAM.maxsize+PARAM.mintrunc));
  ind6 = numbers(DATA>(PARAM.maxsize-PARAM.truncrange));
  if(PARAM.upper>PARAM.lower)
    a = (PARAM.problarge-PARAM.probsmall)/(PARAM.upper-PARAM.lower);
    b = (PARAM.probsmall*PARAM.upper-PARAM.problarge*PARAM.lower)/(PARAM.upper-PARAM.lower);
  end
  iii = zeros(6,NDATA);
  if(length(ind1)>0)
    x = DATA(ind1);
    iii(1,ind1) = PARAM.probsmall/PARAM.truncrange*...
	Integral1(x+PARAM.mintrunc,min(PARAM.lower,x+PARAM.maxtrunc),model,param);
  end
  if(length(ind2)>0 & PARAM.upper>PARAM.lower)
    x = DATA(ind2);
    iii(2,ind2) = ...
	(a*Integral2(max(PARAM.lower,x+PARAM.mintrunc),min(PARAM.upper,x+PARAM.maxtrunc),model,param)+...
	 b*Integral1(max(PARAM.lower,x+PARAM.mintrunc),min(PARAM.upper,x+PARAM.maxtrunc),model,param))/...
	PARAM.truncrange;
  end
  if(length(ind3)>0)
    x = DATA(ind3);
    iii(3,ind3) = PARAM.problarge/PARAM.truncrange*...
	Integral1(max(PARAM.upper,x+PARAM.mintrunc),x+PARAM.maxtrunc,model,param);
  end
  if(length(ind4)>0)
    x = DATA(ind4);
    iii(4,ind4) = PARAM.probsmall/PARAM.truncrange*...
	Integral1(PARAM.maxsize+PARAM.mintrunc,PARAM.lower,model,param);
  end
  if(length(ind5)>0 & PARAM.upper>PARAM.lower)
    x = DATA(ind5);
    iii(5,ind5) = ...
	(a*Integral2(max(PARAM.lower,PARAM.maxsize+PARAM.mintrunc),PARAM.upper,model,param)+...
	 b*Integral1(max(PARAM.lower,PARAM.maxsize+PARAM.mintrunc),PARAM.upper,model,param))/...
	PARAM.truncrange;
  end
  if(length(ind6)>0)
    x = DATA(ind6);
    iii(6,ind6) = PARAM.problarge/PARAM.truncrange*...
	Integral1(max(PARAM.upper,PARAM.maxsize+PARAM.mintrunc),Inf,model,param);
  end
  p = sum(iii,1);
end


%%% --------------------------------------------------------------------
function f = PdfFaultSize(x,model,param)
% pdf of fault size

global PARAM

if(length(x)==0)
  f = 1;
elseif(model==1)
  f = param*PARAM.obslimit^param./x.^(param+1);
else
  f = param*exp(-param*(x-PARAM.obslimit));
end


%%% --------------------------------------------------------------------
function iii = Integral1(y0,y1,model,param)
% integrating f(y|param,model)

global PARAM

if(model==1)
  iii = (PARAM.obslimit./y0).^param-(1-isinf(y1)).*(PARAM.obslimit./y1).^param;
else
  iii = exp(-param*(y0-PARAM.obslimit))-(1-isinf(y1)).*exp(-param*(y1-PARAM.obslimit));
end


%%% --------------------------------------------------------------------
function iii = Integral2(y0,y1,model,param)
% integrating y*f(y|param,model)

global PARAM

if(model==1)
  if(param==1)
    iii = PARAM.obslimit*(log(y1)-log(y0));
  else
    iii = param/(param-1)*(y0.*(PARAM.obslimit./y0).^param-y1.*(PARAM.obslimit./y1).^param);
  end
else
  iii = (y0+1/param).*exp(-param*(y0-PARAM.obslimit))-(y1+1/param).*exp(-param*(y1-PARAM.obslimit));
end


%%% --------------------------------------------------------------------
function p = P_Obs(model,param)
% Probability of observing a fault for given model and model parameters: 
% E{p.obs(x)|model,param}

global PARAM

if(PARAM.upper>PARAM.lower)
  if(model==1)
    if(param~=1)
      aaa = (PARAM.lower*exp(param*(log(PARAM.obslimit)-log(PARAM.lower)))-...
	     PARAM.upper*exp(param*(log(PARAM.obslimit)-log(PARAM.upper))))/(param-1);
    else
      aaa = PARAM.obslimit*(log(PARAM.upper)-log(PARAM.lower));
    end
  else
    aaa = exp(param*PARAM.obslimit)*(exp(-param*PARAM.lower)-exp(-param*PARAM.upper))/param;
  end
  p = PARAM.probsmall + (PARAM.problarge-PARAM.probsmall)/(PARAM.upper-PARAM.lower)*aaa;
else
  if(model==1)
    p = PARAM.probsmall+(PARAM.problarge-PARAM.probsmall)*exp(param*(log(PARAM.obslimit)-log(PARAM.lower)));
  else
    p = PARAM.probsmall+(PARAM.problarge-PARAM.probsmall)*exp(-param*(PARAM.lower-PARAM.obslimit));
  end
end


%%% --------------------------------------------------------------------
function p = P_ObsFaultSize(x)
% Probability of observing fault of size x: p.obs(x)

global PARAM

epsilon = 1e-8;

n = length(x);
if(n>0)
  numbers = 1:n;
  type = ones(1,n);
  type(x>PARAM.lower) = 2;
  type(x>PARAM.upper)  = 3;
  ind1 = numbers(type==1);
  ind2 = numbers(type==2);
  ind3 = numbers(type==3);  
  p(ind1) = PARAM.probsmall;
  p(ind2) = ((PARAM.problarge-PARAM.probsmall)*x(ind2)+...
	     (PARAM.probsmall*PARAM.upper-PARAM.problarge*PARAM.lower))/(PARAM.upper-PARAM.lower);
  p(ind3) = PARAM.problarge;
else
  p = [];
end

p = max(p,epsilon);
p = min(p,1-epsilon);


%%% --------------------------------------------------------------------
function [model_new,fault_new,accept] = UpdateModel(model_old,fault,i)
% Update step for model

global PDF_M MODELRND

model_new = Swap(model_old);
fault_new = fault;

if(isnan(fault(model_old).pdf_pseudo)==1)
  pseudo_old = EmpiricalPdf(fault(model_old));
else
  pseudo_old = fault(model_old).pdf_pseudo;
end

if(isnan(fault(model_new).pdf_pseudo)==1)
  pseudo_new = EmpiricalPdf(fault(model_new));
else
  pseudo_new = fault(model_new).pdf_pseudo;
end

ratio = (PDF_M(model_new)+FullLikelihood(fault(model_new))+...
	 fault(model_new).pdf_nfaults+fault(model_new).pdf_param+...
	 pseudo_old)-...
	(PDF_M(model_old)+FullLikelihood(fault(model_old))+...
	 fault(model_old).pdf_nfaults+fault(model_old).pdf_param+...
	 pseudo_new);

accept = (MODELRND(i)<min(1,exp(ratio)));

if(accept~=1)
  model_new = model_old;
end

fault_new(model_new).pdf_pseudo = pseudo_new;
fault_new(model_old).pdf_pseudo = pseudo_old;


%%% --------------------------------------------------------------------
function [fault,accept] = UpdateOrdinary(fault_old,i)
% Update step for parameter and dimension under active model

global PARAM ORDINARYRND

[fault,ratio(1)] = SuggestParam(fault_old,i);
if(isnan(ratio(1)))
  accept = 0;
else
  if(PARAM.missing)
    if(ORDINARYRND.u(i,1)<=(P_Up(fault.nfaults)/2))
      [fault,ratio(2)] = SuggestIncrease(fault);
    else
      [fault,ratio(2)] = SuggestDecrease(fault);
    end
  else
    ratio(2) = 0;
  end
  accept = (ORDINARYRND.u(i,2)<min(1,exp(sum(ratio))));
end
if(accept==0)
  fault = fault_old;
end


%%% --------------------------------------------------------------------
function [fault,accept] = UpdatePseudo(fault_old,i)
% Update step for parameter and dimension under inactive model

global TRAINING PARAM PSEUDORND

fault = fault_old;

if(PARAM.missing==1)
  k1 = TRAINING.nfaults(PSEUDORND.u(i),fault.model);
  fault.nfaults = k1+IncrementNofFaults(k1,fault.model,i);
end
param1 = TRAINING.param(PSEUDORND.u(i),fault.model);
fault.param = param1+IncrementParam(param1,fault.model,i);
fault = CalculatePdfLikelihood(fault);

accept = 1;


%%% --------------------------------------------------------------------
function [fault,ratio] = SuggestParam(fault_old,i)
% Suggest new parameter and unobserved fault sizes

global PARAM ORDINARYRND

fault = fault_old;

delta = PARAM.delta_mu(fault.model);

% gaussian transition kernel
add = ORDINARYRND.n(i,fault.model);
while(abs(add)>(5*delta))
  add = normrnd(0,delta);
end
fault.param = fault.param+add;

if(fault.param<=0)
  ratio = NaN;
  fault.param = fault_old.param;
else
  fault.pdf_param    = PdfParam(fault.param,PARAM.hp1(fault.model),PARAM.hp2(fault.model));
  fault.likelihood_o = log(PdfObs(fault.model,fault.param));
  if(PARAM.missing==1)
    fault.likelihood_u = log(1-P_Obs(fault.model,fault.param));
  else
    fault.likelihood_u = 0;
  end
  ratio = FullLikelihood(fault)+fault.pdf_param-...
	  (FullLikelihood(fault_old)+fault_old.pdf_param);
end


%%% --------------------------------------------------------------------
function [fault,ratio] = SuggestDecrease(fault_old)
% Suggest decreasing dimension by 1

global NDATA

fault = fault_old;

fault.nfaults     = fault_old.nfaults-1;
fault.pdf_nfaults = PdfNofFaults(fault.nfaults,fault.model);

n = fault.nfaults;
ratio = -fault.likelihood_u+fault.pdf_nfaults-fault_old.pdf_nfaults+...
	log(n-NDATA)-log(n)+log(P_Up(n-1))-log(P_Down(n));


%%% --------------------------------------------------------------------
function [fault,ratio] = SuggestIncrease(fault_old)
% Suggest increasing dimension by 1

global NDATA

fault = fault_old;

fault.nfaults     = fault_old.nfaults+1;
fault.pdf_nfaults = PdfNofFaults(fault.nfaults,fault.model);

n = fault_old.nfaults;
ratio = fault.likelihood_u+fault.pdf_nfaults-fault_old.pdf_nfaults+...
	log(n+1)-log(n+1-NDATA)+log(P_Down(n+1))-log(P_Up(n));


%%% --------------------------------------------------------------------
function p = P_Down(n)

p = (n>0);


%%% --------------------------------------------------------------------
function p = P_Up(n)

p = (n==0)+1;


%%% --------------------------------------------------------------------
function incr = IncrementNofFaults(k,model,i);
% Double Poisson incrementation of sample from empirical distribution,
% used to make the pseudoprior span the entire sample space

global NDATA INCR PSEUDORND

sign = PSEUDORND.sign(i);
incr = sign*PSEUDORND.p(i,model);
while((k+incr)<NDATA)
  incr = sign*poissrnd(INCR.nfaults(model));
end


%%% --------------------------------------------------------------------
function incr = IncrementParam(param,model,i);
% Gaussian incrementation of sample from empirical distribution,
% used to make the pseudoprior span the entire sample space

global INCR PSEUDORND

incr = PSEUDORND.n(i,model);
while(incr<(-param))
  incr = normrnd(0,INCR.param(model));
end


%%% --------------------------------------------------------------------
function map = Mode(theta,k)
% find mode of dataset (theta,k)

if(length(k)~=length(theta) & length(k)>0)
  exit('k and theta must have equal dimension')
end

n = 15;

t1 = min(theta);
t2 = max(theta);
x  = t1:(t2-t1)/(n-1):t2;

if(length(k)==0)
  z = hist(theta,x);
  m = x(z==max(z));
  if(length(m)>1)
    m = m(unidrnd(length(m)));
  end
  map = [m NaN];
else
  k1 = min(k);
  k2 = max(k);
  y  = k1:(k2-k1)/(n-1):k2;
  
  dx = (x(2)-x(1))/2;
  dy = (y(2)-y(1))/2;
  for i=1:n
    data = theta(k<(y(i)+dy) & k>=(y(i)-dy));
    for j=1:n
      z(i,j) = length(data(data<(x(j)+dx) & data>=(x(j)-dx)));
    end
  end

  [m1,j] = max(z);
  [m2,i] = max(m1);
  map = [x(i) y(j(i))];
end


%%% --------------------------------------------------------------------
function y = SampleFaultSizeObserved(model,param)
% Sampling from the pdf of x, given that x is observed.
% Uses rejection sampling, drawing from the unconditioned pdf.

global PARAM DATA NDATA

numbers = 1:NDATA;
type = ones(1,NDATA);
type(DATA>=(PARAM.maxsize-PARAM.truncrange)) = 2;
ind1 = numbers(type==1);
ind2 = numbers(type==2);

if(length(ind1)>0)
  y(ind1) = SampleFaultSizeInterval(DATA(ind1)+PARAM.mintrunc,DATA(ind1)+PARAM.maxtrunc,model,param);
end
if(length(ind2)>0)
  y(ind2) = SampleFaultSize(length(ind2),PARAM.maxsize+PARAM.mintrunc,model,param);
end

alpha = P_ObsFaultSize(y)/PARAM.problarge;
u = unifrnd(0,1,1,NDATA);
ind = numbers(u>alpha);
l = length(ind);
while(l>0)
  ind1 = ind(DATA(ind)<(PARAM.maxsize-PARAM.truncrange));
  ind2 = ind(DATA(ind)>=(PARAM.maxsize-PARAM.truncrange));
  if(length(ind1)>0)
    y(ind1) = SampleFaultSizeInterval(DATA(ind1)+PARAM.mintrunc,DATA(ind1)+PARAM.maxtrunc,model,param);
  end
  if(length(ind2)>0)
    y(ind2) = SampleFaultSize(length(ind2),PARAM.maxsize+PARAM.mintrunc,model,param);
  end
  alpha(ind) = P_ObsFaultSize(y(ind))/PARAM.problarge;
  u = unifrnd(0,1,1,l);
  ind = ind(u>alpha(ind));
  l = length(ind);
end


%%% --------------------------------------------------------------------
function y = SampleFaultSizeUnobserved(n,model,param)
% Samples from the pdf of x, given that x is not observed.
% Uses rejection sampling, drawing from the unconditioned pdf.

global PARAM DATA

if(n<=0)
  y = [];
else
  p = P_Obs(model,param);
  if(PARAM.problarge>0.9999)
    uu = PARAM.upper;
  else
    uu = 5*max(DATA);
  end
  
  y = SampleFaultSize(n,PARAM.obslimit,model,param);
  alpha = (1-P_ObsFaultSize(y))/(1-PARAM.probsmall);
  u = unifrnd(0,1,1,n);
  numbers = 1:n;
  ind1 = numbers(y>=uu);
  ind2 = numbers(u>alpha);
  ind  = union(ind1,ind2);
  l = length(ind);
  while(l>0)
    y(ind) = SampleFaultSize(l,PARAM.obslimit,model,param);
    alpha(ind) = (1-P_ObsFaultSize(y(ind)))/(1-PARAM.probsmall);
    u = unifrnd(0,1,1,l);
    ind1 = numbers(y>=uu);
    ind2 = ind(u>alpha(ind));
    ind  = union(ind1,ind2);
    l = length(ind);
  end
end


%%% --------------------------------------------------------------------
function sample = SampleFaultSize(n,x0,model,param)

if(n>0)
    if(model==1)
	y = x0+exprnd(1,1,n);
	sample = x0.*exp((y-x0)./param);
    else
	sample = x0+exprnd(1/param,1,n);
    end
else
    sample = [];
end


%%% --------------------------------------------------------------------
function sample = SampleFaultSizeInterval(lower,upper,model,param)

global PARAM

n = max(length(lower),length(upper));
u = unifrnd(0,1,1,n);
if(model==1)
  sample = PARAM.obslimit./((1-u).*(PARAM.obslimit./lower).^param+...
			    u.*(PARAM.obslimit./upper).^param).^(1/param);
else
  sample = PARAM.obslimit-log((1-u).*exp(-param*(lower-PARAM.obslimit))+...
			      u.*exp(-param*(upper-PARAM.obslimit)))/param;
end


%%% --------------------------------------------------------------------
function status = WriteStatus(model,iteration,all,fault,old_status,state)

global PDF_M

status = old_status;

for i=1:2
  pp(i) = PDF_M(i)+FullLikelihood(fault(i))+fault(i).pdf_nfaults+...
	  fault(i).pdf_param+EmpiricalPdf(fault(Swap(i)));
end

status(rem(iteration-100,1000)/100+1,:) = ...
    [fault(1).nfaults fault(2).nfaults pp(1) pp(2) model];

if(all)
  disp(sprintf('Status after %d %s iterations:',iteration,state))
  disp(sprintf('%4d %4d %11.3f %9.3f %6d\n',status'))
end


%%% --------------------------------------------------------------------
function p = MLE_Param(data,x0,model)

n = length(data);
if(model==1)
  t = sum(log(data)-log(x0));
else
  t = sum(data-x0);
end
p = n./t;


%%% --------------------------------------------------------------------
function [l,u] = ConfidenceInterval(nof,alpha)

if(nof(2)==0)
  aaa = nof([2 1]);
else
  aaa = nof;
end

z = norminv(1-alpha/2,0,1);
n = sum(aaa);
x = aaa(1);
a = 2*x*(n-x)+n*z^2;
rot    = sqrt(4*n*x*(n-x)*z^2+n^2*z^4);
nevner = 2*(n-x)^2;
l = (a-rot)/nevner;
u = (a+rot)/nevner;

if(nof(2)==0)
  l = 1/u;
  u = Inf;
end


%%% --------------------------------------------------------------------
function a = Swap(b)

a = 3-b;
