#######################################################################
#######################################################################

# Estimating the ZMPL model parameters

#######################################################################
#######################################################################

# Packages

suppressPackageStartupMessages(library('compiler'))
suppressPackageStartupMessages(library('numDeriv'))

enableJIT(3) # For just-in-time compilation

#######################################################################
#######################################################################

# Defining the PL and the ZTPL distributions

pl <- function (x,a,log=F) {
  
  aux <- 2 * log(a) + log1p(a + x + 1) - (x + 3) * log1p(a)
  
  if (log) aux else exp(aux)
  
}

ztpl <- function (x,a,log=F) {
  
  aux <- 2 * log(a) + log1p(a + x + 1) - x * log1p(a) - log1p(a^2 + 3 * a)
  
  if (log) aux else exp(aux)
  
}

#######################################################################
#######################################################################

# Defining log-likelihood and log-posterior function

loglike <- function (a,data) {
  
  if (!is.na(a)) s <- sum(ztpl(data,exp(a),T))
  
  ifelse(exists('s'),s,-Inf)
  
}

logpost <- function (a,data) {
  
  H <- try(-as.numeric(hessian(loglike,a,data=data)),silent=T)
  
  if (!inherits(H,'try-error')) {
    
    if (is.finite(H)) if (H > 0) s <- loglike(a,data) + log(H)/2
    
  }
  
  ifelse(exists('s'),s,-Inf)
  
}

logpost <- cmpfun(logpost) # To gain speed on computations

# For simplicity, the exact Fisher Information was not provided here and
# therefore, a numerical approximation was used.

#######################################################################
#######################################################################

# Input data

x1 <- c(rep(0,182),rep(1,20),rep(2,11),rep(3,7),rep(4,4),rep(5,3),
        rep(6,6),rep(7,1),rep(8,1),rep(17,1))
#x2 <- c(rep(0,9),rep(1,63),rep(2,31),rep(3,12),rep(4,6),rep(5,1),
#        rep(6,2),rep(7,1),rep(10,1))
#x3 <- c(rep(0,213),rep(1,128),rep(2,37),rep(3,18),rep(4,3),rep(5,1))

## Choose a dataset to fit
y <- x1; yp <- y[which(y>0)]; n <- length(y); np <- length(yp)

#######################################################################
#######################################################################

# Parameters for the candidate-generating distribution

tilde.eta <- optim(0,logpost,control=list(fnscale=-1),method='BFGS',
                   data=yp)$par

nu <- 1 # Tuning parameter

#######################################################################
#######################################################################

# Computing posterior normalizing constant by Laplace's method

## Minimizing the negative of the log-posterior distribution
f0 <- optim(0,function (a,data) -logpost(a,data),data=yp,method='BFGS',
            hessian=T)

## Computing the normalizing constant
B <- as.numeric(sqrt(2 * pi) * (f0$hess)^(-.5) * exp(-f0$val))

#######################################################################
#######################################################################

# Computing the value of C

rr <- function (a,data) logpost(a,data) - log(B) - dnorm(a,tilde.eta,nu,T) 
cc <- optim(0,rr,method='BFGS',data=yp,control=list(fnscale=-1))$val

#######################################################################
#######################################################################

# Rejection sampling

## Auxiliary function to generate values
aux.fun <- function (x) {
  
  id <- F; flag <- 0
  
  while (id==F) {
    
    a <- rnorm(1,tilde.eta,sqrt(nu))
    b <- logpost(a,x) - log(B) - dnorm(a,tilde.eta,sqrt(nu),T) - log(cc)
    
    if (log(runif(1)) < b) { 
      
      z <- a; id <- T
    
    } else flag <- flag + 1
    
  }
  
  list(z=z,flag=flag)
  
}

## Rejection sampling algorithm
rej.samp <- function (x,M) {
  
  e <- flag <- numeric(M)
  
  for (i in 1:M) { R <- aux.fun(x); e[i] <- R$z; flag[i] <- R$flag }
  
  list(e=e,flag=flag)
  
}

#######################################################################
#######################################################################

# Model fit

set.seed(1) # Set a seed for reproducibility

M <- 5e3; aux.smp <- rej.samp(yp,M) # M is the desired sample size

eta.smp <- aux.smp$eta.sample
acc.rate <- mean(aux.smp$flag) * 100

theta.smp <- exp(eta.smp)
omega.smp <- rbeta(length(theta.smp),np + .5,(n - np) + .5)
p.smp     <- omega.smp/(1 - pl(0,theta.smp))

#######################################################################
#######################################################################

# Posterior summary

## Theta
mean.theta   <- mean(theta.smp)
median.theta <- median(theta.smp)
sd.theta     <- sd(theta.smp)

## Omega
mean.omega   <- (2 * np + 1)/(2 * (n + 1))
median.omega <- (6 * np + 1)/(2 * (3 * n + 1)) 
sd.omega     <- sqrt(((2 * np + 1) * (2 * (n - np) + 1))/(4 * (n + 1)^2 * (n + 2)))

## p
mean.p   <- mean(p.smp)
median.p <- median(p.smp)
sd.p     <- sd(p.smp)

#######################################################################
#######################################################################
