function Coef1=coef_basis1(V, M, grid, lambda, gamma)

% Calculate the coefficients, with respect to the first basis, of a set of m triangulated surfaces in R^3, from the barycenters and normal area vectors of their triangles.
% 
%  INPUTS:
% *V contains the normal area vectors, M contains the barycenters. Their dimensions are num_triangx3xm. V(:,:,ii) and M(:,:,ii) are the (num_triangx3)-matrices for surface ii.
% *grid is a (Nx3)-matrix, where N is the number of points in the grid. grid(ii,:) is one of the points.
% *lambda is a positive number, the parameter of the Gaussian kernel.
% *gamma is a regularization parameter that is used when calculating the "beta" vectors.
%
%  OUTPUTS:
% *Coef1 are the coefficients with respect to the first basis of the set of m surfaces. Its dimension is mx(3d), where d=rank(k). Each row corresponds with the estimation of the first 3d coefficients of a surface.


%TO OBTAIN Coef1, A FIRST STEP CONSISTS OF EXPRESSING THE DATABASE USING THE "beta" VECTORS:
N=size(grid,1); 
m=size(V,3); 

% Calculate the (NxN)-matrix, where the element (ii,jj) corresponds with the evaluation of the scalar gaussian kernel in grid points, i.e. k(grid(ii,:), grid(jj,:)).
k=zeros(N); 
for j=1:N
	k(:,j)=sum(bsxfun(@minus,grid,grid(j,:)).^2,2);
end
k=exp(-k/lambda^2); 

%Modify the previous matrix k, by using the regularization parameter gamma.
k_gamma=k;
k_gamma(1:N+1:N*N)=1+N*gamma; % Linear indexing: set diagonal values


%Calculate the "beta" vectors, solving a matrix system. In each iteration, b is an (Nx3)-matrix which corresponds with the evaluation of a surface in all the points of the grid (the loop is repeated for each surface). Eval_kernel and aux are auxiliary variable to calculate the evaluations.
betas=zeros(N,3,m);
for i=1:m
	aux=zeros(size(M(:,:,i),1),N); 
	for j=1:N
		aux(:,j)=sum(bsxfun(@minus,M(:,:,i),grid(j,:)).^2,2); 
	end
	Eval_kernel=exp(-aux/lambda^2); 
	b=Eval_kernel.'*V(:,:,i); 
	betas(:,:,i)=k_gamma\b; % beta(:, :, i) is an (Nx3)-matrix.
end




%ONCE THE VECTORS BETA ARE OBTAINED, ESTIMATE COEFFICIENTS OF THE m SURFACES IN THE FIRST BASIS OF FUNCTIONS: 
%Calculate eigenvectors ang eigenvalues of the (NxN)-matrix "k" and order the first "d" of them considering the descending order of the eigenvalues. 
[Eigenveck,Eigenvalk]=eig(k,'vector'); 
[~,idx] = sort(abs(Eigenvalk),'descend'); 
d=rank(k);
Eigenvalk=Eigenvalk(idx(1:d)); 
Eigenveck=Eigenveck(:,idx(1:d));

%By using the "beta" vectors, the eigenvalues and eigenvectors of matrix k, in each iteration, the "3d" coefficients of a surface are calculated and expressed as a row in matrix Coef1. 
Coef1=zeros(m, d*3);
for i=1:m
	Coef1_matrix=(betas(:,:,i).'*Eigenveck)*diag(sqrt(Eigenvalk)); 
	Coef1(i,:)=Coef1_matrix(:); 
end


