# parallel processing in R
library(parallel)

# SOCK works onany system (including windows).
nuncores <- detectCores()
cl <- makeCluster(nuncores) 
clusterEvalQ(cl, library(rstan))

data <- read.csv("Dataset.csv", header = TRUE, row.names=1)
N <- 10
Tp1 <- 109
P <- N*(N-1)/2
y <- as.matrix (data)
b <- as.vector(rep(1,N))

# ----- setting knots -----
K_f <- 5
K_g <- 5
TP <- (Tp1+1)/(Tp1+2) # to predicte next time
t_work <- c(); for(t in 1:Tp1){t_work[t] <- t/(Tp1+1)}

# f function
knots_f <- quantile(unique(t_work), seq(0,1,length=(K_f+2))[-c(1,(K_f+2))])
Z_K.f <- (abs(outer(t_work,knots_f,"-")))^3
OMEGA_all.f <- (abs(outer(knots_f,knots_f,"-")))^3
svd.OMEGA_all.f <- svd(OMEGA_all.f)
sqrt.OMEGA_all.f <- t(svd.OMEGA_all.f$v %*% (t(svd.OMEGA_all.f$u)*sqrt(svd.OMEGA_all.f$d)))
Z_f <- t(solve(sqrt.OMEGA_all.f,t(Z_K.f)))

# g function
knots_g <- quantile(unique(t_work), seq(0,1,length=(K_g+2))[-c(1,(K_g+2))])
Z_K.g <- (abs(outer(t_work,knots_g,"-")))^3
OMEGA_all.g <- (abs(outer(knots_g,knots_g,"-")))^3
svd.OMEGA_all.g <- svd(OMEGA_all.g)
sqrt.OMEGA_all.g <- t(svd.OMEGA_all.g$v %*% (t(svd.OMEGA_all.g$u)*sqrt(svd.OMEGA_all.g$d)))
Z_g <- t(solve(sqrt.OMEGA_all.g,t(Z_K.g)))

# ------------------------------------------- The model specification ------------------------------------------ #

library(rstan)

Model <- "data{
int<lower=1> N;
int<lower=1> P;
int<lower=1> Tp1;
int<lower=1> K_f;
int<lower=1> K_g;

real<lower=0, upper=1> t_work[Tp1];

vector[N] b;

matrix[Tp1,N] y;
matrix[Tp1,K_f] Z_f;
matrix[Tp1,K_g] Z_g;
}
transformed data{
matrix[N,N] I;
matrix[N,N] II; ;

I = diag_matrix(b);
II = b * b';
}
parameters {
real phi1_f;
real phi1_g;
real<lower=0> sigma;
real<lower=0> sigma_u_f;
real<lower=0> sigma_u_g;
real<lower=0> sigma_phi0_f;
real<lower=0> sigma_phi0_g;
real<lower=0> sigma_a;
real<lower=0> sigma_b;
real<lower=0, upper=1> gam;
real<lower=0, upper=gam> lambda;
real<lower=0, upper=(1-gam)> delta;
real<lower=0, upper=(1-lambda)> theta;

vector[P] z;
vector[N] phi0_f;
vector[P] phi0_g;
vector<lower=0>[N] c;

matrix[N,K_f] u_f;
matrix[P,K_g] u_g;
}
transformed parameters {
vector[N] f[Tp1];
vector[P] g[Tp1];

matrix[Tp1,N] uz_f;
matrix[Tp1,P] uz_g;
matrix[N,N] R[Tp1];
matrix[N,N] Q[Tp1];
matrix[N,N] u[Tp1];
matrix[N,N] h[Tp1];
matrix[N,N] Cov[Tp1];
matrix[N,N] tau[Tp1];
matrix[N,N] tau_work[Tp1];

matrix[N,N] A;
matrix[N,N] B;
matrix[N,N] S;          // scale matrix-wishart dist.
matrix[N,N] L;         // Cholesky factor of scale matrix S
matrix[N,N] V;        // lower triangular matrix-wishart dist.
matrix[N,N] W;       // random effects in variance-covariance matrix

{                  // W ~ Wishart(N,S)
int count = 1;
for (j in 1:(N-1)){
for (i in (j+1):N){
V[i, j] = z[count];
count += 1;
}
for (i in 1:(j - 1)){
V[i, j] = 0.0;}
V[j, j] = sqrt(c[j]);}
for (i in 1:(N-1)){
V[i, N] = 0;}
V[N, N] = sqrt(c[N]);
}
S = pow(sigma_a,2)*I + pow(sigma_b,2)*II;
L = cholesky_decompose(S);

W = (L*V*V'*L'); // W = mdivide_left_tri_low(V,S)

A = (gam-lambda)*I + lambda*II;
B = (delta-theta)*I + theta*II;

                                       // ---------- setting low_frequency matrix ---------- //
uz_f = Z_f * u_f';
uz_g = Z_g * u_g';
{
int countor = 1;
for(t in 1:Tp1){
f[t] = phi0_f + phi1_f*t_work[t] + uz_f[t]';  // diagonal elements 
g[t] = phi0_g + phi1_g*t_work[t] + uz_g[t]'; //off diagonal elements

Q[t] = diag_matrix(f[t]);
  
for(i in 1:N){R[t,i,i] = 1;}
for(i in 2:N){
  for(j in 1:(i-1)){ 
      R[t,i,j] = g[t,(i*(i-1)/2)-(i-j-1)];
      R[t,j,i] = 0; } }
      
u[t] = (y[t])'* y[t];
tau[t] = R[t] * Q[t] * Q[t] * R[t]';

for(i in 1:N){
 for(j in 1:N){
 tau_work[t,i,j] = sqrt(tau[t,i,i] * tau[t,j,j]); } }
}
}
h[1] = pow(sigma,2)*W;
Cov[1] = h[1] .* tau[1];
for(t in 2:Tp1){
h[t] = W + A .* u[t-1] ./ tau_work[t-1] + B .* h[t-1];
Cov[t] = h[t] .* tau[t];}
}

model{
z ~ normal(0, 1);
sigma_a ~ cauchy(0,2.5);
sigma_b ~ cauchy(0,2.5);
for(i in 1:N){c[i] ~ chi_square(N-i+1);}

sigma ~ cauchy(0,10);
gam ~ beta(1,1);
lambda ~ beta(1,1);
delta ~ beta(1,1);
theta ~ beta(1,1)T[0,delta];

sigma_phi0_f ~ cauchy(0,5); 
sigma_phi0_g ~ cauchy(0,2.5);
phi0_f ~ normal(0,sigma_phi0_f);
phi0_g ~ normal(0,sigma_phi0_g);
phi1_f ~ normal(0,1);
phi1_g ~ normal(0,1);

sigma_u_f ~ cauchy(0,5);
sigma_u_g ~ cauchy(0,5);

for(i in 1:N){for(k in 1:K_f){u_f[i,k]~normal(0,sigma_u_f);}}
for(l in 1:P){for(k in 1:K_g){u_g[l,k]~normal(0,sigma_u_g);}}

for(t in 1:Tp1){ y[t] ~ multi_normal(rep_vector(0,N),Cov[t]);}
}
"
datalist <- list(y=y, b=b, Tp1=Tp1, N=N, P=P, TP=TP, K_f = K_f, K_g = K_g, Z_f = Z_f, Z_g = Z_g, t_work = t_work)

parms <- c("gam", "lambda", "delta", "theta", "sigma_phi0_f", "sigma_phi0_g",
           "sigma_u_f", "sigma_u_g", "sigma_a", "sigma_b", "sigma", "phi1_f", "phi1_g")

#Setting up 
n_chains <- 1; sample <- 14000; burnin <- 3500; thin <- 1

fit <- stan(model_code = Model, data = datalist, pars = parms, chains = n_chains,
            save_dso = TRUE, iter=sample, warmup=burnin, thin=thin, init="random")

print(fit, digits_summary = 4)

