#	Comparing nonparametric surfaces
#       The SEPA data

source("sm-library.r")
source("surfplot.r")
source("surfdiffplot.r")
source("sigma2.r")
source("rmplot.r")
sm.options(verbose=1)

d <- read.table("sepa.dat")
x     <- d[,1:2]
y     <- d[,3]
group <- d[,4]
group <- (group == 1)
id    <- d[,5]

#	Remove odd case

remove.case <- FALSE
if (remove.case) {
   ind   <- !(id == 840207)
   x     <- x[ind,]
   y     <- y[ind]
   group <- group[ind]
   id    <- id[ind]
   }

sm.options(list(xlab = "Day of year"))
sm.options(list(ylab = "Station"))
sm.options(list(zlab = "DO"))
sm.options(zlim = c(1, 13))
sm.options(list(theta = 0, phi = 30))
fname <- "figures/sepa-"

par(mfrow = c(1,2))
plot(x[,1], y, xlab = "Day of year", ylab = "DO", type = "n")
points(x[ group,1], y[ group], pch = "o", col = 2)
points(x[!group,1], y[!group], pch = "x", col = 1)
rmplot(y, id = factor(id), timept = x[,2], fac = as.factor(group),
         xlabel = "Station", ylabel = "DO")
par(mfrow = c(1,1))


#------------------------------------------------------------------------
#		Construct covariance matrix
#------------------------------------------------------------------------

h   <- h.select(x, y, method = "aicc", group = group)
m1  <- sm.regression(x[group,],  y[group],  h = h, eval.points = x[group,],
                      eval.grid = FALSE)
m2  <- sm.regression(x[!group,], y[!group], h = h, eval.points = x[!group,],
                      eval.grid = FALSE)
res <- y
res[ group] <- y[ group] - m1$estimate
res[!group] <- y[!group] - m2$estimate

plot(x[,2], res, type = "n", xlab = "Station", ylab = "Residual" )
means  <- numeric(0)
vars   <- numeric(0)
num    <- 0
den    <- 0
for (i in unique(id)) {
    indi   <- (id == i)
    mn     <- mean(res[indi])
    means  <- c(means, mn)
    vars   <- c(vars, var(res[indi]))
    resd   <- (res[indi] - mn)
    nn     <- length(res[indi])
    num    <- num + sum(resd[1:(nn-1)]*resd[2:nn])
    den    <- den + sum(resd[1:(nn-1)]^2 + resd[2:nn]^2)
    g      <- group[indi][1]
    if (g) col <- 1 else col <- 2
    lines(x[,2][indi], res[indi], col = col, lty = col)
    }
sigmasq.random <- var(means)
sigmasq.ar <- mean(vars)
rho.tilde <- 0

cat("   sigmasq.random =", round(sigmasq.random, 5), "\n")
cat("   sigmasq.ar     =", round(sigmasq.ar,     5), "\n")

covmat <- diag(rep(0,nrow(x)))
for (i in unique(id)) {
    indi     <- (1:nrow(x))[id == i]
    nindi    <- length(indi)
    Stationi <- x[,2][indi]
    covmati  <- matrix(Stationi, nrow=nindi, ncol=nindi)
    covmati  <- covmati - t(covmati)
    covmati  <- rho.tilde^abs(covmati)
    covmati  <- covmati * sigmasq.ar + sigmasq.random
    covmat[indi,indi] <- covmat[indi,indi] + covmati
    }

#------------------------------------------------------------------------
#		Compare surface models
#------------------------------------------------------------------------

sm.ancova(x, y, group, h, model = "equal",    covar = covmat)
sm.ancova(x, y, group, h, model = "parallel", covar = covmat)


#------------------------------------------------------------------------
#		Plot surfaces
#------------------------------------------------------------------------

par(mfrow=c(1,2))

sm.surfplot(x[!group,], y[!group], h, display = "se",
                covar = covmat[!group, !group], xlim = range(x[,1]))
sm.surfplot(x[ group,], y[ group], h, display = "se",
                covar = covmat[group, group], xlim = range(x[,1]))

sm.surfdiffplot(x, y, group, h = h, covmat)

sm.surfdiffplot(x, y, group, h = h, covmat, shift = TRUE)

par(mfrow=c(1,1))


#------------------------------------------------------------------------
#               Examine the sensitivty to the choice of h.
#------------------------------------------------------------------------

for (df in seq(14, 24, by = 2)) {

h.sens <- h.select(x, y, group = group, df = df)

m1   <- sm.regression(x[group,],  y[group],  h = h.sens, eval.points = x[ group,],
                      eval.grid = FALSE)
m2   <- sm.regression(x[!group,], y[!group], h = h.sens, eval.points = x[!group,],
                      eval.grid = FALSE)
res  <- y
res[group]  <- y[group]  - m1$estimate
res[!group] <- y[!group] - m2$estimate

means  <- numeric(0)
vars   <- numeric(0)
num    <- 0
den    <- 0
for (i in unique(id)) {
    indi   <- (id == i)
    mn     <- mean(res[indi])
    means  <- c(means, mn)
    vars   <- c(vars, var(res[indi]))
    resd   <- (res[indi] - mn)
    nn     <- length(res[indi])
    num    <- num + sum(resd[1:(nn-1)]*resd[2:nn])
    den    <- den + sum(resd[1:(nn-1)]^2 + resd[2:nn]^2)
    }
sigmasq.random <- var(means)
rho.tilde <- (2*num) / den
sigmasq.ar <- mean(vars)
rho.tilde <- 0

covmat <- diag(rep(0,nrow(x)))
for (i in unique(id)) {
    indi     <- (1:nrow(x))[id == i]
    nindi    <- length(indi)
    Stationi <- x[,2][indi]
    covmati  <- matrix(Stationi, nrow=nindi, ncol=nindi)
    covmati  <- covmati - t(covmati)
    covmati  <- rho.tilde^abs(covmati)
    covmati  <- covmati * sigmasq.ar + sigmasq.random
    covmat[indi,indi] <- covmat[indi,indi] + covmati
    }

p.equal    <- sm.ancova(x, y, group, h.sens, model = "equal",    covar = covmat)
p.parallel <- sm.ancova(x, y, group, h.sens, model = "parallel", covar = covmat)
}
