
#------------------------------------------------------------------------------------------------------------------------
# Functions for "Tobit Model Estimation and Sliced Inverse Regression" 
#    by Lexin Li, Jeff Simonoff, and Chih-Ling Tsai
#------------------------------------------------------------------------------------------------------------------------



# Function for Buckley-James fitting of tobit regression model
#
#    Input:
#       X        : matrix of predictors
#       y        : observed response variable
#       delta    : vector of censoring indicators (0 = censored, 1 = uncensored)
#       intercept: should an intercept term be returned? Note that Buckley-James models MUST 
#                  include an intercept in the fitting; this flag merely controls if that 
#                  intercept is reported. (Yes = TRUE)
#       normalize: should the slope coefficients be normalized to have sum of squares equal to 1? (Yes= TRUE)
#     
#    Notes: 
#       Usual Buckley-James algorithm is "fooled" into fitting left-censored data
#       by setting y to -y, fitting, and then setting coefficients to their negatives 
#
#       Design library must be loaded first, and Design requires several other packages
#       (Hmisc, grid, lattice, acepack, survival, splines)
tobit.bj <- function(X, y, delta, intercept=T, normalize=F)
{
   # parameters
   if (!is.matrix(X)) X <- as.matrix(X)
   n<-nrow(X)
   p<-ncol(X)
   if(is.null(colnames(X))) {
      xnames<-NULL; for(i in 1:p) xnames<-c(xnames, paste("x", i, sep=""))
      colnames(X)<-xnames
   } 
   xnames<-colnames(X)

   # formula
   formula<-paste("Surv(y, delta) ~", xnames[1])
   if (p > 1) for(i in 2:p) {
      formula<-paste(formula, "+", xnames[i])
   }
   formula<-as.formula(formula)

   # data
   data<-data.frame(cbind(-y, delta, X))
   colnames(data)<-c("y", "delta", xnames)

   # model fitting
   out<-bj(formula, data=data, link='identity', control=list(iter.max=100))
   if(out$fail) {
      out<-bj(formula, data=data, link='identity', control=list(iter.max=1000))
      if(out$fail) {
         b.est<-rep(NA, p+1)
         b.se<-rep(NA, p+1)
         return(list(b.est=b.est, b.se=b.se))
      }
   }
   b.est <- -coef(out)
   b.se <- sqrt(diag(out$var))
   if (normalize) {
     temp <- b.est
     b.est <- c(b.est[1],norm(b.est[-1]))
     b.se <- b.se*b.est/temp
   }
   if (!intercept) {
     b.est <- b.est[-1]
     b.se <- b.se[-1]
   }
   b.est<-matrix(b.est, ncol=1)
   b.se<-matrix(b.se, ncol=1)
   if (!intercept) {
     rownames(b.est) <- xnames
     rownames(b.se) <- xnames
   } else {
     rownames(b.est) <- c("Intercept",xnames)
     rownames(b.se) <- c("Intercept",xnames)
   }

   # return
   ans<-list(b.est=b.est, b.se=b.se)
   return(ans)
}



# Function for MLE fitting of tobit regression model
#
#    Input: 
#       X        : matrix of predictors
#       y        : observed response variable
#       delta    : vector of censoring indicators (0 = censored, 1 = uncensored)
#       intercept: should an intercept term be included in the model? (Yes = TRUE) 
#       normalize: should the coefficients be normalized to have sum of squares equal to 1? (Yes = TRUE)
#
#    Notes: 
#       survival library must be loaded first
tobit.mle<-function(X, y, delta, intercept=T, normalize=F)
{
   # parameters
   if (!is.matrix(X)) X <- as.matrix(X)
   n<-nrow(X)
   p<-ncol(X)
   if(is.null(colnames(X))) {
      xnames<-NULL; for(i in 1:p) xnames<-c(xnames, paste("x", i, sep=""))
      colnames(X)<-xnames
   } 
   xnames<-colnames(X)

   # formula
   formula<-paste("Surv(y, delta,type='left') ~", xnames[1])
   if (p > 1) for(i in 2:p) {
      formula<-paste(formula, "+", xnames[i])
   }
   if (!intercept) formula<-paste(formula, "- 1")
   formula<-as.formula(formula)

   # data
   data<-data.frame(cbind(y, delta, X))
   colnames(data)<-c("y", "delta", xnames)

   # model fitting
   out<-survreg(formula, data=data, dist='gaussian')
   b.est <- coef(out)
   b.se <- sqrt(diag(out$var))[-nrow(out$var)]
   if (normalize) {
     temp <- b.est
     if (intercept) b.est<-c(b.est[1],norm(b.est[-1])) else b.est <- norm(b.est)
     b.se <- b.se*b.est/temp
   }
   b.est<-matrix(b.est, ncol=1)
   b.se<-matrix(b.se, ncol=1)
   if (!intercept) {
     rownames(b.est) <- xnames
     rownames(b.se) <- xnames
   } else {
     rownames(b.est) <- c("Intercept",xnames)
     rownames(b.se) <- c("Intercept",xnames)
   }
 
   # return
   ans<-list(b.est=b.est, b.se=b.se)
   return(ans)
}



# Function for Powell's LAV fitting of tobit regression model
#
#    Input: 
#       X         : matrix of predictors
#       y         : observed response variable
#       censorval : vector of censoring variable values
#       intercept : should an intercept term be included in the model? (Yes = TRUE) 
#       normalize : should the coefficients be normalized to have sum of squares equal to 1? (Yes = TRUE)
#       init.given: initial start for algorithm given? (Yes = TRUE)
#       init      : initial start for parameter estimates
# 
#    Notes:
#       This function requires all of the censoring values, not just those where
#       censoring occurred along with a censoring indicator
#
#       This function requires the nlrq package; recent versions of the quantreg
#       package include nlrq. The least absolute value minimizer requires a start value; 
#       if it is not given (init.given=F, the default), the MLE is used, which means 
#       that the survival package must also be loaded.
#
#       No standard error estimates for the parameter estimates are given; these
#       depend on smoothing the residuals.
tobit.lav <- function(X, y, censorval, intercept=T, normalize=F, init.given=F, init=NULL)
{
   # parameters
   if (!is.matrix(X)) X <- as.matrix(X)
   n<-nrow(X)
   p<-ncol(X)
   if(is.null(colnames(X))) {
      xnames<-NULL; for(i in 1:p) xnames<-c(xnames, paste("x", i, sep=""))
      colnames(X)<-xnames
   } 
   xnames<-colnames(X)

   # formula
   formula<-paste("y ~ pmax(censorval,")
   if (intercept) formula <- paste(formula, "b0 + ")
   for (i in 1:p){
      formula<-paste(formula, "b",i," * ", xnames[i], sep="")
      if (i < p) formula <- paste(formula, " + ")
   }
   formula<-paste(formula, ")")
   formula<-as.formula(formula)

   # data
   data<-data.frame(cbind(y, censorval, X))
   colnames(data)<-c("y", "censorval", xnames)

   # model fitting
   if (init.given) {
      start <- init
   } else { 
     delta <- as.numeric(y > censorval)
     start <- as.vector(tobit.mle(X, y, delta, intercept=intercept)$b.est)
   }
   bnames <- NULL
   if (intercept) bnames <- c(bnames,paste("b0"))
   for (i in 1:p) bnames <- c(bnames, paste("b",i,sep=""))
   names(start) <- bnames
   out<-nlrq(formula, data=data, start=start)
   b.est <- coef(out)
   if (normalize) {
     temp <- b.est
     if (intercept) b.est<-c(b.est[1],norm(b.est[-1])) else b.est <- norm(b.est)
   }
   b.est<-matrix(b.est, ncol=1)
   if (!intercept) {rownames(b.est) <- xnames} else {rownames(b.est) <- c("Intercept",xnames)}

   # return
   ans<-list(b.est=b.est)
   return(ans)
}



# Function for Chernozhukov and Hong 3-step censored quantile regression fitting of tobit regression model
#
#    Input: 
#       X        : matrix of predictors
#       y        : observed response variable
#       censorval: vector of censoring variable values
#       intercept: should an intercept term be included in the model? (Yes = TRUE) 
#       normalize: should the coefficients be normalized to have sum of squares equal to 1? (Yes = TRUE)
#       iter     : number of final step iterations of algorithm
#
#     Notes: 
#       For 50th percentile; see JASA (2002), vol. 97, pp. 872-882.
#
#       This function requires all of the censoring values, not just those where
#       censoring occurred along with a censoring indicator
#
#       This function requires the quantreg package
#
#       No standard error estimates for the parameter estimates are given; these
#       depend on smoothing the residuals.
tobit.3stepcqr<-function(X, y, censorval, intercept=T, normalize=F, iter=2)
{
   # parameters
   if (!is.matrix(X)) X <- as.matrix(X)
   n<-nrow(X)
   p<-ncol(X)
   if(is.null(colnames(X))) {
      xnames<-NULL; for(i in 1:p) xnames<-c(xnames, paste("x", i, sep=""))
      colnames(X)<-xnames
   } 
   xnames<-colnames(X)

   # formula for logistic regression of noncensoredness
   formula<-paste("delta ~ censorval + ")
   for(i in 1:p) {
      formula<-paste(formula, xnames[i])
      if (i < p) formula <- paste(formula, "+")
   }
   formula<-as.formula(formula)
   delta <- as.numeric(y > censorval)

   # data
   data<-data.frame(cbind(y, censorval, delta, X))
   colnames(data)<-c("y", "censorval", "delta", xnames)

   # model fitting
   outprob <- glm(formula, data=data, family=binomial)
   probest <- fitted(outprob)
   num0 <- sum(as.numeric(probest > .5))
   if(num0 < 2) {
     print("Not enough uncensored values to fit model using this method")
     return(list(b.est=rep(NA,p+1)))
   }
   ppquant <- order(probest)[ceiling(n-.9*num0+1)]
   J0 <- data[probest >= probest[ppquant],]
   if(nrow(J0) < p+1) {
     print("Not enough uncensored values to fit model using this method")
     return(list(b.est=rep(NA,p+1)))
   }
   formula<-paste("y ~", xnames[1])
   if (p > 1) for(i in 2:p) {
      formula<-paste(formula, "+", xnames[i])
   }
   if (!intercept) formula<-paste(formula, "- 1")
   formula<-as.formula(formula)
   outJ0 <- rq(formula, data=J0, ci=F)
   # Predict not working for rq object; create an lm object with the same coefficients and predict from it
   fakemodel <- lm(formula, data=J0)
   fakemodel$coefficients <- coef(outJ0)
   fitJ0 <- predict(fakemodel, data)
   # Paper says that observations with fitted values above the censoring value plus a "small number"
   # form the next set for fitting, but they say nothing about how to get that number, so we will
   # ignore it     
   if (iter > 0) {
     for (ii in 1:iter){
       J1 <- data[fitJ0 >= censorval,]
       if(nrow(J1) < p+1) {
         print("Not enough uncensored values to fit model using this method with iterations")
         return(list(b.est=rep(NA,p+1)))
       }
       outJ1 <- rq(formula, data=J1, ci=F)
       # Predict not working for rq object; create an lm object with the same coefficients and predict from it
       fakemodel <- lm(formula, data=J1)
       fakemodel$coefficients <- coef(outJ1)
       fitJ1 <- predict(fakemodel, data)
     }
   }
   b.est <- coef(outJ1)
   if (normalize) {
     temp <- b.est
     if (intercept) b.est<-c(b.est[1],norm(b.est[-1])) else b.est <- norm(b.est)
   }
   b.est<-matrix(b.est, ncol=1)
   if (!intercept) {rownames(b.est) <- xnames} else {rownames(b.est) <- c("Intercept",xnames)}

   # return
   ans<-list(b.est=b.est)
   return(ans)
}



# Function for Weighted Average Derivative Estimator 
#
#    Input: 
#       X        : matrix of predictors
#       y        : observed response variable
#       h        : bandwidth vector
#       normalize: should the coefficients be normalized to have sum of squares equal to 1? (Yes = TRUE)
#
#    Notes:
#       No standard error estimates for the parameter estimates are given 
tobit.wade<-function(X, y, h, normalize=F)
{
   if (!is.matrix(X)) X <- as.matrix(X)
   # parameters
   n<-nrow(X)
   p<-ncol(X)
   if(is.null(colnames(X))) {
      xnames<-NULL; for(i in 1:p) xnames<-c(xnames, paste("x", i, sep=""))
      colnames(X)<-xnames
   }
   xnames <- colnames(X)
   delta <- rep(0,p)
   for (i in (1:(n-1))){
     for (j in ((i+1):n)){
       delta <- delta + kprime((X[i,] - X[j,])/h)*(y[i] - y[j])}}
   b.est <- -delta/(choose(n,2)*(h^(p+1)))
   if (normalize) {b.est <- norm(b.est)}
   b.est<-matrix(b.est, ncol=1)
   list(b.est=b.est)
}

kprime <- function(u) { -u*dnorm(u) }

# function for Powell and Stoker optimal WADE bandwidth
#
#    Input: 
#       X        : matrix of predictors
#       y        : observed response variable
#       c        : constant for pilot bandwidth, which equals c*n^{-1/(2p+5)}
#       tau      : constant for estimate of bias coefficient
hopt <- function(X, y, c=5, tau=2)
{
   if (!is.matrix(X)) X <- as.matrix(X)
   # parameters
   n<-nrow(X)
   p<-ncol(X)
   gamma <- p+2
   h0 <- c*n^(-1/(2*p+5))
   Q <- rep(0,p)
   for (i in (1:(n-1))){
     for (j in ((i+1):n)){
       Q <- Q + (kprime((X[i,] - X[j,])/h0)*(y[i] - y[j]))^2}}
   Q <- Q*(h0^gamma)/choose(n,2)
   S <- (tobit.wade(X,y,tau*h0)$b.est - tobit.wade(X,y,h0)$b.est)/((tau*h0)^2 - h0^2)
   (gamma*Q/(2*S*S))^(1/(4+gamma))*n^(-2/(4+gamma))
}
 


# Function for SIR fitting of tobit regression model
#    Input: 
#       X:         matrix of predictors
#       y:         response variable
#       delta:     vector of censoring indicators (0 = censored, 1 = uncensored)
#       nslices:   number of sliced used in SIR
#       d:         the structural dimension of the central subspace
#       test:      should the asymptotic test based on normal theory be performed? (Yes = TRUE)
#      
#    Notes: 
#       dr library must be loaded first
tobit.sir<-function(X, y, delta, nslices=5, d=1, test=F)
{
   # parameters
   if (!is.matrix(X)) X <- as.matrix(X)
   n<-nrow(X)
   p<-ncol(X)
   if(is.null(colnames(X))) {
      xnames<-NULL; for(i in 1:p) xnames<-c(xnames, paste("x", i, sep=""))
      colnames(X)<-xnames
   }
   xnames <- colnames(X)

   # standardization
   Sigma.x.inv2<-mat.sqrt.inv(cov.x(X))
   Z<-apply(X, 2, center) %*% Sigma.x.inv2

   # slice y
   sy<-dr.slices.double(cbind(y, delta), nslices)
   hs<-sy$nslices

   # kernel matrix
   M<-matrix(0, p, p)
   for(s in 1:hs) {
      Z.s<-Z[sy$slice.indicator == s, ]
      if(sum(sy$slice.indicator == s) > 1) {
         Z.sm<-as.vector(apply(Z.s, 2, mean))
      } else {
         Z.sm<-Z.s
      }
      M<-M + (sy$slice.sizes[s]/n) * Z.sm %*% t(Z.sm)
   }

   # estimate
   ei<-eigen(M)
   v<-Sigma.x.inv2 %*% ei$vectors[, 1:d]
   b.est<-apply(v, 2, norm)
   rownames(b.est)<-xnames

   # se
   ev<-ei$values
   se.c<-(1 - ev) / ev / n
   se.x<-diag(solve(cov.x(X)))
   b.se<-NULL; for(j in 1:d) b.se<-cbind(b.se, sqrt(se.c[j] * se.x))

   # asymptotic test
   if(test) {
      e<-sort(ei$values)
      dmax<-min(4, p - 1, hs - 2)
      st<-df<-pv<-0
      for(m in 0:dmax) {
         st[m+1]<-n*sum(e[seq(1,p-m)])
         df[m+1]<-(hs - m - 1)*(p - m)
         pv[m+1]<-1 - pchisq(st[m+1],df[m+1])
      }
      asy.test<-data.frame(cbind(st,df,pv))
      rr<-paste(0:dmax,"D vs >= ",1:(dmax+1),"D",sep="")
      dimnames(asy.test)<-list(rr,c("Stat","df","p-value")) 
   }

   # return 
   ans<-b.est
   if(test) ans<-list(b.est=b.est, b.se=b.se, asy.test=asy.test) else
     ans<-list(b.est=b.est, b.se=b.se)
   return(ans)
}

# supporting functions for vector and matrix computation
norm <- function(v)   { v/sqrt(sum(v^2)) }

center <- function(v) { v - mean(v) }

cov.x <- function(X)
{
   Xc<-apply(X, 2, center)
   t(Xc) %*% Xc / nrow(Xc)
}

mat.sqrt <- function(A)
{
   if(nrow(A) != ncol(A)) stop("the matrix must be symmetric")
   ei<-eigen(A)
   d<-ei$values
   d<-(d+abs(d))/2
   d2<-sqrt(d)
   ans<-ei$vectors %*% diag(d2) %*% t(ei$vectors)
   return(ans)
}

mat.sqrt.inv <- function(A)
{
   if(nrow(A) != ncol(A)) stop("the matrix must be symmetric")
   ei<-eigen(A)
   d<-ei$values
   d<-(d+abs(d))/2
   d2<-1 / sqrt(d)
   d2[d == 0]<-0
   ans<-ei$vectors %*% diag(d2) %*% t(ei$vectors)
   return(ans)
}

# function to do double slicing for censored data
dr.slices.double<-function(y, nslices, mincl=2, del.cluster=FALSE)
{
  u<-sort(unique(y[,2]))
  h<-nslices
  ind<-sizes<-0
  count.slices<-0
  for (j in 1:length(u)) {
      pos<-which(y[,2]==u[j])
      s2<-dr.slice.1d(y[pos, 1], h)
      ind[pos]<-count.slices + s2$slice.indicator 
      sizes[(count.slices + 1) : (count.slices + s2$nslices)]<-s2$slice.sizes 
      count.slices<-count.slices + s2$nslices
  } 
  list(slice.indicator=ind, nslices=count.slices, slice.sizes=sizes)
}

# function for robust correlations based on Wilcox (1997); see
# http://www.unt.edu/benchmarks/archives/2001/december01/rss.htm
bicov<-function(x, y)
{
    mx <- median(x)
    my <- median(y)
    ux <- abs((x - mx)/(9 * qnorm(0.75) * mad(x)))
    uy <- abs((y - my)/(9 * qnorm(0.75) * mad(y)))
    aval <- ifelse(ux <= 1, 1, 0)
    bval <- ifelse(uy <= 1, 1, 0)
    top <- sum(aval * (x - mx) * (1 - ux^2)^2 * 
               bval * (y - my) * (1 - uy^2)^2)
    top <- length(x) * top
    botx <- sum(aval * (1 - ux^2) * (1 - 5 * ux^2))
    boty <- sum(bval * (1 - uy^2) * (1 - 5 * uy^2))
    bi <- top/(botx * boty)
    bi
}

# calculate biweight midcorrelation.  
bicor<-function(x,y) 
{
    x<-as.numeric(x)
    y<-as.numeric(y)
    bicov(x,y)/(sqrt(bicov(x,x)*bicov(y,y)))
}



